/**
 * "Стакан" крипты, содержит в себе прогнозатор аудитории и сводку параметров влияющих на прогноз
 *
 * @param {PredictParams} data - данные для блока
 */
BEM.DOM.decl('b-crypta-inventory', {

    /**
     * Флаг о готовности
     */
    _isReady: null,

    onSetMod: {

        js: function() {

            this.setMod('loading', 'yes');

            this._data = this.params.data;

            this._updateMediareachDebounce = $.debounce(this._updateMediareach, 200);
            this._predictor = this.blockInside('b-crypta-predictor');
            this._advisor = this.blockInside('crypta-advisor');
            this._summary = this.blockInside('crypta-summary-provider');

            this._initClipboard();

            this._hideHint = this._hideHint.bind(this);
            this.bindToWin('scroll', this._hideHint);
            this.bindTo('summary-wrap', 'scroll', this._hideHint);
        },

        'predictor-loading': {

            yes: function() {
                this._predictor.setMod('loading', 'yes');
            },

            '': function() {
                this._predictor.delMod('loading');
            }

        }

    },

    /**
     * Обновляет содержимое блока
     * @param {PredictParams} data - данные для блока
     */
    update: function(data) {
        if (u._.isEqual(this._data, data)) {
            this.delMod('predictor-loading');

            return;
        }
        this._data = data;

        this._updateSummary();
        this._updateMediareachDebounce();
    },

    /**
     * Возвращает значение флага
     * @returns {boolean}
     */
    isReady: function() {
        return !!this._isReady;
    },

    /**
     * Обработчик готовности сводки
     */
    _onReadyContent: function() {
        this.delMod('loading');
        this._isReady = true;
        this.trigger('ready');
    },

    /**
     * Обновляет прогноз и рекомендации
     * @private
     */
    _updateMediareach: function() {

        if (u._.isEmpty(this._data)) {
            return;
        }

        var requestData = this._prepareMediareachData(),
            isRecommendationEnabled = this._data.cpmGroupType === 'cpm_audio';

        this._predictor.update(requestData);

        this.toggleMod(this.elem('advisor'), 'show', 'yes', (this._data.banners || []).length > 0 || isRecommendationEnabled);
        this._advisor.delMod('open');
        this._advisor.updateData(requestData);
    },

    /**
     * Подготавливает данные для запроса прогноза и рекомендаций
     * @returns {ReachParams}
     * @private
     */
    _prepareMediareachData: function() {
        var geoIds = u._.get(this._data, 'geo.ids', null),
            result = {
                adgroup_id: this._data.adgroup_id,
                campaign_id: this._data.campaign_id,
                has_adaptive_creative: this._data.has_adaptive_creative,
                geo: geoIds ? geoIds.split(',') : geoIds,
                conditions: this._data.interests,
                platform_corrections: this._data.corrections,
                excluded_domains: null,
                group_type: this._data.cpmGroupType.replace(/cpm_/, '').toUpperCase() // DIRECT-98713
            };

        switch (this._data.cpmGroupType) {
            case 'cpm_video':
                result.video_creatives = this._data.banners.map(function(banner) {
                    return u._.pick(banner, 'duration');
                });
                break;
            case 'cpm_banner':
                result.block_sizes = this._data.banners;
                break;
            default:
                result.block_sizes = this._data.banners;

        }

        return result;
    },

    /**
     * Обновляет сводку
     * @private
     */
    _updateSummary: function() {
        this._summary.update(this._data);
    },

    /**
     * Показывает сообщение при удачном копировании в буфер обмена
     * @private
     */
    _showCopiedMessage: function(e) {
        e.clearSelection(); // чтобы в FF не скролилось

        clearTimeout(this._copiedTimeout);

        this._getTipman().show({
            owner: this.elem('copy-button'),
            content: iget2('b-crypta-inventory', 'copied-text', 'Скопировано')
        });

        this._copiedTimeout = setTimeout(function() {
            this._getTipman().hide();
        }.bind(this), 1000);

    },

    /**
     * Возвращает инстанс блока tipman
     *
     * @returns {BEM.DOM}
     * @private
     */
    _getTipman: function() {
        return this._tipman || (this._tipman = BEM.create('tipman', {
            tipMods: { size: 's', theme: 'normal', autoclosable: 'no' },
            tipJs: { to: ['right'] }
        }));
    },

    /**
     * Прячет подсказку
     * @private
     */
    _hideHint: function() {
        this.blockInside(this.elem('summary-title'), 'b-hintable-popup').hideHint();
    },

    /**
     * Инициализирует работу библиотеки для копирования текста по нажатию на кнопку
     * @private
     */
    _initClipboard: function() {
        this.clipboard = new Clipboard('.b-crypta-inventory__copy-button', {
            text: function(trigger) {
                return this.findElem('summary')[0].innerText;
            }.bind(this)
        }).on('success', this._showCopiedMessage.bind(this));
    },

    destruct: function() {
        this.unbindFromWin('scroll', this._hideHint);
        this.unbindFrom('summary-wrap', 'scroll', this._hideHint);
        this.clipboard.destroy();
        this.__base.apply(this, arguments);
    }

}, {

    live: function() {

        this.liveInitOnBlockInsideEvent('ready', 'crypta-summary-provider', function(e, data) {
            this._onReadyContent();
        });

        return false;
    }

});
