BEM.DOM.decl('b-crypta-predictor', {

    onSetMod: {

        js: function() {
            this._progress = this.findBlockInside('b-progress');

            this.setMod('loading' , 'yes');

            Promise.all([
                BEM.blocks['i-crypta-segments-data'].getSegmentsHash(),
                BEM.blocks['i-retargeting-goals-data'].getGoalsHash()
            ]).then(function(responses) {
                this._segmentsHash = responses[0];
                this._goalsHash = responses[1];

                this.delMod('loading');
                this._progress.delMod('loading');
            }.bind(this)).catch(function(e) {
                this.showError();
            }.bind(this));

            this._hideHints = this._hideHints.bind(this);
            this.bindToWin('scroll', this._hideHints);
        },

        loading: {

            yes: function() {
                this._progress.setMod('loading', 'yes');
            },

            '': function() {
                this._progress.delMod('loading');
            }

        }

    },

    /**
     * @typedef {Object} ReachParams - параметры для запроса прогноза
     *
     * @property {String} adgroup_id - ид группы
     * @property {String} campaign_id - ид кампании
     * @property {Array<String>} geo - ид регионов
     * @property {Array<Object>} block_sizes - размеры баннеров
     * @property {Array<RetConditionGroup>} conditions - формула сегментов
     * @property {{[desktop]: Number, [mobile]: Number, [mobile_os_type]: string }} platform_corrections - формула корректировок
     */

    /**
     * Обновляет прогноз
     * @param {ReachParams} data - параметры запроса
     */
    update: function(data) {
        this.hideError();
        this.setMod('loading', 'yes');
        this._getReach(data);
    },

    /**
     * Показать ошибку поверх содержимого блока
     * @param {*} content
     */
    showError: function(content) {
        this._progress.update(0);

        this._renderNumber('detailed', 'of', { status: 'empty' });
        this._renderNumber('basic', 'out-of', { status: 'error' });

        BEM.DOM.update(this.elem('title'), BEMHTML.apply({
            block: 'b-crypta-predictor',
            elem: 'error',
            text: content
        }));
        this.setMod('error', 'yes');

    },

    /**
     * Спрятать ошибку
     */
    hideError: function() {
        BEM.DOM.update(this.elem('title'), BEMHTML.apply({
            block: 'b-crypta-predictor',
            elem: 'title-content'
        }));
        this.delMod('error');
    },

    /**
     * Отрисовывает правую цифру
     * @param {String} name - название элемента
     * @param {'of'|'out-of'} type - тип цифры
     * @param {ReachNumberClient} data
     * @private
     */
    _renderNumber: function(name, type, data) {
        BEM.DOM.update(this.elem(name), BEMHTML.apply({
            block: 'b-crypta-predictor',
            elem: 'predict-value',
            elemMods: {
                status: data.status
            },
            type: type,
            data: data
        }));
    },

    /**
     * Обновляет подсказку
     * @param {ReachServerResult.requestId} requestId
     * @private
     */
    _updateHelp: function(requestId) {
        this.blockInside('help', 'b-hintable-popup').setHintContent(BEMHTML.apply({
            block: 'b-crypta-predictor',
            elem: 'help-predictor-content',
            profilesDisabled: this.params.profilesDisabled,
            requestId: requestId
        }));
    },

    /**
     * Выполняет запрос прогноза
     * @param {ReachParams} data - параметры запроса
     */
    _getReach: function(data) {
        // DIRECT-106445: [dna] Ошибка в прогнозаторе
        var requestId = this._requestId = u._.uniqueId('request-id');

        this._request = BEM.blocks['i-web-api-request'].mediareach
            .getReach(u.consts('ulogin'), data)
            .then(function(response) {
                if (requestId === this._requestId) {
                    this._lastReachQuery = data;

                    if (response.success) {
                        this._onReachSuccess(response.result);
                    } else {
                        this._onReachError(response);
                    }
                }

            }.bind(this))
            .catch(function(response) {
                this._lastReachQuery = data;

                if (response.status === 501 && u._.get(response, 'obj.success')) {
                    if (requestId === this._requestId) {
                        this._onReachSuccess(response.obj.result);
                    }
                } else {
                    this._onReachError(response);
                }
            }.bind(this));
    },

    /**
     * @typedef {Object} ReachNumber - цифра прогноза
     *
     * @description одновременно может быть заполнено только одно поле
     *
     * @property {Number} [reach] - цифра точного прогноза
     * @property {Number} [reach_less_then] - цифра примерного прогноза напр. <5000
     * @property {Object} [errors] - массив ошибок, если прогноз не удалось определить
     */

    /**
     * @typedef {Object} ReachServerResult - результат прогноза
     *
     * @property {String} requestId - ид запроса (чтобы можно было найти в логах)
     * @property {ReachNumber} detailed - детальный прогноз с учетом крипты
     * @property {ReachNumber} basic - общий прогноз без учета крипты
     */

    /**
     * Обработчик успешного ответа сервера
     * @param {ReachServerResult} result
     * @private
     */
    _onReachSuccess: function(result) {
        var detailed, basic;

        this.delMod('loading');

        detailed = this._getInfoFromResult(result.detailed);
        basic = this._getInfoFromResult(result.basic);

        if (u._.contains(['empty', 'error'], detailed.status) && basic.status !== 'error') {
            this._progress.update(1);

            this._renderNumber('detailed', 'of', detailed);
            this._renderNumber('basic', 'of', basic);
        } else if (detailed.status === 'error' && basic.status === 'error') {
            this._progress.update(0);

            detailed.status = 'empty';

            this._renderNumber('detailed', 'of', detailed);
            this._renderNumber('basic', 'out-of', basic);
        } else if (u._.contains(['less', 'error'], detailed.status) || basic.status === 'error') {
            this._progress.update(0);

            this._renderNumber('detailed', 'of', detailed);
            this._renderNumber('basic', 'out-of', basic);
        } else {
            this._progress.update(u.numberFormatter.round(detailed.value / basic.value, { precision: 2, roundType: 'ceil' }));

            this._renderNumber('detailed', 'of', detailed);
            this._renderNumber('basic', 'out-of', basic);
        }

        this._updateHelp(result.requestId);
        this.trigger('ready');
    },

    /**
     * Обработчик неудачного ответа с сервера
     * @param {Object} response
     * @private
     */
    _onReachError: function(response) {
        this.delMod('loading');
        this.showError(u._.get(response, 'obj.text'));
        this._updateHelp(u._.get(response, 'obj.requestId'));
        this.trigger('ready');
    },

    /**
     * @typedef {Object} ReachNumberClient - описание цифры для шаблонизации
     *
     * @property {'error'|'normal'|'less'|'empty'} status - статус числа
     * @property {Number|null} value - значение цифры
     * @property {Array<Object>} error - инфа об ошибках
     *
     */

    /**
     * Достает инфу из ответа сервера
     * @param {ReachServerResult} result
     * @returns {ReachNumberClient}
     * @private
     */
    _getInfoFromResult: function(result) {
        var status;

        result || (result = {});

        if (result.errors) {
            status = 'error';
        } else if (result.reach) {
            status = 'normal';
        } else if (result.reach_less_than) {
            status = 'less';
        } else {
            status = 'empty';
        }

        return {
            status: status,
            value: result && (result.reach || result.reach_less_than),
            errors: result && (result.errors || []).map(this._getGoalInfo.bind(this))
        }
    },

    /**
     * Прячет подсказки
     * @private
     */
    _hideHints: function() {
        var hintable = this.findBlockInside('b-hintable');

        this.blockInside('b-hintable-popup').hideHint();
        hintable && hintable.hideHint();
    },

    /**
     * Возвращает данные цели/сегмента по ид
     * @param {Object} error
     * @returns {*}
     * @private
     */
    _getGoalInfo: function(error) {

        if (this._segmentsHash[error.goal_id]) {
            return this._segmentsHash[error.goal_id];
        } else if (this._goalsHash[error.goal_id]) {
            var goal = u._.assign({}, this._goalsHash[error.goal_id]);

            if (!goal.allow_to_use) {
                goal.name = u['retargeting'].getLostGoalName(goal);
            }

            return goal;
        }

        return error;
    },

    destruct: function() {
        this.unbindFromWin('scroll', this._hideHints);
        this.__base.apply(this, arguments);
    }

});
