BEM.DOM.decl('b-date-range-picker', {

    onSetMod: {
        js: function() {
            this._isRelativeRange = !!this.params.isRelativeRange;

            this.setRangeLimits(this.params.limits || {});
            this._updatePeriod(this.getRange());
        }
    },

    /**
     * Выставляет период из шаблона
     * @param {Object} params
     * @param {String} [params.dependId] - зависимость от другого календаря
     * @param {Number|Object} params.start - начало периода
     * @param {Number|Object} params.finish - конец периода
     * @param {jQuery} [domElem] - owner для history-popup
     * @private
     */
    _applyTemplate: function(params, domElem) {
        if (params.type == 'history') {
            this._openHistoryPopup(domElem, this.params.historyRanges);
            this._isRelativeRange = false;
        } else {
            var templateRange = this._getTemplateRange(params);

            this.setRange({
                start: templateRange.start,
                finish: templateRange.finish
            });

            this._isRelativeRange = true;
        }
    },

    /**
     * Вычисляет период шаблона
     * @param {Object} params
     * @param {String} [params.dependId] - зависимость от другого календаря
     * @param {Number|Object} params.start - начало периода
     * @param {Number|Object} params.finish - конец периода
     * @returns {{start: *, finish: *, dependStart: *, dependFinish: *}}
     * @private
     */
    _getTemplateRange: function(params) {
        var depend = {},
            start = params.start,
            finish = params.finish,
            dependPeriodDays,
            days,
            isDateTypeString = function(date) {
                return typeof date === 'string' && u.moment(date).isValid();
            };

        if (params.dependId) {
            depend = this.__self.getPeriodById(params.dependId); // период другого календаря
        }

        // start.period - требует специальной обработки
        if (u._.isObject(start) && start.period) {

            // количество дней в периоде другого календаря
            dependPeriodDays = u.moment(depend.finish).diff(u.moment(depend.start), 'days');

            // что бы даты периодов не пересекались сдвигаем на один день +1
            start = finish = (dependPeriodDays + 1) * params.start.period;
        }

        return {
            start: isDateTypeString(start) ? start : u['b-date-range-picker'].getDateFrom(start, depend.start),
            finish: isDateTypeString(finish) ? finish : u['b-date-range-picker'].getDateFrom(finish, depend.finish)
        }
    },

    _isRelativeRange: false,

    /**
     * Находит, кэширует и возвращает инпуты
     *
     * @returns {{start: BEM, finish: BEM}} Инпуты
     * @private
     */
    _getDateInputs: function() {
        return this._dateInputs || (this._dateInputs = {
            start: this.findBlockInside('start', 'b-date-input'),
            finish: this.findBlockInside('finish', 'b-date-input')
        });
    },

    /**
     * Преобразует строку в дату
     *
     * @param {String} date Дата
     * @returns {*}
     * @private
     */
    _toDate: function(date) {
        return u.moment(date).toDate();
    },

    /**
     * Устанавливает лимит для конкретного календаря
     *
     * @param {'start'|'finish'} type Тип календаря
     * @param {Object} limit Содержит лимиты
     * @private
     */
    _setRangeLimit: function(type, limit) {
        this._getDateInputs()[type].setLimits(
                limit.earlier && this._toDate(limit.earlier),
                limit.later && this._toDate(limit.later));
    },

    /**
     * Устанавливает лимиты в календарях
     *
     * @param {Object} limits Лимиты
     */
    setRangeLimits: function(limits) {
        limits.start && this._setRangeLimit('start', limits.start);
        limits.finish && this._setRangeLimit('finish', limits.finish);
    },

    /**
     * Устанавливает новое значение или возвращает текущее для конкретного инпута
     *
     * @param {'start'|'finish'} type Тип инпута
     * @param {String|Number} [value] Значение даты
     * @returns {String} Значение даты
     * @private
     */
    _dateValue: function(type, value) {
        if (value == undefined) return this._getDateInputs()[type].val();

        this._getDateInputs()[type].val(value);
    },

    /**
     * Текущий выбранный период
     */
    _period: null,

    /**
     * Устанавливает новое значение диапазона
     *
     * @param {Object} ranges Значения периодов
     */
    setRange: function(ranges) {
        this._dateValue('start', ranges.start);
        this._dateValue('finish', ranges.finish);

        this._updatePeriod(ranges);
    },

    /**
     * Возвращает текущие значение диапазона
     *
     * @returns {{start: String, finish: String}} Значения периодов
     */
    getRange: function() {
        return {
            start: this._dateValue('start'),
            finish: this._dateValue('finish')
        }
    },

    /**
     * Возвращает разницу между start и finish
     * @param {String} [measurement] - Supported measurements are years, months, weeks, days, hours, minutes, and seconds.
     * @returns {Number}
     */
    getDiff: function(measurement) {
        return u['b-date-range-picker'].getDiff(this._dateValue('start'), this._dateValue('finish'), measurement);
    },

    /**
     * Возвращает разницу между start и текущей датой
     * @param {String} [measurement] - Supported measurements are years, months, weeks, days, hours, minutes, and seconds.
     * @returns {Number}
     */
    getShiftFromNow: function(measurement) {
        return u.moment(this._dateValue('start')).diff(u.moment(), this._dateValue('start'), measurement);
    },

    /**
     * Добавляет короткое тире между различными датами
     *
     * @param {String} start Начало периода
     * @param {String} finish Конец периода
     * @returns {String}
     * @private
     */
    _addDash: function(start, finish) {
        return start + '&nbsp;&ndash;&nbsp;' + finish;
    },

    /**
     * Преобразовывает период в строку
     *
     * @param {String} start Начало периода
     * @param {String} finish Конец периода
     * @returns {String}
     * @private
     */
    _getStringRange: function(start, finish) {
        start = u.moment(start);
        finish = u.moment(finish);

        if (start.year() != finish.year()) {
            return this._addDash(start.format('DD MMM YYYY'), finish.format('DD MMM YYYY'));
        }

        if (start.month() != finish.month()) {
            return this._addDash(start.format('DD MMMM'), finish.format('DD MMMM YYYY'));
        }

        if (start.days() != finish.days()) {
            return this._addDash(start.date(), finish.format('DD MMMM YYYY'));
        }

        return start.format('DD MMMM YYYY');
    },

    /**
     * Флаг, возвращает true, если выбран относительный промежуток(с помощью link, кроме history)
     * @returns {Boolean}
     */
    isRelativeRange: function() {
        return this._isRelativeRange;
    },

    /**
     * Строит попап с кнопками
     *
     * @param {Array} ranges Массив периодов
     * @returns {BEM} Попап.
     * @private
     */
    _getPopup: function(ranges) {
        if (this._popup) return this._popup;

        var popup = $(BEMHTML.apply({
            block: 'popup',
            mix: [{ block: 'b-date-range-picker', elem: 'history-popup' }],
            mods: { theme: 'ffffff' },
            content: [
                    { elem: 'tail' },
                {
                    elem: 'content',
                    content: ranges.map(function(r) {
                        return {
                            elem: 'item',
                            content: {
                                block: 'link',
                                js: { start: r.start, finish: r.finish },
                                mods: { pseudo: 'yes' },
                                content: this._getStringRange(r.start, r.finish)
                            }
                        };
                    }, this)
                }
            ]
        }));

        BEM.blocks['link'].on(popup, 'click', function(e) {
            this.setRange(e.block.params);
            this._popup.hide();
        }, this);

        this._popup = popup.bem('popup');

        return this._popup;
    },

    /**
     * Показывает попап по клику
     *
     * @param {jQuery} domElem DOM-нода кнопки
     * @param {Array} ranges Массив периодов
     * @private
     */
    _openHistoryPopup: function(domElem, ranges) {
        this._getPopup(ranges).toggle(domElem);
    },

    /**
     * Записывает период текущего календаря в общее хранилище по id
     * @param {Object} period
     * @param {String} period.start - начало периода
     * @param {String} period.finish - конец периода
     * @returns {BEM}
     * @private
     */
    _updatePeriod: function(period) {
        this._period = period;

        if (this.params.id) {
            this.__self.setPeriodById(period, this.params.id);

            this.channel('b-date-range-picker').trigger('period-change', {
                id: this.params.id,
                period: period
            });
        }

        return this;
    }

}, {

    /**
     * Хранилище периодов
     */
    _datesById: null,

    /**
     * Записывает период текущего календаря в общее хранилище по id
     * @param {Object} period
     * @param {String} period.start - начало периода
     * @param {String} period.finish - конец периода
     * @param {String} id
     */
    setPeriodById: function(period, id) {
        this._datesById || (this._datesById = {});

        this._datesById[id] = period;
    },

    /**
     * Получает период календаря по id
     * @param {String} id
     * @returns {Object}
     */
    getPeriodById: function(id) {
        return this._datesById[id];
    },

    live: function() {
        this
            .liveInitOnBlockInsideEvent('click', 'link', function(e) {
                this._applyTemplate(e.block.params, e.block.domElem);
            })
            .liveInitOnBlockInsideEvent('change', 'b-date-input', function(e) {
                this._isRelativeRange = false;
                this.trigger('change', this.getRange());
            });

        return false;
    }

});
