/**
 * Событие сохранения выбора списка интересов
 * @event b-edit-interests-targeting#save
 * @type {Object}
 * @property {String[]} selectedIds идентификаторы выбранных интересов
 *
 * @fires b-edit-interests-targeting#save событие сохранения выбора списка интересов
 * @fires b-edit-interests-targeting#cancel событие нажатия на кнопку отмены
 */
BEM.DOM.decl({ block: 'b-edit-interests-targeting', implements: 'i-modal-popup-inner-block-interface' }, {
    onSetMod: {
        js: function() {
            this._chooser = this.findBlockInside('chooser', 'b-chooser');
            this._selectedItemsList = this.findBlockInside('selected-items', 'b-items-list');
            this._clearLink = this.findBlockOn('clear-items', 'link');
            this._cancelBtn = this.findBlockOn('cancel', 'button');
            this._saveBtn = this.findBlockOn('save', 'button');
            this._subscribe = BEM.create('i-subscription-manager');

            this._subscribe
                .on(
                    BEM.blocks['b-edit-interests-targeting-item'],
                    this.elem('selected-items'),
                    'delete',
                    this._onSelectedItemDelete,
                    this
                )
                .on(this._chooser, 'change', this._onChooserChange, this)
                .on(this._clearLink, 'click', this._onClearLinkClick, this)
                .on(this._cancelBtn, 'click', this._onCancelBtnClick, this)
                .on(this._saveBtn, 'click', this._onSaveBtnClick, this);
        }
    },

    /**
     * Инициализирует блок
     * @param {Object} interestsTree данные о всех категорях интересов
     * @param {String} interestsTree.id идентификатор категории интересов
     * @param {String} interestsTree.name имя интереса
     * @param {Number} interestsTree.orderIndex индекс категории интереса для сортировки
     * @param {Number} interestsTree.available флаг, определяющий доступность интереса для выбора
     * @param {String} [interestsTree.parentId] идентификатор родительской категории
     * @param {String[]} selectedInterestsIds идентификаторы выбранных категорий интересов
     */
    init: function(interestsTree, selectedInterestsIds) {
        this._interestsTree = this._transformInterestsTreeData(interestsTree);

        this._initialSelectedInterestsIds = selectedInterestsIds || [];

        // формируем список для выбора интереса
        this._getAllInterestsList().forEach(function(item) {
            this._chooser.add(this._generateChooserItem(item));
            item.isSelected && this._selectedItemsList.addItem(this._generateSelectedListItem(item));
        }, this);

        this._updateControlsState();
    },

    /**
     * Были ли изменения
     * @returns {$.Deferred<Boolean>}
     */
    isChanged: function() {
        var deferred = $.Deferred();

        deferred.resolve(this._isSelectionChanged());

        return deferred;
    },

    /**
     * Удаляет блок и подписки на события
     */
    destruct: function() {
        this._subscribe.dispose();

        return this.__base.apply(this, arguments);
    },

    /**
     * Преобразует дерево интересов в удобный для работы блока вид
     * @param {Object} interestsTreeData исходные данные категорий интересов
     * @returns {Object}
     * @private
     */
    _transformInterestsTreeData: function(interestsTreeData) {
        return Object.keys(interestsTreeData).reduce(function(result, itemId) {
            var treeItem = interestsTreeData[itemId];

            if (treeItem.available) {
                treeItem.path = u['interests-targeting'].getPath(treeItem.parentId, interestsTreeData);

                result[itemId] = treeItem;
            }

            return result;
        }, {});
    },

    /**
     * Возвращает массив всех доступных для выбра категорий интересов, отсортированный по полю orderIndex
     * и с признаком выбранности каждого из его эелементов
     * @returns {Object[]}
     * @private
     */
    _getAllInterestsList: function() {
        var result = [],
            interestsTree = this._interestsTree,
            selectedInterestsIdsMap = (this._initialSelectedInterestsIds || []).reduce(function(result, id) {
                result[id] = true;

                return result;
            }, {});

        Object.keys(interestsTree).forEach(function(listItemId) {
            var treeItem = interestsTree[listItemId];

            // добавляем информацию о выбранности и родительских категориях
            result.push(u._.extend({}, treeItem, {
                isSelected: !!selectedInterestsIdsMap[listItemId]
            }));
        }, this);

        // сортируем массив в соответсвии с их порядковыми индексами
        result.sort(function(a, b) {
            return a.orderIndex - b.orderIndex;
        });

        return result;
    },

    /**
     * Формирует bemjson для элемента блока `b-chooser`
     * @param {Object} interestData данные интереса
     * @returns {Object}
     * @private
     */
    _generateChooserItem: function(interestData) {
        var interestId = interestData.id,
            interestName = interestData.name;

        return {
            elemMods: interestData.isSelected ? { selected: 'yes' } : {},
            js: {
                search: { indexed: interestName }
            },
            name: interestId,
            content: {
                block: 'b-edit-interests-targeting-item',
                itemData: {
                    id: interestId,
                    name: interestName,
                    path: interestData.path
                },
                isChooserItem: true
            }
        };
    },

    /**
     * Формирует bemjson для элемента списка выбранных интересеов
     * @param {Object} interestData данные интереса
     * @returns {Object}
     * @private
     */
    _generateSelectedListItem: function(interestData) {
        var interestId = interestData.id;

        return {
            id: interestId,
            orderIndex: interestData.orderIndex,
            content: {
                block: 'b-edit-interests-targeting-item',
                mix: {
                    block: 'b-edit-interests-targeting',
                    elem: 'selected-item'
                },
                itemData: {
                    id: interestId,
                    name: interestData.name,
                    path: interestData.path
                },
                isSelectedItemsListItem: true
            }
        };
    },

    /**
     * Обработчик события изменения выбора интереса
     * @param {Event} event
     * @param {Object[]|Object} items данные выбранных категорий интересов
     */
    _onChooserChange: function(event, items) {
        !Array.isArray(items) && (items = [items]);

        items.forEach(function(item) {
            var id = item.name,
                itemData = this._interestsTree[id];

            item.selected ?
                this._selectedItemsList.addItem(this._generateSelectedListItem(itemData)) :
                this._selectedItemsList.removeItem(id);
        }, this);

        this._updateControlsState();
    },

    /**
     * Обработчик события нажатия на кнопку удаления одного из выбранных интересов
     * @param {Event} event
     * @param {Object} data
     * @param {Object} data.id идентификатор категории интереса для удаления
     */
    _onSelectedItemDelete: function(event, data) {
        this._chooser.uncheck(data.id);
    },

    /**
     * Обработчик события нажатия на ссылку 'Очистить все'
     */
    _onClearLinkClick: function() {
        this._chooser.uncheckAll();
    },

    /**
     * Меняет состояние элементов, зависящих от количества выбранных категорий интересов
     */
    _updateControlsState: function() {
        var selectedItems = this._chooser.getSelected(),
            selectedItemsLength = selectedItems.length;

        this.setMod(this.elem('selected-items-num'), 'hidden', selectedItemsLength === 0 ? 'yes' : '');
        this.setMod(this.elem('empty-list-text'), 'hidden', selectedItemsLength === 0 ? '' : 'yes');
        selectedItemsLength && this.elem('selected-items-num-value').text(selectedItemsLength);

        this._clearLink.setMod('disabled', selectedItemsLength === 0 ? 'yes' : '');
        this._saveBtn.setMod('disabled', this._isSelectionChanged() ? '' : 'yes');
    },

    /**
     * Определяет изменился ли список выбранных интересов
     * @returns {Boolean}
     */
    _isSelectionChanged: function() {
        return !u._.isEqual(this._initialSelectedInterestsIds.sort(), this._getSelectedInterestsIds().sort());
    },

    /**
     * Возвращает список идентификаторов выбранных интересов
     * @returns {String[]}
     */
    _getSelectedInterestsIds: function() {
        return this._chooser.getSelected().map(function(item) {
            return item.name;
        });
    },

    /**
     * Обработчик события нажатия на кнопку Сохранения
     */
    _onSaveBtnClick: function() {
        this.trigger('save', { selectedIds: this._getSelectedInterestsIds() });
    },

    /**
     * Обработчик события нажатия на кнопку Отмены
     */
    _onCancelBtnClick: function() {
        this.trigger('cancel');
    }
});
