/**
 * @fires b-edit-metrics-key-goals#change событие изменения состояния контрола
 */
BEM.DOM.decl({ block: 'b-edit-metrics-key-goals' },
    {
        onSetMod: {
            js: function() {
                this._model = this.findBlockOn('i-glue').model;
                this._subscriptionManager = BEM.create('i-subscription-manager');

                this._subscriptionManager.wrap(this._model)
                    .on('itemsCount', 'change', this._onItemsCountChange, this)
                    .on('selectedGoalIds', 'change', this._onSelectedGoalIdsChange, this)
                    .on('hasChanges', 'change', this._updateSaveBtnAvailability, this)
                    .on('hasErrors', 'change', this._updateError, this);

                this._model.set('nextItemId', this._model.get('itemsCount'));
            }
        },

        /**
         * Удаляет блок и подписки на события
         */
        destruct: function() {
            this.unsubscribe();
            this._subscriptionManager.destruct();

            return this.__base.apply(this, arguments);
        },

        /**
         * Убирает подписки на события
         * */
        unsubscribe: function() {
            this._subscriptionManager.dispose();
        },

        /**
         * Валидирует состояние контрола
         * @returns {Boolean}
         */
        validate: function() {
            var items = this.findBlocksOn('item', 'b-edit-metrics-key-goals-item'),
                errors = u._.uniq(items.reduce(function(acc, item) {
                    return acc.concat(item.validate());
                }, []));
            if (errors.length) {
                BEM.DOM.update(this.elem('error'), errors[0]);
                this._model.set('hasErrors', true);
                return false;
            }
            return true;
        },

        /**
         * Возвращает текущее значение контрола
         * @returns {Array<{goalId: String, price: Number}>}
         */
        getValue: function() {
            return u._.map(this.findElem('item'), function(item) {
                var itemModel = this._getItemModelById(this.getMod($(item), 'id')).toJSON();

                return {
                    goalId: itemModel.goalId,
                    price: itemModel.price
                };
            }, this);
        },

        /**
         * Обработчик события изменения поля hasErrors в модели
         * */
        _updateError: function(event, hasErrorsField) {
            if (!hasErrorsField.value) {
                BEM.DOM.update(this.elem('error'), '');
            }
        },

        /**
        * Добавляет контрол для выбора цели к списку
        */
        _appendItem: function() {
            var model = this._model,
                id = model.get('nextItemId'),
                itemsCount = model.get('itemsCount');

            BEM.DOM.append(
                this.elem('items'),
                BEMHTML.apply({
                    block: 'b-edit-metrics-key-goals-item',
                    mix: [
                        { block: 'b-edit-metrics-key-goals', elem: 'item', elemMods: { id: id } }
                    ],
                    id: id,
                    goals: this.params.goals,
                    index: model.get('nextItemId'),
                    disabledGoalsIds: model.get('selectedGoalIds'),
                    currency: this.params.currency,
                    isDeleteEnabled: true
                })
            );

            model.set('itemsCount', itemsCount + 1);
            model.set('nextItemId', id + 1);

            setTimeout((function() {
                this.domElem && u.scrollNodeTo(this.findElem('item', 'id', id), this.domElem, { over: 50 });
            }).bind(this), 100);
        },

        /**
         * Обрабатывает событие изменения количества элементов в списке целей
         * */
        _onItemsCountChange: function(event, itemsCountField) {
            this._updateAddGoalBtnAvailability(itemsCountField.value);
            this._updateSaveBtnAvailability();
            this._onChange();
        },

        /**
         * Обрабатывает событие изменения выбранных целей
         * */
        _onSelectedGoalIdsChange: function() {
            this._updateSaveBtnAvailability();
            this._onChange();
        },

        /**
        * Регулирует доступность кнопок удаления и добавления контролов выбора цели
        */
        _updateAddGoalBtnAvailability: function(itemsCount) {
            this.setMod(this.elem('add'), 'hidden', itemsCount < u._.size(this.params.goals) ? '' : 'yes');
        },

        /**
         * Метод, сигнализирующий о том, что нужно активировать кнопку "сохранить"
         * */
        _updateSaveBtnAvailability: function() {
            if (this._model.get('selectedGoalIds').length === this._model.get('itemsCount')) {
                this.trigger('enableSave', true);
            } else {
                this.trigger('enableSave', false);
            }
        },

        /**
        * Возвращает модель элемента списка целей по идентификатору
        * @param {String} id
        */
        _getItemModelById: function(id) {
            return BEM.MODEL.getOne({ name: 'b-edit-metrics-key-goals-item', id: id })
        },

        /**
        * Обработчик события изменения состояния контрола
        */
        _onChange: function() {
            this.trigger('change');
        }

    },
    {
        live: function() {
            this.liveBindTo('add', 'click', function() {
                this._appendItem();
            });

            this.liveInitOnBlockInsideEvent('change', 'input', function() {
                this._onChange();
            });
        }
    }
);
