/**
 * @event b-edit-regions-popup#save
 * @type {Object}
 * @property {String} geo id выбранных регионов, разделенных запятой
 * @property {String} geoText текстовое описание выбранных регионов
 * @property {Object} changesInfo информация о том, в каких галках регионов были изменения
 * @property {Object} resolvedRegions список регионов, для которых регионы в группах кампании различались
 * и были приведены к единому виду
 * @property {Boolean} wasReset флаг, идентифицирующий факт сброса всех регионов
 * @property {Boolean} hasIndependentRegionsWarning
 * @property {Boolean} [isBidsCorrectionsModeOn] включен ли режим отображения ставок
 * @property {Object} [bid] информация о установленных для регионов ставках
 *
 * @fires b-edit-regions-popup#save событие сохранения выбора списка регионов показа и их ставок
 * @fires b-edit-regions-popup#cancel событие нажатия на кнопку отмены
 */
BEM.DOM.decl({ block: 'b-edit-regions-popup', implements: 'i-modal-popup-inner-block-interface' }, {
    onSetMod: {
        js: function() {
            BEM.blocks['i-utils'].graspSelf.call(this, {
                _regionsBlock: 'b-regions on regions',
                _saveBtn: 'button on save',
                _cancelBtn: 'button on cancel',
                _progressSpin: 'spin2 on progress-spin'
            });

            this._subscribe = BEM.create('i-subscription-manager');

            this.setSelectedRegions(this.params.selectedRegions);
            this._initialRegionsData = this._regionsBlock.getRegionsData();

            this._subscribe
                .on(this._cancelBtn, 'click', this._onCancelBtnClick, this)
                .on(this._saveBtn, 'click', this._onSaveBtnClick, this);
        },
        progress: function(modName, modVal) {
            this._saveBtn.setMod('disabled', modVal);
            this._progressSpin.setMod('progress', modVal);
        }
    },

    /**
     * Инициализирует блок
     * @param {Object} params - объект с параметрами
     * @param {BEM.DOM<popup>} params.parentPopup - родительский попап
     */
    initialize: function(params) {
        this._regionsBlock.setParentPopup(params.parentPopup);
    },

    /**
     * Устанавливает выбор регионов показа
     * @param {String} idsString строка регионов показа, разделенных запятой для установки выбора
     */
    setSelectedRegions: function(idsString) {
        this._regionsBlock.setSelectedRegions(idsString.split(/[\s|,]+/));
    },

    /**
     * Были ли изменения
     * @returns {$.Deferred<Boolean>}
     */
    isChanged: function() {
        var deferred = $.Deferred();

        deferred.resolve(this._isSelectionChanged());

        return deferred;
    },

    /**
     * Удаляет блок и подписки на события
     */
    destruct: function() {
        this._subscribe.dispose();

        return this.__base.apply(this, arguments);
    },

    /**
     * Определяет изменился ли список выбранных регионов показа
     * @returns {Boolean}
     */
    _isSelectionChanged: function() {
        var currentRegionsBlockData = this._regionsBlock.getRegionsData(),
            initialRegionsBlockData = this._initialRegionsData;

        return initialRegionsBlockData.ids.join('') !== currentRegionsBlockData.ids.join('') ||
            Object.keys(currentRegionsBlockData.resolvedRegions).length ||
            initialRegionsBlockData.isBidsCorrectionsModeOn !== currentRegionsBlockData.isBidsCorrectionsModeOn ||
            !u._.isEqual(initialRegionsBlockData.bids, currentRegionsBlockData.bids);
    },

    showError: function(errorText) {
        BEM.DOM.replace(this.findElem('errors'), BEMHTML.apply({
            block: 'b-edit-regions-popup',
            elem: 'errors',
            text: errorText
        }));
    },

    hideError: function() {
        this.setMod(this.findElem('errors'), 'visible', 'no');
    },

    /**
     * Обработчик события нажатия на кнопку Сохранения
     */
    _onSaveBtnClick: function() {
        var regionsData = this._regionsBlock.getRegionsData();

        this.trigger('save', {
            geo: regionsData.ids.join(',') || 0,
            geoText: regionsData.text,
            resolvedRegions: regionsData.resolvedRegions,
            changesInfo: regionsData.changesInfo,
            wasReset: regionsData.wasReset,
            hasIndependentRegionsWarning: this._regionsBlock.hasMod('has-independent-filter-warning'),
            isBidsCorrectionsModeOn: regionsData.isBidsCorrectionsModeOn,
            bids: regionsData.bids
        });
    },

    /**
     * Обработчик события нажатия на кнопку Отмены
     */
    _onCancelBtnClick: function() {
        this.trigger('cancel');
    }
});
