/**
 * Данные модели фильтра, пригодные для сохранения
 * @typedef {Object} FilterEditData
 */

BEM.DOM.decl({
    name: 'b-feed-filter-edit',
    baseBlock: 'i-filter-edit',
    implements: 'i-modal-popup-inner-block-interface'
}, {

    /**
     * Были ли изменения
     * @returns {$.Deferred<Boolean>}
     */
    isChanged: function() {
        var deferred = $.Deferred();

        deferred.resolve(this.model.get('hasChanges'));

        return deferred;
    },

    /**
     * Выполняет подписку на события через i-subscription-manager
     * @private
     */
    _initEvents: function() {
        this.__base.apply(this, arguments);

        var changingFields = [
            'tab',
            'condition',
            'conditionTree',
            'filter_name',
            'price_cpc',
            'price_cpa',
            'price',
            'price_context',
            'available',
            'target_funnel',
            'is_suspended',
            'retargetings'
        ];

        if (this._getSettingsTabs()) {
            this._getSubscriptionManager().on(this._getSettingsTabs(), 'change', function(e, data) {
                var businessType = this.model.getFeedData().business_type,
                    feedType = this.model.getFeedData().feed_type;

                this.model.set('tab', u.feedFilterData.getTabs(feedType, businessType)[data.currentIndex].value);
            }, this);
        }

        this._getSubscriptionManager().wrap(this.model)
            .on(changingFields.join(' '), 'change', function() {
                var hasChanges = changingFields.some(function(field) { return this.model.isChanged(field) }, this);

                if (hasChanges && this.model.isValid()) {
                    !this.hasMod('cantsave', 'yes') && this._getSaveBtnBlock().delMod('disabled');
                } else {
                    this._getSaveBtnBlock().setMod('disabled', 'yes');
                }

                this.model.set('hasChanges', hasChanges);
            }, this)
            .on('tab', 'change', function() {
                // при переключении табов подчищать ошибки в состоянии сохранения фильтра
                this.updateStatusMessage('');
            }, this)
            .on('isSaving', 'change', function(e, data) {
                this.toggleMod(this.elem('save'), 'saving', 'yes', data.value);

                this._getSaveBtnBlock().toggleMod('disabled', 'yes', data.value);
                this._getCancelBtnBlock().toggleMod('disabled', 'yes', data.value);
            }, this)
            .on('is_deleted', 'change', function(e, data) {
                this._emitEditCompleteEvent();
            }, this)
            .on('isRetargetingAvailable', 'change', function(e, data) {
                !this.model.get('skipRetargeting') && this._toggleRetargetingView(data.value);
            }, this);

    },

    /**
     * Триггерит событие завершения редактирования
     * @event filter:edit:completed
     * @private
     */
    _emitEditCompleteEvent: function() {
        this.trigger('filter:edit:completed', {
            id: this.model.id,
            dataToSave: this._getData()
        });
    },

    /**
     * Возвращает данные модели в пригодном для сохранения виде
     * @returns {FilterEditData}
     */
    _getData: function() {
        return u._.extend({}, this.model.prepareDataToDM(this.model.toJSON()));
    },

     /**
     * Обновляет текст-состояние сохранения фильтра
     * @param {String} text - строка-состояние
     */
    updateStatusMessage: function(text) {
        this.elem('edit-status').text(text);
    },

    /**
     * Возвращает блок с табами переключения типа настройки
     * @returns {BEM}
     * @private
     */
    _getSettingsTabs: function() {
        return this._settingsTabs || (this._settingsTabs = this.findBlockOn('settings-tabs', 'tabs'));
    },

    /**
     * Получение данных фида
     * @returns {Object}
     * @private
     */
    _getFeedData: function() {
        return this.model.getFeedData();
    },

    /**
     * Обновляет содержимое таба настроек по типу type
     * @private
     */
    _renderSettings: function() {
        var filter = this.model.toJSON(),
            feedData = this._getFeedData(),
            feedType = feedData.feed_type,
            businessType = feedData.business_type,
            categories = feedData.categories;

        BEM.DOM.update(this.elem('filter-target-settings-area'), BEMHTML.apply({
            block: 'b-feed-filter-edit',
            elem: 'filter-settings-target',
            filter: filter,
            feedType: feedType,
            businessType: businessType,
            categories: categories
        }));
    },

    /**
     * Выводит блок для задания ретаргетинга и инициализирует его
     * @private
     */
    _renderRetargeting: function() {
        var filter = this.model.toJSON(),
            retAreaElem = this.elem('filter-retargeting-area'),
            groupRetargetingBlock;

        BEM.DOM.update(retAreaElem, BEMHTML.apply({
            block: 'b-feed-filter-edit',
            elem: 'filter-retargeting',
            elemMods: {
                disabled: this.getMod(retAreaElem, 'disabled')
            },
            filter: filter,
            conditions: BEM.MODEL.get('m-retargeting-condition').map(function(condition) {
                return condition.toJSON();
            }, this)
        }));

        groupRetargetingBlock = this.findBlockInside('b-group-retargeting2');

        groupRetargetingBlock.initialize({
            parentPopup: this._parentPopup
        });

        this._getSubscriptionManager().wrap(this.model).on('retargetings', 'change', function() {
            groupRetargetingBlock.renderConditions(this.model.get('retargetings'));
        });

        this._getSubscriptionManager().wrap(groupRetargetingBlock).on('conditions:update', function(event, data) {
            this.model.set('retargetings', data.conditions);
        }, this);
    },

    /**
     * Выводит блок для задания ретаргетинга
     * @param {Boolean} flag - флаг, показывать ли строку с ретаргетингом
     * @private
     */
    _toggleRetargetingView: function(flag) {
        this.setMod(this.elem('row', 'setting', 'retargeting'), 'visible', flag ? 'yes' : '');
    },

    /**
     * Инициализирует блок
     * @override
     * @param {Object} params - объект с параметрами
     * @param {BEM.DOM<popup>} params.parentPopup - родительский попап
     */
    initialize: function(params) {
        this._parentPopup = params.parentPopup;
        // Сервер принимает тип фильтра в формате "retail_YandexMarket"
        this._currentFilterType = this.params.businessType + '_' + this.params.feedType;

        return this.__base.apply(this, arguments);
    },

    /**
     * Реакция на событие "конфиг для фильтров загрузился"
     * @override
     * @private
     */
    _onConfigLoaded: function() {
        //инициализируем и проклеиваем модель
        this.glue();

        this.model.init().fix();

        this._renderSettings();

        if (!this.model.get('skipRetargeting')) {
            this._renderRetargeting();
            this._toggleRetargetingView(this.model.get('isRetargetingAvailable'));
        }

        u.graspSelf.call(this, {
            _list: '? filter-conditions-list inside',
            _tree: '? filter-conditions-tree inside',
            _funnel: '? b-feed-filter-target-funnel inside'
        });

        this._audience = this._funnel && this._funnel.initBlock();

        this._initEvents();

        this.trigger('repaint');
    },

    /**
     * Обработчик события "клик на кнопку Сохранить"
     * @override
     * @private
     */
    _onSaveBtnClick: function() {
        var validateResult = {},
            currentTab = this.model.get('tab'),
            filterValidateResult = {
                isValid: true
            },
            data = this.model.validate();

        if (currentTab == 'tree') {
            filterValidateResult = this._tree.validate();
        } else if (currentTab == 'condition') {
            filterValidateResult = this._list.validate();
        }

        this._clearErrors();

        validateResult = u._.extend({}, data.errorsData);

        if (!filterValidateResult.isValid) {
            validateResult.filter = filterValidateResult.errors;
        }

        validateResult.valid = data.valid && filterValidateResult.isValid;

        validateResult.valid ?
            this._emitEditCompleteEvent() :
            this._showErrors(validateResult);
    }
});
