/**
 * @fires b-feed-filter#filter:edit:completed
 */
BEM.DOM.decl({ name: 'b-feed-filters', baseBlock: 'i-glue' }, {
    onSetMod: {
        js: function() {

            this.__base();

            this.model.init(this.params.groupModelParam).fix();

            this._initEvents();

            !this.hasMod('state', 'disabled') && this._toggleAddAbility(this.model.get('canAddFilter'));
        },
        state: function(modName, modVal) {
            this._toggleAddAbility(modVal !== 'disabled');
        }
    },

    /**
     * Выполняет подписку на события через i-subscription-manager
     * @private
     */
    _initEvents: function() {
        var subscriptionManager = BEM.create('i-subscription-manager');

        subscriptionManager.wrap(this.findBlockOn('i-controls-overseer'))
            .on('add', function() {
                this.model.get('canAddFilter') && this._addItem();
            }, this)
            .on('remove', function(e, params) {
                this._removeItem(params.data.itemId, params.target.domElem);
            }, this)
            .on('edit', function(e, params) {
                this._editItem(params.data.itemId);
            }, this);

        subscriptionManager.wrap(this.model)
            .on('canAddFilter', 'change', function(e, data) {
                !this.hasMod('disabled', 'yes') && this._toggleAddAbility(data.value);
            }, this);

        this.on('destruct', function() { subscriptionManager.dispose(); });
    },

    /**
     * Перерисовывает список, запросив актуальные данные
     */
    actualizeFilters: function() {
        this.model.syncFromDM();

        this._refreshList();
    },

    /**
     * Удаляет все фильтры
     */
    clearFilters: function() {
        this.model.destroyAllFeedFilters();
        this._refreshList();
    },

    /**
     * Удаляет модель фильтра и строку списка
     * @param {String} id - id фильтра
     * @param {Jquery} target - JQ объект с инициатором события
     * @private
     */
    _removeItem: function(id, target) {
        this.model.destroyFeedFilter(id);

        this._removeRow(target);
    },

    /**
     * Создает модель фильтра и строчку в список
     * @private
     */
    _addItem: function() {
        var dm = this.model.getDM(),
            model = this.model
            .get('items')
            .add({
                filter_id: this.model.createFeedFilter(),
                available: false,
                adgroupModelId: dm.get('modelId'),
                adgroup_type: dm.get('adgroup_type'),
                currency: this.model.getCurrency(),
                strategy: this.model.getStrategy()
            }),
            lastIndex = this.model.get('items').length() - 1;

        this._openEditPopup(model)
            .then(function(data) {
                this._appendRow(data, lastIndex);
            }.bind(this))
            .fail(function() {
                // при отмене нужно удалить временную модель, создаваемую при добавлении
                this.model.destroyFeedFilter(model.get('filter_id'));
            }.bind(this));
    },

    /**
     * Открывает попап для редактирования фильтра с указанным ID
     * @param {String} id id фильтра
     * @private
     */
    _editItem: function(id) {
        this._openEditPopup(this.model.get('items').getById(id))
            .then(function() {
                this._refreshList();
            }.bind(this));
    },

    /**
     * Перерисовывает список фильтров
     * @private
     */
    _refreshList: function() {
        var filters = this.model.toJSON().items;

        BEM.DOM.update(this.elem('items'),
            BEMHTML.apply(filters.map(function(filter) {
                return {
                    block: 'b-feed-filters',
                    elem: 'item',
                    filter: filter
                };
            })));
    },

    /**
     * Шаблонизирует и добавляет в список новую строчку
     * @param {Object} data - данные модели фильтра
     * @param {Number} index - индекс добавляемой строки
     * @private
     */
    _appendRow: function(data, index) {
        BEM.DOM.append(this.elem('items'), BEMHTML.apply({
            block: 'b-feed-filters',
            elem: 'item',
            filter: data,
            errorPath: this.params.errorPath + '[' + index + ']'
        }));
    },

    /**
     * Удаляет строчку из списка
     * @private
     */
    _removeRow: function(node) {
        this.elemInstance(this.closestElem(node, 'item')).destruct();
    },

    /**
     * Открывает попап для редактирования условия
     * @param {Object} model данные модели
     * @returns {$.Deferred}
     * @private
     */
    _openEditPopup: function(model) {
        var strategy = this.model.getStrategy(),
            groupDM = this.model.getDM(),
            isDynamicGroup = groupDM.get('adgroup_type') == 'dynamic',
            deferred = $.Deferred(),
            // @todo убрать в https://st.yandex-team.ru/DIRECT-77231
            popup = BEM.DOM.blocks['b-modal-popup-decorator'].create2(null, { bodyScroll: false }, $),
            feedType = this.model.getFeedData().feed_type,
            businessType = this.model.getFeedData().business_type,
            cancelSubscription = BEM.create('i-subscription-manager');

        // при закрытии считаем, что редактирование/добавление отменено и подчищаем
        cancelSubscription.wrap(popup).on('close', function() {
            cancelSubscription.dispose();

            deferred.reject();
        });

        // блокируем возможность редактирования/удаления/добавления
        this
            ._toggleAddAbility(false)
            ._toggleRemoveAbility(false)
            ._toggleEditAbility(false);

        // сначала показываем попап, а уже потом задаём его контент, так как он
        // содержит блоки требующие от попапа находиться в DOM для инициализации
        popup
            .show()
            .on('close-blocked', function() {
                BEM.blocks['b-confirm'].open({
                    message: iget2('b-feed-filters', 'izmeneniya-ne-budut-sohraneny', 'Изменения не будут сохранены. Продолжить?'),
                    onYes: function() { popup.hide({ force: true }); }
                });
            }, this)
            .on('close', function() {
                // возвращаем возможность редактирования/удаления/добавления
                this
                    ._toggleAddAbility(this.model.get('canAddFilter'))
                    ._toggleRemoveAbility(true)
                    ._toggleEditAbility(true);

                popup.destruct();
            }, this)
            .setPopupContent({
                block: 'b-feed-filter-edit',
                mods: {
                    'no-prices': (strategy.name || strategy.net.name) == 'autobudget_roi' ? 'yes' : ''
                },
                businessType: businessType,
                feedType: feedType,
                noRetargeting: isDynamicGroup || model.get('retargetings').length === 0,
                noTargetAudience: isDynamicGroup,
                filter: u._.extend(model.toJSON(), { feed: this.model.getFeedData() }),
                hintUrl: isDynamicGroup ?
                    u.getHelpUrl('dynamic-text-ads-feed-filters') :
                    u.getHelpUrl('smart-banners-feed-filters')
            })
            .initialize({ parentPopup: popup.getPopup() }).then(function(filter) {
                filter
                    .on('filter:edit:completed', function(e, resp) {
                        var editBlock = e.target,
                            modelData = resp.dataToSave;

                        editBlock.model.set('isSaving', true);

                        this.trigger('filter:edit:completed', {
                            modelData: modelData,
                            controller: {
                                save: function() {
                                    this.model.saveFeedFilter(modelData)
                                        .then(function() {
                                            // отписываемся от события закрытия, обработки которого не нужны при успешном сохранении
                                            cancelSubscription.dispose();

                                            this.model.syncFromDM();

                                            deferred.resolve(modelData);

                                            popup.hide({ force: true });
                                        }.bind(this))
                                        .fail(function(errors) {
                                            //обновляем сообщение об ошибке при сохранении фильтра и продолжаем работу не закрывая попап
                                            editBlock.updateStatusMessage(errors.map(function(error) {
                                                return error.message;
                                            }).join('. '));
                                        })
                                        .always(function() {
                                            editBlock.model.set('isSaving', false);
                                        });
                                }.bind(this),
                                onRequestError: function() {
                                    editBlock.model.set('isSaving', false);
                                    BEM.blocks['b-confirm'].alert(iget2('b-feed-filters', 'ne-udalos-sohranit-filtr', 'Не удалось сохранить фильтр.'));
                                },
                                showErrors: function(errors) {
                                    editBlock.model.set('isSaving', false);
                                    editBlock._showErrors(errors);
                                }
                            }
                        });

                    }.bind(this))
                    .on('filter:edit:canceled', function() {
                        // DIRECT-59123: отмена закрывает попапы с force
                        popup.hide({ force: true });
                    });
            }.bind(this));

        return deferred;
    },

    /**
     * Удаляет блок, модель и подписки i-subscription-manager
     * @override
     */
    destruct: function() {
        this.trigger('destruct');

        this.model.destruct();
        this.__base.apply(this, arguments);
    },

    /**
     * Снимает / устанавливает disabled модификатор с кнопки добавления условия
     * @param {Boolean} flag - флаг, выключить/включить возможность добавления
     * @returns {BEM}
     * @private
     */
    _toggleAddAbility: function(flag) {
        this._getAddControl().toggleMod('state', 'disabled', !flag);

        return this;
    },

    /**
     * Снимает / устанавливает disabled модификатор с кнопок удаления условия
     * @param {Boolean} flag - флаг, выключить/включить возможность удаления
     * @returns {BEM}
     * @private
     */
    _toggleRemoveAbility: function(flag) {
        this._getRemoveControls().forEach(function(control) {
            control.toggleMod('state', 'disabled', !flag);
        });

        return this;
    },

    /**
     * Снимает / устанавливает disabled модификатор с кнопок редактирования условия
     * @param {Boolean} flag - флаг, выключить/включить возможность редактирования
     * @returns {BEM}
     * @private
     */
    _toggleEditAbility: function(flag) {
        this._getEditControls().forEach(function(control) {
            control.toggleMod('state', 'disabled', !flag);
        });

        return this;
    },

    /**
     * Возвращает блок ссылки "Добавить"
     * @returns {BEM}
     * @private
     */
    _getAddControl: function() {
        return this._addControl || (this._addControl = this.findBlockOn('add-control', 'b-control-add-button'));
    },

    /**
     * Возвращает блоки кнопок "Удалить"
     * @returns {BEM}
     * @private
     */
    _getRemoveControls: function() {
        return this.findBlocksInside('b-control-remove-button');
    },

    /**
     * Возвращает блоки кнопок "Изменить"
     * @returns {BEM}
     * @private
     */
    _getEditControls: function() {
        return this.findBlocksInside('b-control-edit-button');
    }
});
