BEM.MODEL.decl({ name: 'b-feeds-list', baseModel: 'vm-sync-dm' }, {

    // номер текущей страницы
    page: {
        type: 'number',
        default: 1
    },

    // кол-во фидов на странице
    rowsPerPage: {
        type: 'number',
        default: 20
    },

    // текст для поиска фида
    search: {
        type: 'string'
    },

    // параметры сортировки: поле
    sort: {
        type: 'string'
    },

    // параметры сортировки: направление
    reverse: {
        type: 'number',
        default: 0
    },

    // выбранные фиды
    selected: {
        type: 'array',
        // обновляем хэш состояния выбранных фидов
        preprocess: function(list) {
            var model = this.model,
                selected = model.get('selectedHash'),
                items = model.get('items');

            if (items) {
                items.forEach(function(item) {
                    var id = item.get('feedId');

                    // все текущие снимаем и выставляем только выбранные
                    selected[id] = !!~list.indexOf(id);
                });

                model.set('selectedHash', selected);
            }

            return this._preprocess(list);
        }
    },

    //У клиента есть права на редактирование/удаление фидов
    allowEditFeeds: {
        type: 'boolean',
        default: true
    },

    // флаг о блокировке редактирования
    lockEdit: {
        type: 'boolean',
        default: false
    },

    // флаг о блокировке
    lockState: {
        type: 'boolean',
        default: false
    },

    // флаг о том, что удаление не разрешено
    deleteDenied: {
        type: 'boolean',
        dependsFrom: ['selected', 'lockState', 'allowEditFeeds'],
        calculate: function(deps) {
            return !deps.allowEditFeeds || deps.lockState || !deps.selected.length;
        }
    },

    // список фидов
    items: {
        type: 'models-list',
        modelName: 'vm-feed'
    },

    // полное количество фидов
    totalCount: {
        type: 'number',
        default: 0
    },

    // хэш состояния выбранных фидов
    selectedHash: {
        internal: true,
        type: 'extended-object'
    },

    /**
     * декларация DM модели
     */
    _dmDecl: {
        type: 'string',
        internal: true,
        value: 'dm-feed-list'
    }

}, {

    /**
     * Синхронизируется с DM
     */
    syncFromDM: function() {
        var hash = this.get('selectedHash'),
            selected = [];

        this.__base.apply(this, arguments);

        this.get('items').forEach(function(item) {
            var id = item.get('feedId');

            hash[id] && selected.push(id);

            item.syncFromDM();
        });

        this.set('selected', selected);
    },

    /**
     * Подготавливает данные из DM
     * Дополнительно составляет список выбранных фидов из хэша состояния выбранных фидов
     * @param {Object} data
     * @return {Object}
     */
    prepareDataFromDM: function(data) {
        return {
            items: data.items.map(function(item) { return { feedId: item.feed_id }; }),
            totalCount: data.total_count
        };
    },

    /**
     * Просит DM вытянуть данные и синхронизируется с ней
     * @return {$.Deferred}
     */
    fetch: function() {
        return this
            .getDM()
            .fetch(this.get('page'), this.get('rowsPerPage'), this.get('sort'), this.get('reverse'), this.get('search'))
            .done(this.syncFromDM.bind(this))
            .promise();
    },

    /**
     * Удаляет выбранные фиды
     * @return {$.Deferred}
     */
    removeSelected: function() {
        return this
            .getDM()
            .remove(this.get('selected'))
            .done(function(data) {
                // ничего не было удалено
                if (!data.length) return;

                // чтобы до синхронизации актуализировать хэш состояния
                // сначала очищаем список выбранных
                this.clear('selected');

                // а уже потом синхронизируем
                this.syncFromDM();

                // обновляем номер страницы в тех случаях, когда данных стало меньше
                this.set(
                    'page',
                    Math.min(this.get('page'), Math.ceil(this.get('totalCount') / this.get('rowsPerPage')))
                );
            }.bind(this))
            .promise();
    }

});
