BEM.DOM.decl('b-form-map', {
    onSetMod: {

        mode: function(modName, modVal) {
            var manualPoint = this.params.pointManual,
                autoPoint = this.params.pointAuto,
                manualBounds = this.params.boundsManual,
                autoBounds = this.params.boundsAuto,
                point = this.params.point,
                bounds = this.params.bounds;

            // если выбран ручной режим и есть ручная точка, выбираем ручную
            // если авто или нет ручной точки (еще нет, например) - ставим автоточку
            // см. DIRECT-72859
            this.params.point = modVal === 'manual' && manualPoint ?
                manualPoint :
                autoPoint ?
                    autoPoint :
                    point;

            this.params.bounds = modVal === 'manual' && manualBounds ?
                manualBounds :
                autoBounds ?
                    autoBounds :
                    bounds;

            this.afterCurrentEvent(function() {
                this._onModeChange(modVal);
            });

        },

        position: {

            auto: function() {
                this.afterCurrentEvent(function() {
                    var params = this.params,
                        data = {
                            point: params.pointAuto || params.point,
                            bounds: params.boundsAuto || params.bounds
                        };
                    this._update(data);
                });

            }

        },

        status: {

            '*': function(modName, modVal) {

                !modVal && this.hasMod('mode', 'manual') && this._toggle();

                this
                    .elemInstance('link', 'action', 'toggle')
                    .setMod('visibility', modVal ? 'visible' : 'hidden');

            },

            failed: function() {

                this._clearParams();
                this._updateMap();

            }

        },

        disabled: function(modName, modVal) {

            this.findBlocksOn(this.elem('link'), 'link').forEach(function(link) {
                link.setMod('disabled', modVal);
            });

        }

    },

    /**
     * Реакция на установку модификатора mode - режима работы с картой:
     *  auto - автоматическое определение адреса
     *  manual - ручная установка точки на карте
     * @param {String} mode режим работы с картой
     * @private
     */
    _onModeChange: function(mode) {
        var isManual = mode === 'manual',
            paramsPoint = this.params.point,
            paramsBounds = this.params.bounds,
            point = isManual ?
                this._currentManualPoint || paramsPoint :
                paramsPoint,
            bounds = isManual ?
                this._currentManualBounds || paramsBounds :
                paramsBounds;

        this
            .elemInstance('map')
            .setMod('visibility', mode === 'manual' ? 'visible' : 'hidden');

        this
            .elemInstance('link', 'action', 'toggle')
            .update();

        point && this._updateMap(true, { point: point, bounds: bounds });
    },

    /**
     * Переключает режим карты
     * @private
     */
    _toggle: function() {

        var _this = this;

        this
            .elem('map')
            .slideToggle('fast', function() {
                _this.toggleMod('mode', 'manual', 'auto');
            });

    },

    updateBounds: function() {
        // обновляться должно только при ручном переносе
        if (this.getMod('position') === 'manual') {
            this._updateMessage('manual');
            this._currentManualBounds = this.elemInstance('map').params.bounds;
        }

        this._triggerChange();
    },

    /**
     * Обновляет блок при изменении карты (элемента map)
     * @private
     */
    _updateByMap: function() {
        var distance;

        this._updateMessage('manual');
        this.setMod('position', 'manual');
        this._currentManualPoint = this.elemInstance('map').params.point;

        if (this.params.point) this._setDistance();

        this._triggerChange();

    },

    /**
     * Рассчитывает и отображает расстояние между автоматически определенной и вручную установленной точкой
     * @private
     */
    _setDistance: function() {

        var pt1 = this.elemInstance('map').params.point || this.params.pointManual,
            pt2 = this.params.pointAuto || this.params.point,
            distance;

        if (!pt1 || !pt2) return;

        try {
            distance = ymaps.coordSystem.geo.getDistance(pt1.split(','), pt2.split(','));

            this.elem('distance').html(ymaps.formatter.distance(distance));
        } catch (e) {}

        return distance;
    },

    /**
     * Заглушка для метода установки нового сообщения
     * @param {*} key
     * @private
     */
    _updateMessage: function(key) {},

    /**
     * Устанавливает новый статус блока
     * @param {String} status новый статус
     * @param {String} key ключ отображаемого сообщения
     * @private
     */
    _updateStatus: function(status, key) {

        this
            .setMod('status', status)
            ._updateMessage(key);

    },

    /**
     * Геокодирует адрес
     * @param {String} [address] адрес
     */
    geocode: function(address) {
        address = address || this.params.address;

        if (!address) {
            this._updateStatus('failed', 'empty');
            return;
        }

        this.params.address = address;

        this._geocode(address);

    },

    /**
     * Отправляет запрос геокодеру
     * @param {String} address адрес
     */
    _geocode: function(address) {

        var _this = this;

        ymaps.ready(function() {
            ymaps
                .geocode(address, { results: 1 })
                .then(function(response) {
                    _this._onLoad(response);
                },
                    function(error) {
                        _this._updateStatus('failed', 'error');
                    });
        });

    },

    /**
     * Реакция на ответ геокодера
     * @param {Object} response ответ геокодера
     * @private
     */
    _onLoad: function(response) {

        var result = response && response.geoObjects.get(0);

        result ?
            this
                .setMod('status', 'success')
                ._onSuccess(result) :
            this._updateStatus('failed', 'empty');

    },

    /**
     * Реакция на успешное геокодирование
     * @param {Object} result - результат геокодирования
     * @private
     */
    _onSuccess: function(result) {

        var props = result.properties;

        this.params = u._.extend(this.params, {
            precision: props.get('metaDataProperty.GeocoderMetaData.precision'),
            //kind: props.get('metaDataProperty.GeocoderMetaData.kind'),
            point: result.geometry.getCoordinates().toString(),
            bounds: props.get('boundedBy').toString()
        });

        this._update();

    },

    /**
     * Обновляет блок
     * @param {Object} data данные карты
     * @private
     */
    _update: function(data) {

        this._updateMessage(this.params.precision);
        this._updateMap(data);

        this._triggerChange();

    },

    /**
     * Генерирует событие изменения параметров блока
     * @private
     */
    _triggerChange: function() {

        this.trigger('change', this._getVal());

    },

    /**
     * Обновляет карту
     * @param {Object} data данные карты
     * @private
     */
    _updateMap: function(data) {

        this.elemInstance('map').update(u._.extend({}, this.params, data));

    },

    /**
     * Устанавливает/возвращает параметры блока
     * @param {Object} [val] новые параметры
     * @returns {*}
     */
    val: function(val) {

        return typeof val === 'undefined' ?
            this._getVal() :
            this._setVal(val);

    },

    /**
     * Устанавливает новые параметры блока
     * @param {Object} val новые параметры
     * @param {[Number, Number]} val.point-auto автоматически определенные координаты точки
     * @param {[[Number, Number], [Number, Number]]} val.bounds-auto автоматически определенные координаты границ карты
     * @param {String} val.precision точность геокодирования
     * @param {[Number, Number]} val.point-manual установленные вручную координаты точки
     * @param {[[Number, Number], [Number, Number]]} val.bounds-manual установленные вручную координаты границ карты
     * @returns {*}
     * @private
     */
    _setVal: function(val) {

        if (!val) return this.clear(true);

        var auto = this.params,
            manual = this.elemInstance('map').params;

        auto.point = val['point-auto'];
        auto.bounds = val['bounds-auto'];
        auto.precision = val['precision'];

        manual.point = val['point-manual'];
        manual.bounds = val['bounds-manual'];

        if (manual.point) {
            this._updateMessage('manual');
            this.setMod('position', 'manual');
            this.elemInstance('map').update(manual);

            if (this.hasMod('mode', 'auto')) {
                this.elemInstance('link', 'action', 'toggle').update();
            }

            if (auto.point && auto.precision) {
                this
                    .setMod('status', 'success')
                    ._setDistance();
            }
        } else if (auto.point && auto.precision) {
            this.setMod('status', 'success');
            this.hasMod('position', 'auto') ?
                this._update() :
                this.setMod('position', 'auto');
        } else {
            this.setMod('position', 'auto');
            this._updateMessage('empty');
        }

        this._triggerChange();

    },

    /**
     * Возвращает параметры блока
     * @returns {Object}
     * @private
     */
    _getVal: function() {

        var params = this.params;

        return {
            'point-auto': params.point,
            'bounds-auto': params.bounds,
            precision: params.precision
        };

    },

    /**
     * Приводит параметры блока к исходному состоянию
     * @private
     */
    _clearParams: function() {

        this.params = $.extend(this.params, this.getDefaultParams());

    },

    /**
     * Переводит блок в исходное состояние
     */
    clear: function(force) {

        this.delMod('status')._clearParams();

    },

    /**
     * Возвращает параметры блока по умолчанию
     * @returns {{point: string, bounds: string, precision: string, address: string}}
     */
    getDefaultParams: function() {

        return {
            point: '',
            bounds: '',
            precision: '',
            address: ''
        };

    }

}, {

    live: true

});

/**
 * Точка на карте установлена вручную
 */
BEM.DOM.decl({

    block: 'b-form-map',
    modName: 'position',
    modVal: 'manual'

}, {

    onSetMod: {

        mode: {

            manual: function() {
                var distance;

                if (this.params.point) {
                    distance = this._setDistance();
                    this.toggleMod(this.elem('position'), 'hidden', '', 'yes', distance > 0);
                } else {
                    this.geocode();
                }

            }

        }

    },

    /**
     * Обновляет блок
     * @private
     */
    _update: function() {

        this.__base();
        this._setDistance();

    },

    /**
     * В режиме ручной установки точки карта не обновляется при геокодировании
     * @param {Boolean} doUpdate - флаг, что обновление действительно необходимо
     * @param {Object} data - ручные границы
     * @private
     */
    _updateMap: function(doUpdate, data) {
        if (doUpdate) {
            this.elemInstance('map').update(u._.extend({}, this.params, data));
        }
    },

    /**
     * Возвращает параметры блока
     * @returns {extend|*}
     * @private
     */
    _getVal: function() {

        var params = this.elemInstance('map').params;

        return $.extend(
            this.__base(),
            {
                'point-manual': this._currentManualPoint || params.point,
                'bounds-manual': this._currentManualBounds || params.bounds
            });

    },

    /**
     * Приводит блок к исходному состоянию
     * @param {Boolean} [force] перевести блок из ручного режима в автоматический
     */
    clear: function(force) {

        this.__base(force);
        force && this.setMod('position', 'auto');

    }

});

/**
 * Точка на карте определена автоматически
 */
BEM.DOM.decl({

    block: 'b-form-map',
    modName: 'position',
    modVal: 'auto'

}, {

    /**
     * Показывает сообщение о начале процесса геокодирования
     * @param {String} address адрес для геокодирования
     * @private
     */
    _geocode: function(address) {

        this._updateStatus('loading', 'loading');
        this.__base(address);

    },

    /**
     * Устанавливает новое сообщение о состоянии блока по ключу
     * @param {String} key ключ (именованный список передается в js-параметры блока)
     * @private
     */
    _updateMessage: function(key) {

        this.elemInstance('message').update(key);

    },

    /**
     * Обновляет блок при переводе его в исходное состояние
     * @param {*} force
     */
    clear: function(force) {

        this.__base(force);
        this._update();

    }

});
