/**
 * Элемент, реализующий поведение контрола выбора станции метро
 * Требует наличия элементов __input_type_select и __input_type_checkbox внутри себя
 */
BEM.DOM.decl({

    block: 'b-form-vcard',
    elem: 'metro'

}, {

    onSetMod: {
        js: function() {
            // при инициализации контрола убираем блокировку
            this.elemInstance('input', 'type', 'select').delMod('locked');
        }
    },

    /**
     * Заполняет контрол данными
     * @param {Object} data данные
     */
    fill: function(data) {
        var input = this.elemInstance('input', 'type', 'select'),
            autoDetect = typeof data.metro === 'undefined',
            disabled = data.metro == '0',
            notDefined = data.metro === null,
            // сохранение состояния для последующего восстановления
            locked = input.hasMod('locked', 'yes');

        // для автоопределения по адресу надо разлочить
        autoDetect && input.delMod('locked');

        // для выставления "не найдена", надо удалить закэшированное значение
        notDefined && (this._lastSelectedVal = '');

        // вкл/выкл чекбокса "не показывать на поиске"
        this
            .findBlockOn(this.elem('input', 'type', 'checkbox'), 'checkbox')
            .toggleMod('checked', 'yes', disabled);

        // не для автоопределения надо залочить, чтоб не срабатывал автопоиск
        // ранее залоченные надо залочить обратно
        (!autoDetect || locked) &&
            input.setMod('locked', 'yes');

        // если не был залочен, то разлочить
        !locked &&
            this.afterCurrentEvent(function() { input.delMod('locked'); });
    },

    /**
     * Реакция на событие окончания заполнения формы
     * @param {Event} e объект события
     * @param {Object} data данные события
     * @private
     */
    _onFormFill: function(e, data) {

        var cityId = data['city-id'];

        this._val = data['metro'];

        cityId && this._onCityUpdate({ geoId: cityId });

    },

    /**
     * Реакция на событие обновления формы
     * @param {Event} e объект события
     * @param {Object} data данные события
     * @private
     */
    _onFormUpdate: function(e, data) {

        if (data.name == 'city')
            if (data.geoId) {
                this._onCityUpdate(data);
            } else {
                this._toggle(false);
            }

    },

    /**
     * Реакция на событие изменения одного из контролов внутри элемента
     * Если станция метро выбрана из селекта вручную, то соответствующий контрол блокируется,
     *  что позволяет предотвратить перезаписывание выбранного значение при изменении адреса
     * @param {Event} e объект события
     * @param {Object} data данные события
     * @private
     */
    _onInputUpdate: function(e, data) {

        switch (data.type) {
            case 'checkbox':
                this._update(data);
                break;
            case 'select':
                e.block.toggleMod('locked', 'yes', '', !!e.block.val());
                break;
        }

    },

    /**
     * Реакция на событие обновления города
     * @param {Object} data данные
     * @private
     */
    _onCityUpdate: function(data) {

        this.setMod(this.elem('icon'), 'metro', this.params.iconSet[data.geoId]);

        this
            .getDataprovider()
            .get(data.geoId, this._updateOptions);

    },

    /**
     * Обновляет состояние активности селекта в соответствии с состоянием чекбокса
     * Когда селект становится активным, у него выставляется последнее выбранное значение
     * @param {Object} data данные
     * @private
     */
    _update: function(data) {

        var input = this.elemInstance('input', 'type', 'select'),
            checkbox = this.elemInstance('input', 'type', 'checkbox'),
            checked = data.checked;

        if (checked) {
            this._lastSelectedVal = input.val();
            input.val('');
        } else {
            input.val(this._lastSelectedVal || '');
        }

        this.toggleMod('disabled', 'yes', '', checked);

        input
            .toggleMod('locked', 'yes', '', checked)
            .getInput()
            .toggleMod('disabled', 'yes', '', checked);

        input.runUpdate({
            metroDisabled: checked
        });
    },

    /**
     * Обновляет список опций селекта
     * @param {Array} options новые опции
     * @private
     */
    _updateOptions: function(options) {

        var input = this.elemInstance('input', 'type', 'select'),
            select = input.getInput(),
            firstOption = select.elem('option').eq(0),
            emptyOption = {
                item: 'option',
                content: firstOption.text(),
                value: firstOption.attr('value')
            },
            item = this.closestElem('item');

        this._toggle(!!options.length);

        select.setOptions([emptyOption].concat(options));

        if (this._val) {
            select.val(this._val);
            input.delMod('locked');
            delete this._val;
        }

    },

    /**
     * Скрывает или показывает настройки метро
     * @param {Boolean} toggle true - показать, false - скрыть
     * @returns {Object} this
     * @private
     */
    _toggle: function(toggle) {
        this.afterCurrentEvent(function() {
            var item = this.closestElem('item');

            item[toggle ? 'slideDown' : 'slideUp']('fast', function() {
                this.toggleMod(item.removeAttr('style'), 'visibility', '', 'hidden', toggle);
            }.bind(this));
        });

        return this;
    },

    /**
     * Возвращает датапровайдер для получения списка станций метро для соответствующего города
     * @returns {BEM|*|_dataprovider}
     */
    getDataprovider: function() {

        var dataprovider = {
            path: '/',
            scriptName: 'metro.js'
        };

        return this._dataprovider ||
            (this._dataprovider = BEM.create(
                    dataprovider.name || 'b-form-vcard__dataprovider_name_metro',
                $.extend(dataprovider, { callbackCtx: this })));

    },

    /**
     * Реакция на событие очищения формы
     * Устанавливает пустой список опций селекта
     * @param {Event} e объект события
     * @param {Object} data данные события
     * @private
     */
    _onFormClear: function(e, data) {

        this._updateOptions([]);

    },

    destruct: function() {
        this.__base.apply(this, arguments);
    }

}, {

    live: function() {

        this
            .liveInitOnBlockInsideEvent('update', 'b-form-vcard__input', function(e, data) {
                this._onInputUpdate(e, data);
            })
            .liveInitOnParentEvent('update fill clear', function(e, data) {
                switch (e.type) {
                    case 'update':
                        this._onFormUpdate(e, data);
                        break;
                    case 'fill':
                        this._onFormFill(e, data);
                        break;
                    case 'clear':
                        this._onFormClear(e, data);
                        break;
                }
            });

    }

});

/**
 * Датапровайдер для получения списка станций метро
 */
BEM.decl({

    block: 'b-form-vcard__dataprovider_name_metro',
    baseBlock: 'i-request_type_script'

}, {

    /**
     * Возвращает список опций для селекта станций метро, соответствующий текущему id города
     * @param {Array} data данные, загруженные с сервера
     * @param {Number} geoId id города
     * @returns {Array}
     * @private
     */
    _filter: function(data, geoId) {

        if (geoId == 213) geoId = 1;

        var filtered = data.filter(function(item) {
                return item.city_id == geoId;
            }),
            result = filtered.length ? filtered[0].metro : [];

        return result.map(function(item) {
            return {
                item: 'option',
                content: item.name,
                value: item.region_id
            };
        });

    }

});
