/**
 * Элемент, реализующий поведение контрола для указания ОГРН организации
 * Требует наличия элементов __input_type_select и __input_type_input внутри себя
 */
BEM.DOM.decl({

    block: 'b-form-vcard',
    elem: 'ogrn'

}, {

    /**
     * Реакция на событие заполнения формы
     * @param {Event} e объект события
     * @param {Object} data данные события
     * @private
     */
    _onFormFill: function(e, data) {

        this._update(data);

    },

    _update: function(data) {},

    _onFormClear: function(e, data) {},

    _onInputUpdate: function(e, data) {}

}, {

    live: function() {

        this
            .liveInitOnParentEvent('fill clear', function(e, data) {
                switch (e.type) {
                    case 'fill':
                        this._onFormFill(e, data);
                        break;
                    case 'clear':
                        this._onFormClear(e, data);
                        break;
                }
            })
            .liveInitOnBlockInsideEvent('update', 'b-form-vcard__input', function(e, data) {
                this._onInputUpdate(e, data);
            });

    }

});

/**
 * Простой инпут для ввода нового ОГРН организации
 */
BEM.DOM.decl({

    block: 'b-form-vcard',
    elem: 'ogrn',
    modName: 'type',
    modVal: 'simple'

}, {

    /**
     * Обновляет контрол при заполнении формы
     * Если в данных указан существующий ОГРН, то инпут преобразуется к составному типу select+input (_type_complex)
     * @param {Object} data данные заполнения формы
     * @private
     */
    _update: function(data) {

        var list = this.params.list,
            id = data.org_details_id;

        if (list && id) {
            BEM.DOM.update(this.domElem, this._buildHtml(list, id));
            this.setMod('type', 'complex');
        } else if (data.ogrn) {
            this.elemInstance('input', 'type', 'input').val(data.ogrn);
        }

    },

    /**
     * Возвращает HTML для обновления контрола
     * @param {Array} list новый список значений селекта
     * @param {String} id выбранное значение селекта
     * @returns {HTML}
     * @private
     */
    _buildHtml: function(list, id) {

        var params = this.params,
            ogrn;

        return BEMHTML.apply({
            block: 'b-form-vcard',
            elem: 'layout',
            content: [
                {
                    elem: 'cell',
                    elemMods: { for: 'select' },
                    content: {
                        elem: 'input',
                        elemMods: { type: 'select', name: 'ogrn' },
                        id: params.controlId,
                        name: params.selectName,
                        value: id,
                        options: [params.firstOption].concat(
                            list.map(function(item) {
                                if (item.org_details_id == id) ogrn = item.ogrn;

                                return {
                                    value: item.org_details_id,
                                    content: item.ogrn
                                };
                            }))
                    }
                },
                { elem: 'gap' },
                {
                    elem: 'cell',
                    content: {
                        elem: 'input',
                        elemMods: {
                            type: 'input',
                            name: 'ogrn-new',
                            visibility: 'hidden'
                        },
                        value: ogrn
                    }
                }
            ]
        });

    }

});

/**
 * Составной инпут для указания существующего или ввода нового ОГРН организации
 */
BEM.DOM.decl({

    block: 'b-form-vcard',
    elem: 'ogrn',
    modName: 'type',
    modVal: 'complex'

}, {

    /**
     * Устанавливает значение ОГРН в селекте и скрывает текстовый инпут, если значение установлено успешно
     * @param {Object} data данные заполнения формы
     * @private
     */
    _update: function(data) {

        var select = this.elemInstance('input', 'type', 'select');

        select.val(data.org_details_id);

        // Если при копировании визитки у нас _type_complex
        // мы должны показать инпут с новым значением
        if (select.val()) {
            this._updateInput(false, select.getInput().getSelectedOptionText());
        } else if (data.ogrn) {
            this._updateInput(true, data.ogrn);
        }

    },

    /**
     * Реакция на событие изменения селекта
     * @param {Event} e объект события
     * @param {Object} data данные события
     * @private
     */
    _onInputUpdate: function(e, data) {

        if (data.type === 'select') {
            var index = data.index;
            this._updateInput(!index, index ? e.block.getInput().getSelectedOptionText() : '');
        }

    },

    /**
     * Обновляет состояние видимости и значение текстового инпута
     * @param {Boolean} show показать инпут
     * @param {String} val значение
     * @private
     */
    _updateInput: function(show, val) {

        var _this = this,
            input = this.elemInstance('input', 'type', 'input');

        if (show && input.hasMod('visibility', 'visible')) return;

        function callback() {
            input.setMod('visibility', show ? 'visible' : 'hidden');
            show || input.val(val);
            input.domElem.removeAttr('style');
            _this.getParent().validate();
        }

        show && input.val(val);

        input.domElem[show ? 'fadeIn' : 'fadeOut']('fast', callback);

    },

    /**
     * Реакция на событие очищения формы
     * @param {Event} e объект события
     * @param {Object} data данные события
     * @private
     */
    _onFormClear: function(e, data) {

        this._updateInput(true);

    }

});
