/**
 * @event b-gallery-file-chooser#select
 * @type {Object}
 * @property {Object} item объект файла
 * @property {String} item.name имя файла
 * @property {String} item.url ссылка на файл
 * @property {Number} item.width ширина изображения
 * @property {Number} item.height высота изображения
 *
 * @fires b-gallery-file-chooser#select событие выбора одного из файлов из галлереи
 * @fires b-gallery-file-chooser#reset событие возврата состояния галлереи к первоначальному
 */
BEM.DOM.decl({ block: 'b-gallery-file-chooser' }, {
    onSetMod: {
        js: function() {
            this._galleryItems = this.params.galleryItems || [];

            this._galleryItems.forEach(function(item, i) {
                !item.id && (item.id = i);
            });

            this._pages = [];

            this._itemsPerPage = this.params.itemsPerPage || 50;

            this._showGalleryPage();

            // навигация по страницам
            this.bindTo('pagination-button', 'click', function(e) {
                var button = $(e.currentTarget),
                    type = this.getMod(button, 'type');

                this._showGalleryPage(this._currentPage + (type == 'gallery-fwd' ? 1 : -1))
            });
        }
    },

    /**
    * Сбрасывает состояние галлереи к первоначальному
    */
    resetGallery: function() {
        this.delMod(this.findElem('list-item'), 'selected');
        this._showGalleryPage(0);

        this.trigger('reset');
    },

    destruct: function() {
        this.unbindFrom('pagination-button', 'click');
        this.liveUnbindFrom('list-item', 'click');

        return this.__base.apply(this, arguments);
    },

    /**
     * Переключает галеррею к определенной странице
     * @param {Number=0} pageNumber Номер страницы. Отсчет ведется с нуля.
     * @private
     */
    _showGalleryPage: function(pageNumber) {
        pageNumber = pageNumber || 0;

        this._currentPage = pageNumber;
        this._pages[pageNumber] = this._pages[pageNumber] || this._generatePageElem(pageNumber);

        this
            .setMod(this.findElem('page'), 'hidden', 'yes')
            .delMod(this._pages[pageNumber], 'hidden', 'yes');

        this._refreshPaginationControlsState();

        this.elem('list').scrollTop(0);
    },

    /**
     * Формирует определенную страницу галереи сохраненных изображений
     * @param {Number} pageNumber Номер страницы. Отсчет ведется с нуля.
     * @private
     */
    _generatePageElem: function(pageNumber) {
        var firstItemIndex = this._itemsPerPage * pageNumber;

        return BEM.DOM.append(
            this.elem('list'), BEMHTML.apply(
                {
                    block: 'b-gallery-file-chooser',
                    elem: 'page',
                    items: this._galleryItems.slice(firstItemIndex, firstItemIndex + this._itemsPerPage)
                }));
    },

    /**
     * Меняет состояние контролов пэйджинга галлереи при переходах между страницами
     * @private
     */
    _refreshPaginationControlsState: function() {
        var currentPage = this._currentPage,
            lastPage;

        if (this._galleryItems.length <= this._itemsPerPage) {
            return;
        }

        lastPage = Math.ceil(this._galleryItems.length / this._itemsPerPage);

        this.setMod(this.elem('pagination'), 'visible', 'yes');

        this.elem('pagination-text').text(iget2('b-gallery-file-chooser', 'stranica-s-iz-s', 'Страница {foo} из {bar}', {
            foo: currentPage + 1,
            bar: lastPage
        }));

        this.findBlockOn(this.elem('pagination-button', 'type', 'gallery-back'), 'button')
            .toggleMod('disabled', 'yes', currentPage === 0);

        this.findBlockOn(this.elem('pagination-button', 'type', 'gallery-fwd'), 'button')
            .toggleMod('disabled', 'yes', currentPage + 1 === lastPage);
    },

    /**
     * Обрабочик собятия клика на элемент галлереи
     * @param {JQuery} picListItem DOM node элемента галлереи
     * @private
     */
    _onListItemClick: function(picListItem) {
        this
            .delMod(this.findElem('list-item'), 'selected')
            .setMod(picListItem, 'selected', 'yes');

        this._selectedItem = u._.find(this._galleryItems, 'id', +picListItem[0].dataset.id);

        this.trigger('select', { item: this._selectedItem });
    }
}, {
    live: function() {
        // меняем выбранный элемент при клике
        this.liveBindTo('list-item', 'click', function(e) {
            this._onListItemClick(e.data.domElem);
        });
    }
});
