/**
 * Модель состояния попапа превью группы
 */
BEM.MODEL.decl({ model: 'b-group-preview2' }, {

    // Выбран архивный таб
    // декларация tabIsArchive должна находиться выше декларации tab, чтобы при изменении tabHash поле tabIsArchive вычислялось
    // раньше поля tab, поскольку один из обработчиков изменения tab использует значение tabIsArchive
    // alkaline@todo придумать что-то надежнее порядка вычисления зависимых полей
    tabIsArchive: {
        type: 'boolean',
        dependsFrom: ['tabHash'],
        calculate: function(currentTab) {
            return currentTab && currentTab.type === 'archive';
        },
        default: false
    },

    // Фильтр баннеров по виду либо принадлежности к архиву
    tab: {
        type: 'enum',
        enum: [
            '',
            'base',
            //для РМП, где не нужен переключатель десктоп/мобильные
            'search',
            'search-desktop',
            'search-mobile',
            'search-mobile-target',
            'search-desktop-target',
            'context',
            '0x0',
            '240x400',
            '240x600',
            '300x250',
            '728x90',
            '160x600',
            '300x300',
            '300x600',
            '300x500',
            '970x250',
            '1000x120',
            'archive'
        ],
        dependsFrom: ['tabHash'],
        calculate: function() {
            var tabHash = this.get('tabHash');

            return tabHash ?
                [tabHash.type, tabHash.device, tabHash.target ? 'target' : false].filter(Boolean).join('-') :
                '';
        }
    },

    //Выбранный таб в виде BannerPreviewFilterState
    tabHash: {
        type: 'object',
        preprocess: function(value) {
            var firstTab = (this.model.get('tabsList') || [])[0],
                firstTabValue = firstTab ? (typeof firstTab == 'string' ? { type: firstTab } : firstTab) : null;

            return this._preprocess.call(this, value || firstTabValue);
        }
    },

    // Там по умолчанию (рассчитывается как первый элемент списка табов)
    defaultTab: {
        type: 'string',
        dependsFrom: ['tabsList'],
        calculate: function(tabsList) {
            return tabsList.length ? tabsList[0] : '';
        }
    },

    // Статус занятости, если больше 0 то занят, не может быть отрицательным
    busyStatus: {
        type: 'number',
        default: 0
    },

    /**
     * Массив объектов данных, необходимых для отрисовки одного баннера в попапе групп
     */
    shownBannersData: {
        type: 'array',
        default: []
    },

    // Флаг говорящий о том, что попап без возможности редактирования
    readonly: 'boolean',

    // Поле оповещающее об ошибке, содержит timestamp ошибки
    errorAlert: 'number',

    // Список доступных видов превью (не для таба 'Архив')
    tabsList: {
        type: 'array',
        default: []
    },

    // Флаг говорящий о том, что доступна кнопка всех форматов баннера
    isPartnerAllFormatsEnabled: 'boolean',

    // Количество не архивных баннеров в группе
    countNoArchiveBanners: 'number',

    // Количество архивных баннеров в группе
    countArchiveBanners: 'number',

    // Флаг говорящий о том, что статус хотя бы одного баннера был изменен
    hasChangedStatuses: 'boolean',

    // Поле отвечающее за показ только одного баннера из группы
    showOneBid: 'string',

    // bid баннера, при изменении которого (включение/выключение/архивирование/разархивирование/удаление),
    // нужен редирект, для обновления этого баннера на странице вне попапа (используется на стр. кампании)
    generalBid: 'string',

    // Идентификатор кампании
    cid: 'string',

    // Идентификатор группы
    adgroupId: 'string',

    // Тип группы
    adgroupType: 'string',

    //Флаг "Остановлено мониторингом сайта"
    statusMetricaStop: {
        type: 'boolean',
        default: false
    },

    // Не очень красивый флаг наличия кнопок "Редактировать" и "Редактировать группу" в попапе групп
    editable: {
        type: 'boolean',
        default: false
    },

    zero_banners_quantity: {
        type: 'boolean',
        default: false
    },

    campaignMediaType: 'string'

}, {

    /**
     * Инициализация view-модели
     * @param {String|Number} adgroupId номер группы
     * @param {String} adgroupType тип группы
     */
    init: function(adgroupId, adgroupType) {
        this._incrBusyStatus();

        this
            .on('tab', 'change', this._onTabChange, this)
            .on('destruct', this._onDestruct, this);

        var groupModel = this._groupModel = BEM.MODEL.getOrCreate({
                name: u.campaign.getGroupModelName(this.get('campaignMediaType')),
                id: adgroupId
            }),
            //@heliarian 0 - это хорошее валидное значение adgroup_id для свежесозданной группы
            groupModelIsEmpty = !groupModel.get('adgroup_type') ||
                (!groupModel.get('adgroup_id') && groupModel.get('adgroup_id') !== 0),
            requestArchiveBanners = !groupModelIsEmpty && groupModel.get('banners_quantity') &&
                groupModel.get('banners_quantity') === groupModel.get('banners_arch_quantity');

        groupModelIsEmpty && groupModel.update({
            adgroup_id: adgroupId,
            adgroup_type: adgroupType
        });

        groupModel.requestGroupDataAndUpdate({ isArchive: requestArchiveBanners, onlyBanners: !groupModelIsEmpty })
            .then(this._fillFieldsAndBind.bind(this))
            .then(this._updateTabAndTabsList.bind(this))
            .then(this._updateShownBanners.bind(this))
            .then(this._decrBusyStatus.bind(this))
            .fail(this._onRequestFail.bind(this));
    },

    /**
     * Обработчик изменения таба
     * @private
     */
    _onTabChange: function() {
        // т.к. можем менять таб внутри методов модели
        if (!this.get('busyStatus')) {
            this._incrBusyStatus();

            // Если переключились с архива - то нужно получить список табов
            this[this.isChanged('tabIsArchive') ? '_updateTabAndTabsList' : '_fakePromise']()
                .then(this._updateShownBanners.bind(this))
                .then(this._decrBusyStatus.bind(this))
                .fail(this._onRequestFail.bind(this));
        }
    },

    /**
     * Заполняет поля view-модели из data-модели и подписывается на изменения некоторых из них
     * @param {BEM.MODEL} groupModel
     * @returns {boolean}
     * @private
     */
    _fillFieldsAndBind: function(groupModel) {
        var countNoArchiveBanners = groupModel.get('banners_quantity') - groupModel.get('banners_arch_quantity'),
            campaignModel = this.getCampaignModel();

        this.update({
            cid: campaignModel.get('cid'),
            adgroupId: groupModel.get('adgroup_id'),
            statusMetricaStop: groupModel.get('statusMetricaStop') === 'Yes',
            adgroupType: groupModel.get('adgroup_type'),
            countNoArchiveBanners: countNoArchiveBanners,
            countArchiveBanners: groupModel.get('banners_arch_quantity'),
            editable: campaignModel.get('editable'),
            zero_banners_quantity: !groupModel.get('banners_quantity')
        });

        this.get('readonly') || this.set('readonly', campaignModel.get('archived') === 'Yes');

        groupModel.on('banners_quantity banners_arch_quantity', 'change', this._onGroupDMBannersQuantityChange, this);

        return true;
    },

    /**
     * Обработчик на изменения полей `banners_quantity` и `banners_arch_quantity` в data-модели группы
     * @param {jQuery.Event} e
     * @private
     */
    _onGroupDMBannersQuantityChange: function(e) {
        var groupModel = e.target.model;

        this.update({
            countNoArchiveBanners: groupModel.get('banners_quantity') - groupModel.get('banners_arch_quantity'),
            countArchiveBanners: groupModel.get('banners_arch_quantity'),
            zero_banners_quantity: !groupModel.get('banners_quantity')
        });
    },

    /**
     * Обработчик удаления view-модели
     * @private
     */
    _onDestruct: function() {
        this
            .un('tab', 'change', this._onTabChange, this)
            .un('destruct', this._onDestruct, this);

        this.getGroupModel()
            .un('banners_quantity banners_arch_quantity', 'change', this._onGroupDMBannersQuantityChange, this);
    },

    /**
     * Повышает busyStatus
     * @private
     */
    _incrBusyStatus: function() {
        // так-как метод будет вызываться асинхронно, защищаем от race condition
        this._withMutex(function(unLock) {
            this.set('busyStatus', this.get('busyStatus') + 1);
            unLock();
        });

        return this;
    },

    /**
     * Понижает busyStatus
     * @private
     */
    _decrBusyStatus: function() {
        // так-как метод будет вызываться асинхронно, защищаем от race condition
        this._withMutex(function(unLock) {
            this.get('busyStatus') > 0 && this.set('busyStatus', this.get('busyStatus') - 1);
            unLock();
        });

        return this;
    },

    /**
    * Обработчик в который оборачивается исполняемая функция в методе _withMutex
     *
    * @callback withMutexCallback
    * @param {Function} unLock - Обработчик разблокирующий мьютекс
    */

    /**
     * Оборачивает функцию в мьютекс и исполняет ее, тем самым защищая ресурс от состояния гонки
     *
     * @param {withMutexCallback} fn - Обертка над функцией, которую надо исполнить, принимает unLock обработчик
     * @param {number} [loopGuard] - Счетчик предотвращающая зацикливание
     * @private
     *
     * @example
     *     ```
     *     this._withMutex(function(unLock) {
     *         this.get('busyStatus') > 0 && this.set('busyStatus', this.get('busyStatus') - 1);
     *         unLock();
     *     });
     *     ```
     */
    _withMutex: function(fn, loopGuard) {
        var unLock = function() {
            this._mutex = false;
        }.bind(this);

        if (this._mutex) {
            if (loopGuard > 5) {
                throw new Error('Maximum call stack size exceeded');
            }

            return setTimeout(function() {
                this._withMutex(function() {
                    fn.call(this, unLock);
                }, (loopGuard || 0) + 1);
            }.bind(this), 10);
        }

        this._mutex = true;
        fn.call(this, unLock);
    },

    /**
     * Запрашивает модели баннеров у data-модели группы, вызывает фильтрацию по текущему табу и обновляет
     * поле `shownBanners` во view-модели
     * Если отображаемые баннеры не поменялись - триггерит на view-модели событие `changeViewBanners`
     * @returns {BEM.MODEL}
     */
    _updateShownBanners: function() {
        return this.getBannersFilteredByCurrentTab()
            .then(function(shownBannersModels) {
                this
                    .set('shownBannersData', shownBannersModels.map(this._getBannerData, this))
                    .fix(); // фиксируем модель, чтобы определить переключение с архива на не архив
            }.bind(this));
    },

    /**
     * Обновляет список доступных табов в зависимости от отображаемых баннеров
     * @param {BEM.MODEL[]} banners
     * @returns {BEM.MODEL}
     * @private
     */
    _getTabsList: function(banners) {
        var campaignModel = this.getCampaignModel();

        return u['b-banner-preview-filter2'].getTabsList({
            strategySearchName: campaignModel.get('strategy').search.name,
            dontShowYacontext: campaignModel.get('dontShowYacontext'),
            campMediaType: this.get('adgroupType'),
            banners: banners.map(function(banner) {
                return u._.pick(banner.provideData(), ['image', 'creative', 'banner_type', 'sitelinks', 'ad_type']);
            })
        });
    },

    /**
     * Обновляет список доступных табов в зависимости от отображаемых баннеров
     * @param {BEM.MODEL[]} banners
     * @returns {BEM.MODEL}
     * @private
     */
    _isPartnerAllFormatsEnabled: function(banners) {

        return u['b-banner-preview-filter2'].isPartnerAllFormatsEnabled({
            campMediaType: this.get('adgroupType'),
            banners: banners.map(function(banner) {
                return u._.pick(banner.provideData(), ['image', 'creative', 'banner_type', 'sitelinks', 'ad_type']);
            })
        });
    },

    /**
     * Возвращает модель группы с идентичным id
     * @returns {BEM.MODEL}
     */
    getGroupModel: function() {
        return this._groupModel;
    },

    /**
     * Возвращает модель кампании.
     * @returns {BEM.MODEL}
     */
    getCampaignModel: function() {
        return this._campaignModel || (this._campaignModel = this.getGroupModel().getCampaignModel());
    },

    /***
     * Запоминает изменение статуса(вкл./выкл.) баннера.
     * @param {Boolean} isShown - Показывать/ не показывать
     * @param {String} bid - Id баннера
     */
    commitShownStatus: function(isShown, bid) {
        // если в хранилище нету записи или null, то делаем запись и запоминаем предыдущее значение
        // если в хранилище есть запись и предыдущее значение совпадает с новым, то удаляем запись
        var showStatusesBank = this.getShownStatusBank();

        if (!showStatusesBank[bid]) {
            showStatusesBank[bid] = { prev: !isShown, newValue: isShown }
        } else {
            if (showStatusesBank[bid].prev === isShown) showStatusesBank[bid] = null;
        }

        this.set('hasChangedStatuses', Object.keys(showStatusesBank).some(function(key) {
            return !!showStatusesBank[key];
        }));
    },

    /**
     * Возвращает хранилище изменений статусов для баннеров.
     * @returns {{}}
     */
    getShownStatusBank: function() {
        return this._showStatusesBank || (this._showStatusesBank = {});
    },

    /**
     * Применяет сохраненные изменения статусов для баннеров.
     * @param {Function} done - обработчик при успешном завершении.
     */
    applyChangedShownStatuses: function(done) {
        this._incrBusyStatus();

        var bank = this.getShownStatusBank(),
            groupModel = this.getGroupModel(),
            statuses = Object.keys(bank).reduce(function(target, bid) {
                var item = bank[bid];

                return target.concat(item ? { bid: bid, value: item.newValue } : []);
            }, []),
            statusGeneralBannerIsChanged = statuses
                .some(function(statuse) { return this.isGeneralBanner(statuse.bid) }, this);

        groupModel.bulkSetBannerStatusById(statuses)
            .then(function() {
                if (statusGeneralBannerIsChanged) {
                    window.location.reload();
                } else {
                    done();
                    this._decrBusyStatus();
                }
            }.bind(this))
            .fail(this._onRequestFail.bind(this));
    },

    /**
     * Обработчики не удачного запроса на сервер.
     * @param {Error} err
     * @returns {$.Deferred}
     * @private
     */
    _onRequestFail: function(err) {
        this
            ._decrBusyStatus()
            // устанавливаем timestamp что-бы выстрелило событие change
            .set('errorAlert', +new Date());

        return $.Deferred().rejectWith(err);
    },

    /**
     * Объект для кеша моделей баннеров
     */
    _cacheBanners: null,

    /**
     * Возвращаем закэшированный набор отфильтрованных по табу баннеров
     * @param {String} tabName
     * @returns {BEM.MODEL[]}
     * @private
     */
    _getBannersFilteredByTabNameAndCache: function(tabName) {
        this._cacheBanners || (this._cacheBanners = {});

        // заполняем таб баннерами
        return this._cacheBanners[tabName] ||
            (this._cacheBanners[tabName] = this._getBannersFilteredByTabName(tabName));
    },

    /**
     * Возвращаем набор отфильтрованных по табу баннеров
     * @param {String} tabName
     * @returns {BEM.MODEL[]}
     * @private
     */
    _getBannersFilteredByTabName: function(tabName) {
        var banners = this.getGroupModel().getBannersWhere(
                this.get('showOneBid') ?
                    { archive: tabName === 'archive' ? 'Yes' : 'No', bid: this.get('showOneBid') } :
                    { archive: tabName === 'archive' ? 'Yes' : 'No' }),
            filterbyCreativeSizeTag = function(sizeTag) {
                return function(banner) { return banner.get('creative').get('sizeTag') == sizeTag; };
            },
            filterByType = function(type) {
                return function(banner) { return banner.get('banner_type') === type };
            },
            filterByHasImage = function(banner) {
                return banner.get('image_model').get('image');
            },
            withoutImageAd = function(banner) {
                return banner.get('ad_type') !== 'image_ad';
            },
            withoutCpcVideo = function(banner) {
                return banner.get('ad_type') !== 'cpc_video';
            };

        switch (tabName) {
            case 'base':
            case 'archive':
                return banners;

            case 'search':
                return banners
                    .filter(withoutImageAd)
                    .filter(withoutCpcVideo);

            case 'search-desktop':
            case 'search-desktop-target':
                return banners
                    .filter(filterByType('desktop'))
                    .concat(banners.filter(filterByType('mobile')))
                    .filter(withoutImageAd)
                    .filter(withoutCpcVideo);

            case 'search-mobile':
            case 'search-mobile-target':
                return banners
                    .filter(filterByType('mobile'))
                    .concat(banners.filter(filterByType('desktop')))
                    .filter(withoutImageAd)
                    .filter(withoutCpcVideo);

            case 'context':
                return banners;

            case '0x0':
            case '240x400':
            case '240x600':
            case '300x250':
            case '728x90':
            case '300x600':
            case '300x500':
            case '970x250':
            case '160x600':
            case '300x300':
            case '1000x120':
                return banners.filter(filterbyCreativeSizeTag(tabName));

            // нужно для получения всех не архивных баннеров
            default:
                return banners;
        }
    },

    /**
     * Возвращает отфильтрованные баннеры по текущему табу и кэширует
     * @returns {$.Deferred}
     */
    getBannersFilteredByCurrentTab: function() {
        return this.getBannersFilteredByTab(this.get('tab'));
    },

    /**
     * Возвращает отфильтрованные баннеры по указанному табу и кэширует
     * @param {String} tabName
     * @returns {$.Deferred}
     */
    getBannersFilteredByTab: function(tabName) {
        return this.getGroupModel()
            .requestGroupDataAndUpdate({ isArchive: tabName === 'archive', onlyBanners: true })
            .then(function() {
                return this._getBannersFilteredByTabNameAndCache(tabName);
            }.bind(this));
    },

    /**
     * Формирует данные по баннеру для отрисовки в попапе групп
     * @param {BEM.MODEL} bannerModel
     * @returns {Object}
     * @private
     */
    _getBannerData: function(bannerModel) {
        var adgroupType = this.get('adgroupType'),
            mediaType = this.get('campaignMediaType'),
            groupModel = this.getGroupModel(),
            tabIsArchive = this.get('tabIsArchive'),
            shownStatusBank = this.getShownStatusBank(),
            isNewBanner = bannerModel.get('isNewBanner'),
            adType = bannerModel.get('ad_type'),
            bid = '' + bannerModel.get('bid'),
            dmsIds = {
                bannerId: bannerModel.id,
                groupId: groupModel.id
            },
            // создаем view-модель превью баннера, чтобы получить входные данные для bemhtml блока
            bannerViewModel = BEM.MODEL.create(
                u['b-banner-preview2'].getNameByType(mediaType, this._getPreviewView(adType))
            ),
            randPhrase,
            bannerData;

        if (u._.contains(['text', 'mobile_content', 'cpm_banner', 'cpm_deals'], adgroupType)) {
            randPhrase = groupModel.getRandomActivePhraseModel();

            if (randPhrase) {
                dmsIds.phraseId = randPhrase.id;
            }
        }

        bannerViewModel.init(dmsIds, adgroupType);
        bannerData = bannerViewModel.toJSON();

        // РМП-изображения всегда широкоформатные
        // https://st.yandex-team.ru/DIRECT-72852
        if (adgroupType === 'mobile_content' && bannerData.image !== '') {
            bannerData.image_type = 'wide'
        }

        // для превью "баннера в метро" нужно знать группу
        if (groupModel.get('cpmGroupType') === 'cpm_geoproduct') {
            bannerData.cpmBannersType = 'cpm_geoproduct'
        }

        return {
            bannerId: bannerModel.get('BannerID'),
            bid: bid,
            //heliarian флаг isMobile нужен для показа строки "мобильное" в заголовке группы
            //для мобильного контента эта строка не нужна
            isMobile: adgroupType !== 'mobile_content' && bannerModel.get('banner_type') === 'mobile',
            isNewBanner: isNewBanner,
            isShown: shownStatusBank[bid] ?
                shownStatusBank[bid].newValue :
                bannerModel.get('statusShow') === 'Yes',
            previewCtx: {
                mods: { type: u.beminize(adgroupType), view: this._getPreviewView(adType) },
                data: bannerData,
                // при инициализации новая модель не создастся, а будет использоваться ранее созданная
                modelsParams: {
                    vmParams: { name: bannerViewModel.name, path: bannerViewModel.path() },
                    dmsIds: dmsIds
                }
            },
            statusBsSynced: bannerModel.get('statusBsSynced'),
            statusCtx: {
                banner: u._.extend(bannerModel.toJSON(), {
                    // передаём и internal поле showAdminAttrs для показа супер пользователям дополнительной информации
                    showAdminAttrs: bannerModel.get('showAdminAttrs'),
                    // DIRECT-76837 блок b-banner-status завязан на поле adgroup_type
                    adgroup_type: adgroupType
                }),
                js: { modelParams: { name: bannerModel.name, path: bannerModel.path() } },
                // DIRECT-41778: для динамический баннеров не нужно отображать статус "Идут показы предыдущей версии"
                hideOngoingShowsStatus: adgroupType === 'dynamic' || adgroupType === 'performance'
            },
            statusModerate: bannerModel.get('statusModerate'),
            can: {
                delete: bannerModel.get('can_delete_banner'),
                archive: !tabIsArchive && bannerModel.get('can_archive_banner'),
                unArchive: tabIsArchive,
                // условия активности тумблера описаны в задаче DIRECT-43274
                changeShownStatus: !isNewBanner && bannerModel.get('archive') === 'No' &&
                    // менять статус показа смарт-баннеров можно при всех статусах кроме "Нового"
                    (bannerModel.get('BannerID') != '0' || adgroupType == 'performance' ?
                        bannerModel.get('statusModerate') != 'New' :
                        bannerModel.get('statusModerate') == 'Yes'),
                edit: bannerModel.get('archive') === 'No' && this.get('editable')
            }
        };
    },

    /**
     * Возвращает значение модификатора `view` и `b-banner-preview2`
     * @returns {String}
     * @private
     */
    _getPreviewView: function(adType) {
        if (u._.contains(['image_ad', 'mcbanner'], adType)) {
            return 'base-image';
        }
        if (u._.contains(['cpm_banner', 'cpm_deals', 'cpc_video', 'cpm_yndx_frontpage'], adType)) {
            return 'creative';
        }

        switch (this.get('adgroupType')) {
            case 'performance':
                return 'performance-base';
            case 'text':
            case 'dynamic':
                return ({
                    base: 'base',
                    'search-desktop': 'search-desktop',
                    'search-mobile': 'search-mobile',
                    'search-desktop-target': 'search-desktop-target',
                    'search-mobile-target': 'search-mobile-target',
                    context: 'partner',
                    archive: 'base'
                })[this.get('tab')] || '';
            case 'mobile_content':
                return ({
                    base: 'mobile-content-campaign',
                    context: 'mobile-content-partner-poster',
                    archive: 'mobile-content-search',
                    search: 'mobile-content-search'
                })[this.get('tab')] || '';
            case 'content_promotion':
                return 'content-promotion-group';
        }
    },

    /**
     * Архивирует баннер по его bid с редиректом
     * @param {String} bid
     */
    archiveBannerByIdWithRedirect: function(bid) {
        this._incrBusyStatus();

        this.getGroupModel().archiveBannerByIdWithRedirect(bid);
    },

    /**
     * Архивирует баннера по его bid
     * @param {String} bid
     */
    archiveBannerById: function(bid) {
        this._incrBusyStatus();

        this.getGroupModel().archiveBannerById(bid)
            .then(this._dropTabsListAndCacheBanners.bind(this))
            .then(this._updateTabAndTabsList.bind(this))
            .then(this._updateShownBanners.bind(this))
            .then(this._decrBusyStatus.bind(this))
            .fail(this._onRequestFail.bind(this));
    },

    /**
     * Разархивирует баннер по его bid c редиректом
     * @param {String} bid
     */
    unArchiveBannerByIdWithRedirect: function(bid) {
        this._incrBusyStatus();

        this.getGroupModel().unArchiveBannerByIdWithRedirect(bid);
    },

    /**
     * Разархивирует баннер по его bid
     * @param {String} bid
     * @returns {$.Deferred}
     */
    unArchiveBannerById: function(bid) {
        this._incrBusyStatus();

        return this.getGroupModel().unArchiveBannerById(bid)
            .then(this._dropTabsListAndCacheBanners.bind(this))
            .then(this._updateTabAndTabsList.bind(this))
            .then(this._updateShownBanners.bind(this))
            .then(this._decrBusyStatus.bind(this))
            .fail(this._onRequestFail.bind(this));
    },

    /**
     * Разархивирует все баннеры в группе
     * @returns {*}
     */
    unArchiveAll: function() {
        this._incrBusyStatus();

        var groupModel = this.getGroupModel(),
            archiveBanners = groupModel.getBannersWhere({ archive: 'Yes' }),
            needRedirect = archiveBanners.some(function(mBanner) {
                return this.isGeneralBanner(mBanner.get('bid'));
            }, this);

        if (needRedirect) {
            return groupModel.unArchiveAllBannersWithRedirect();
        } else {
            groupModel.unArchiveAllBanners()
                .then(this._dropTabsListAndCacheBanners.bind(this))
                .then(this._updateTabAndTabsList.bind(this))
                .then(this._updateShownBanners.bind(this))
                .then(this._decrBusyStatus.bind(this))
                .fail(this._onRequestFail.bind(this));
        }
    },

    /**
     * Удаляет баннер по его bid с редиректом
     * @param {String} bid
     */
    deleteBannerByIdWithRedirect: function(bid) {
        this._incrBusyStatus();

        this.getGroupModel().deleteBannerByIdWithRedirect(bid);
    },

    /**
     * Удаляет баннер по его bid
     * @param {String} bid
     */
    deleteBannerById: function(bid) {
        this._incrBusyStatus();

        this.getGroupModel().deleteBannerById(bid)
            .then(this._dropTabsListAndCacheBanners.bind(this))
            .then(this._updateTabAndTabsList.bind(this))
            .then(this._updateShownBanners.bind(this))
            .then(this._decrBusyStatus.bind(this))
            .fail(this._onRequestFail.bind(this));
    },

    /**
     * Вызывает методы установки списка табов и проверки корректности выставленного таба
     * При первом открытии попапа (поле tab пустое), когда все баннеры архивные,
     * либо мы во вкладке "Архив" на странице кампании - устанавливает архивный таб в попапе групп
     * @returns {$.Deferred}
     * @private
     */
    _updateTabAndTabsList: function() {
        // открытие попапа групп
        if (!this.get('tab')) {
            (!this.get('countNoArchiveBanners') || this.getCampaignModel().isArchiveTab()) &&
                this.set('tabHash', { type: 'archive' });
        }

        return this._setTabsList().then(this._checkBannersForAbsent.bind(this));
    },

    /**
     * На основе не архивных баннеров устанавливает список табов и устройство по умолчанию
     * Если tabsList уже посчитан или выбран архивный таб - то ничего не делает
     * @returns {$.Deferred}
     * @private
     */
    _setTabsList: function() {
        var groupModel = this.getGroupModel(),
            requiredTabs = !this.get('tabsList').length;

        // если список табов не задан, то при выбранном архивном табе особые условия
        if (requiredTabs && this.get('tabIsArchive')) {
            // запрашиваем список табов при выбранном архивном в тех случаях,
            // когда кроме архивного есть и другие баннеры
            requiredTabs = !!groupModel.get('banners_quantity') &&
                groupModel.get('banners_quantity') > groupModel.get('banners_arch_quantity');
        }

        return !requiredTabs ?
            this._fakePromise() :
            // выбирает все не архивные баннеры
            this.getBannersFilteredByTab('')
                .then(function(banners) {
                    var currentTab = this.get('tab'),
                        tabsList = this._getTabsList(banners);

                    // Обновляем набор табов
                    this.set('tabsList', tabsList);
                    // Обновляем доступность кнопки всех форматов баннера
                    this.set('isPartnerAllFormatsEnabled', this._isPartnerAllFormatsEnabled(banners));

                    // Если текущий таб не указан или его нет в списке доступных
                    if (!currentTab || !this.get('tabIsArchive') && !u._.includes(tabsList, currentTab)) {
                        // вызываем preprocess у поля tab, который установит таб по умолчанию, для текущего списка табов
                        this.set('tabHash', null);
                    }
                }.bind(this));
    },

    /**
     * Сбрасывает кэш баннеров и список табов
     * Вызывается при изменяющем действии: архивирование, удаление
     * @returns {*}
     * @private
     */
    _dropTabsListAndCacheBanners: function() {
        this._cacheBanners = null;

        return this.set('tabsList', []);
    },

    /**
     * Если баннеров к отображению нет - переключает таб с архивного на не архивный и наоборот
     * @returns {$.Deferred|Boolean}
     * @private
     */
    _checkBannersForAbsent: function() {
        return this.getBannersFilteredByCurrentTab()
            .then(function(banners) {
                // Если нет баннеров для отображения - значит мы на неправильном табе
                return banners.length ?
                    true :
                    this
                        .set('tabHash', this.get('tabIsArchive') ? null : { type: 'archive' })
                        ._updateTabAndTabsList();
            }.bind(this));
    },

    /**
     * Возвращает флаг, нужен ли редирект на стр. кампании при
     * включении/выключении/архивировании/разархивировании/удалении переданного bid баннера
     * @param {String} bid
     * @returns {boolean}
     */
    isGeneralBanner: function(bid) {
        return bid == this.get('generalBid');
    },

    /**
     * Заглушка c $.Deferred объектом
     * @returns {$.Deferred}
     * @private
     */
    _fakePromise: function() {
        return $.Deferred().resolve().promise();
    }

});
