/**
 * Данные о введенных пользователем тегах
 * @typedef {Object} EditTagsData
 * @property {String[]} tags
 * @property {String[]} newTags
 */

/**
 * Событие запроса на сохранение тегов
 * @event b-group-tags2-edit#save
 * @type {EditTagsData}
 */
/**
 * Событие изменения выбранного набора тегов
 * @event b-group-tags2-edit#tagsChanged
 * @type {EditTagsData}
 */

/**
 * @fires b-group-tags2-edit#cancel
 * @fires b-group-tags2-edit#save
 * @fires b-group-tags2-edit#tagsChanged
 */
BEM.DOM.decl({ block: 'b-group-tags2-edit', baseBlock: 'i-glue' }, {
    onSetMod: {
        js: function() {
            this.__base.apply(this, arguments);

            u.graspSelf.call(this, {
                _newTagsInput: 'input inside new-tags',
                _errors: 'errors',
                _checkboxContainer: 'checkbox-container',
                _selectedTagsList: 'i-selectable inside',
                _save: 'button on save',
                _cancel: 'button on cancel'
            });

            this._doBindings();
            this._renderCheckboxes();
        },
        state: {
            '': function() {
                this._save.delMod('disabled');
            },
            saveDisabled: function() {
                this._save.setMod('disabled', 'yes');
            }
        }
    },

    /**
     * Возвращает выбранные теги
     * @returns {Object}
     */
    getValue: function() {
        return {
            tags: this.model.get('tags').slice(),
            newTags: this.model.getNewTags()
        };
    },

    /**
     * Устанавливает сообщение об ошибке.
     * @param {String} message
     */
    setError: function(message) {
        this._errors.html(message);
        this.setMod('state', message.length ? 'saveDisabled' : '');
    },

    /**
     * Подписывается на события элементов
     */
    _doBindings: function() {
        this._selectedTagsList.on('selectedChanged', function(e, params) {
            var selectedTags = u._.extend({}, this.model.get('selectedTags'));

            selectedTags[params.data.key] = params.data.isSelected;

            this.model.set('selectedTags', selectedTags);
        }, this);

        this._newTagsInput.on('change', function() {
            this.model.set('newTags', this._extractTagsFromInput());
        }, this);

        this.bindTo(this._newTagsInput.elem('control'), 'keypress', function(e) {
            if (e.which === 13) {
                this._triggerSave();
            }
        });

        this._save.on('click', function() {
            this._triggerSave();
        }, this);

        this._cancel.on('click', function() {
            this.trigger('cancel');
        }, this);

        this.model
            .on('tags', 'change', function() {
                this.trigger('tagsChanged', this.getValue());
            }, this);
    },

    /**
     * Триггерит событие сохранения с выбранными тегами
     */
    _triggerSave: function() {
        this.trigger('save', this.getValue());
    },

    /**
     * Возвращает массив тегов, введенных в инпут новых тегов
     * @returns {String[]}
     */
    _extractTagsFromInput: function() {
        return this._newTagsInput.val().replace(/\s*,\s*/g, ',') // удаляем пробелы вокруг запятых
            .replace(/,(?=,)/g, '') // оставляем только одну запятую подряд
            .replace(/^\s*/, '') // удаляем висячие пробелы в начале
            .replace(/\s*$/, '') // удаляем висячие пробелы
            .replace(/^,/, '') // удаляем висячую запятую в начале
            .replace(/,$/, '') // удаляем висячую запятую в конце
            .split(',');
    },

    /**
     * Отрисовывает теги с чекбоксами
     */
    _renderCheckboxes: function() {
        var selectedTagsMap = this.model.get('selectedTags'),
            sortedTags = this.model.get('availableTags').slice().sort(function(a, b) {
                return a.toLowerCase() > b.toLowerCase() ? 1 : -1;
            });

        BEM.DOM.update(
            this._checkboxContainer,
            BEMHTML.apply(sortedTags.map(function(tag) {
                return {
                    block: 'b-group-tags2-edit',
                    elem: 'checkbox',
                    checked: selectedTagsMap[tag] ? 'yes' : '',
                    value: tag
                };
            })));
    }

});
