BEM.DOM.decl('b-href-control', {

    onSetMod: {
        js: function() {

            this._protocol = this.findBlockOn('protocol', 'select');
            this._href = this.findBlockOn('href', 'input');
            this._hint = this.elem('hint');
            this._maxLength = this.params.limit;

            // events
            this._subscribe = BEM.create('i-subscription-manager');
            this._subscribe.wrap(this._protocol).on('change', this._onChange, this);

            this._subscribe.wrap(this._href).on('change', this._onChange, this);
        }
    },

    /**
     * Возвращает текущее значение ссылки (включая протокол)
     * @returns {String}
     */
    val: function() {
        var href = $.trim(this._href.val()),
            protocol = this._protocol.val();

        // убираем http(s):// из урла
        // сплит по «?» что бы не искать протокол в get парметрах
        if (href.split('?')[0].match(/(http(s?):\/\/)+(.*)/)) {
            var aux = href;
            href = aux.replace(/(http(s?):\/\/)+(.*)/, '$3');
            protocol = aux.replace(/(http(s?):\/\/)+(.*)/, '$1');
        }

        return href ?
            {
                href: href,
                protocol: protocol,
                href_domain: u.extractDomain(href),
                url: protocol + href
            } :
            undefined;
    },

    /**
     * Меняет состояние инпута
     * @param {String} modVal - значение модификатора
     */
    toggleHref: function(modVal) {
        this._href.setMod('disabled', modVal);
        this._protocol.setMod('disabled', modVal);
    },

    /**
     * обновляет контрол по присланным данным
     * @param {String} href - ссылка на сайт
     * @param {String} protocol - протокол
     * @param {String} needQuiet - нужно ли не проверять ссылку
     */
    update: function(href, protocol, needQuiet) {
        // не дергаем перепроверку урла, если данные пришли извне
        // это либо из копирования баннера, либо после валидации
        this._quiet = needQuiet || false;
        this._href.val(href);
        this._protocol.val(protocol);
        this._quiet = false;
    },

    /**
     * Деструктор
     * @override
     */
    destruct: function() {
        this._subscribe.dispose();
        this.__base.apply(this, arguments);
    },

    /**
     * Устанавливает текст под полем ввода
     * @param {String|Array} content - текст или массив текстов
     * @private
     */
    _setHint: function(content) {
        this._hint && this._hint.text([].concat(content || '').join(' '));
    },

    /**
     * Обновляет состояние элементов управления после окончания валидации
     * @param {Array} alerts - список предупреждений валидации
     * @param {Object} data - ответ сервера при удачной валидации
     * @param {String} href - отправленный серверу href
     * @param {String} protocol - отправленный серверу протокол
     * @private
     */
    _onValidate: function(alerts, data, href, protocol) {
        if (alerts.length) {
            this._changeState('error', alerts, data);
            this._setHint(alerts);
        } else {
            if (data) {
               //обновляем данные (необходимо когда обрезали http(s) из урла)
                this.update(href, protocol, true);
            }
            this._changeState('', alerts, data);
            this._setHint('');
        }
    },

    /**
     * Меняет состояние блока и триггерит эвент
     * @param {String} modVal состояние блока
     * @param {Array} alerts - предупреждения сервера
     * @param {Object} validatedData данные сервера
     * @private
     */
    _changeState: function(modVal, alerts, validatedData) {
        this.setMod('state', modVal);
        this.trigger('state:changed', { isReady: modVal === '', validatedData: validatedData, alerts: alerts });
    },

    /**
     * Обработчик изменения ссылки или протокола
     * @private
     */
    _onChange: function() {
        var protocol = this._protocol.val(),
            maxUrlLength = this._maxLength - protocol.length;

        // максимальная длина ссылки меняется при смене протокола
        this._href.setLimit(maxUrlLength);
        this.setMod('protocol-warning', this.params.forceHttps && protocol !== 'https://' ? 'yes' : '');

        this._validate();
    },

    /**
     * Проверяет формат ссылки, максимальную длину и ответ от сервера (асинхронно)
     * @private
     */
    _validate: function() {
        if (this._quiet) return true;

        var value = this.val(),
            errors = [];

        this._validationState && this._validationState.reject();
        this._validationState = $.Deferred();
        this._validationState.done(this._onValidate.bind(this));

        if (value && value.url) {
            !u.isUrl(value.url) && errors.push(iget2('b-href-control', 'nepravilnyy-format-ssylki', 'Неправильный формат ссылки'));
            value.url.length > this._maxLength && errors.push(iget2('b-href-control', 'maksimalnaya-dlina-ssylki-s', 'Максимальная длина ссылки с учетом протокола ({foo}) превышена', {
                foo: this._maxLength
            }));

            if (!errors.length) {
                this._changeState('loading', []);
                this._setHint(iget2('b-href-control', 'proverka-ssylki', 'Проверка ссылки'));

                this._ajaxHrefCheck(value.protocol, value.href, function(state, data) {

                    if (!data.code) {
                        errors.push(data.text || iget2('b-href-control', 'ne-udalos-proverit-adres', 'Не удалось проверить адрес'));
                    }

                    // если протокол взяли из урла, а не контрола,
                    // его надо выставить
                    state.resolve(errors, data, value.href, value.protocol);
                }, this._validationState);
            } else {
                this._validationState.resolve(errors);
            }
        } else {
            this._validationState.resolve(errors);
        }
    },

    /**
     * Отправляет запрос на сервер для проверки ссылки
     * @param {String} protocol - протокол, например 'https://'
     * @param {String} url - ссылка без протокола
     * @param {Function} callback - функция, которую нужно вызвать после завершения
     * @param {Promise} state - promise, в котором накапливаются ошибки
     * @private
     */
    _ajaxHrefCheck: function(protocol, url, callback, state) {
        this._ajaxCheckXHR = u.urlCheckAndGetMarketRating({
            ctx: this,
            url: url,
            protocol: protocol,
            callback: callback.bind(this, state)
        });
    }
});
