/**
 * Данные об изображении
 * @typedef {Object} ImageData
 * @property {String} name имя файла
 * @property {String} width ширина изображения
 * @property {String} height высота изображения
 * @property {String} hash хэш изображения для генерации ссыылки на него
 * @property {String} group_id id группы изображения для генерации ссыылки на него
 * @property {Number} scale коэффициент масштабирования изображения
 */

/**
 * @event b-image-ad-selector#change
 * @type {Object} data данные выбранного изображения
 * @type {ImageData[]} data.images данные о выбранных изображениях
 * @type {CreativeData[]} data.creatives данные о выбранных креативах
 *
 * @event b-image-ad-selector#select
 * @type {ImageData} данные выбранного изображения
 *
 * @fires b-image-ad-selector#change событие изменения изображения.
 * Если один из подписчиков события change вернет false, то загрузка изображения будет отменена
 * @fires b-image-ad-selector#select событие выбора изображения
 * @fires b-image-ad-selector#reset событие сброса выбранного изображения
 * @fires b-image-ad-selector#cancel событие нажатия на кнопку отмены
 */
BEM.DOM.decl({ name: 'b-image-ad-selector' }, {
    onSetMod: {
        js: function() {
            this._imageAdLoader = this.findBlockOn(this.elem('loader'), 'b-image-add-loader');

            this._imageAdLoader.on('change:creatives', function(e, data) {
                var event = $.Event('change'),
                    creatives = data.creatives || data.images;

                this.trigger(event, data);

                // если один из подписчиков события change вернет false, то event.isDefaultPrevented вернет true
                if (event.isDefaultPrevented()) {
                    return false;
                } else {
                    if (data.images) {
                        this._imageData = creatives;
                        this._creativeData = null;
                    }

                    if (data.creatives) {
                        this._imageData = null;
                        this._creativeData = creatives;
                    }

                    this.trigger('button-state-changed', true);
                    this._updatePreview(creatives[0], !!data.creatives);
                }
            }, this);
        }
    },

    getImageLoader: function() {
        return this._imageAdLoader;
    },

    /**
     * Устанавливает заголовок блока
     * @param {String} text текст заголовка
     */
    setTitle: function(text) {
        if (this.findElem('title').length) {
            BEM.DOM.replace(this.findElem('title'), BEMHTML.apply({
                block: this.__self.getName(),
                elem: 'title',
                content: text
            }));
        } else {
            text && BEM.DOM.prepend(this.elem('pane'), BEMHTML.apply({
                block: this.__self.getName(),
                elem: 'title',
                content: text
            }));
        }
    },

    /**
     * Устанавливает текст подсказки
     * @param {Object} data текст подсказки
     */
    setHint: function(data) {
        BEM.DOM.replace(this.findElem('hint'), BEMHTML.apply({
            block: this.__self.getName(),
            elem: 'hint',
            data: data
        }, this));
    },

    /**
     * Сбрасывает выбранное изображение.
     */
    reset: function() {
        this._imageData = null;
        this._creativeData = null;
        this._imageAdLoader.resetImage();
        this._updatePreview();
        this.trigger('button-state-changed', false);
        this.trigger('reset');
    },

    /**
     * Внешний триггер сохранения
     */
    triggerSave: function() {
        this.trigger('select', {
            images: this._imageData,
            creatives: this._creativeData
        });
    },

    /**
     * Обновляет превью баннера
     * @param {ImageData} [imageData] данные нового изображения
     * @param {Boolean} [isCreative] тип отображаемого объекта
     */
    _updatePreview: function(imageData, isCreative) {
        var previewUrl,
            previewHtml,
            data = imageData;

        if (data) {
            data = u.imageAd.fixDimensions(data);
            previewUrl = isCreative ?
                data.preview_url :
                u.getImageUrl({ namespace: 'direct-picture', mdsGroupId: data.group_id, hash: data.hash });

            previewHtml = BEMHTML.apply({
                block: 'b-image-ad-selector',
                elem: 'preview-frame',
                content: [
                    {
                        block: 'image-preview',
                        mods: {
                            type: 'scale'
                        },
                        imageUrl: previewUrl,
                        videoPreview: data.live_preview_url,
                        scale: data.scale,
                        width: data.width,
                        height: data.height
                    }
                ]
            });
        }

        BEM.DOM.update(this.elem('preview'), previewHtml || '');
    },

    destruct: function() {
        this._imageAdLoader.un('change');

        return this.__base.apply(this, arguments);
    }
});
