/**
 * Данные об изображении
 * @typedef {Object} ImageData
 * @property {String} name имя файла
 * @property {String} width ширина изображения
 * @property {String} height высота изображения
 * @property {String} hash хэш изображения для генерации ссылки на него
 * @property {String} group_id id группы изображения для генерации ссылки на него
 * @property {Number} scale коэффициент масштабирования изображения
 */

/**
 * @event b-image-add-loader#change
 * @type {ImageData} данные сохраненного изображения
 * @fires b-image-add-loader#change событие успешной загрузки изображения.
 * Если один из подписчиков события change вернет false, то загрузка изображения будет отменена
 * @fires b-image-add-loader#reset событие сброса сохраненного значения файла
 */
BEM.DOM.decl({ name: 'b-image-add-loader', baseBlock: 'i-glue' }, {
    onSetMod: {
        js: function() {
            this.__base.apply(this, arguments);

            this._availableBannerCount = this.params.availableBannerCount;

            this._dropdown = this.findBlockOn('select', 'dropdown');

            this.bindTo('chooser-item', 'click', this._onFileSelectItemClick);

            this._subscriptionManager = BEM.create('i-subscription-manager');

            this.updateItemStates();

            this._subscriptionManager.wrap(this.model)
                .on('needDisabled', 'change', function(e, data) {
                    this._getSwitcherButton().setMod('progress', data.value ? 'yes' : '');
                }, this);

            this._constructorType = undefined;
        },

        loading: function(modName, modVal) {
            this._getSpin().setMod('progress', modVal);
        }
    },

    /**
     * Возвращает спинер
     * @returns {BEM.DOM}
     * @private
     */
    _getSpin: function() {
        return this._spin || (this._spin = this.findBlockOn('spin', 'spin2'));
    },

    /**
     * Возвращает кнопку выпадушки
     * @returns {BEM.DOM}
     * @private
     */
    _getSwitcherButton: function() {
        return this._switcherButton || (this._switcherButton = this.findBlockOn('switcher-button', 'button2'));
    },

    /**
     * Обновляет модель блока новыми данными об изображении
     * @param {Object} imageData данные нового изображения
     */
    setImage: function(imageData) {
        this.model.setImage(imageData);
    },

    /**
     * Обновляет модель блока новыми данными о креативе
     * @param {Object} creativeData
     */
    setCreative: function(creativeData) {
        this.model.setCreative(creativeData);
    },

    /**
     * Запоминает новое доступное для добавления количество баннеров
     * @param {Number} count - доступное для добавления количество баннеров
     */
    updateAvailableBannerCount: function(count) {
        this._availableBannerCount = count;
    },

    /**
     * Сбрасывает сохраненное значение файла.
     */
    resetImage: function() {
        this.model.clear();
        this.trigger('reset');
    },

    /**
     * @type {Number}
     * Максимальный размер изображения Медийного Баннера (mcbanner)
     */
    _maxFileSize: 0.12,

    /**
     * @type {Number}
     * Таймаут загрузки изображения
     */
    _fileUploadTimeout: 180000,

    /**
     * @type {Number}
     * Таймаут загрузки посредством объекта xhr
     */
    _ajaxTimeout: 60000,

    /**
     * @type {Object}
     * Объект с текстами клиентских ошибок
     */
    _errors: function(params) {
        return {
            upload: iget2('b-image-add-loader', 'oshibka-dobavleniya-fayla', 'Ошибка добавления файла'),
            tooBig: iget2('b-image-add-loader', 'razmer-fayla-bolshe-dopustimogo', 'Размер файла больше допустимого (120 КБ)'),
            format: iget2('b-image-add-loader', 'nedopustimyy-tip-fayla-ispolzuyte', 'Недопустимый тип файла, используйте графические форматы GIF, JPEG, PNG'),
            server: iget2('b-image-add-loader', 'oshibka-servera', 'Ошибка сервера. Попробуйте, пожалуйста, позже'),
            500: iget2('b-image-add-loader', 'oshibka-servera', 'Ошибка сервера. Попробуйте, пожалуйста, позже'),
            tooMuch: iget2('b-image-add-loader', 'too-much', 'В эту группу можно загрузить только {bannerCount} изображение', {
                some: 'В эту группу можно загрузить только {bannerCount} изображения',
                many: 'В эту группу можно загрузить только {bannerCount} изображений',
                count: params.count || 0,
                bannerCount: params.count || 0
            })
        };
    },

    _defaultItems: null,

    /**
     *
     * @returns {Array}
     * @private
     */
    _getDefaultItems: function() {
        return this._defaultItems ||
            (this._defaultItems = u['b-image-add-loader'].getDefaultItems({
                mediaType: this.params.mediaType
            }));
    },

    /**
     * Обновляет видимые типы креативов
     * @param {Array} [items] — типы креативов, которые нужно показать
     * @private
     */
    updateItemStates: function(items) {
        var params = this.params,
            modelBanner = params.dmParams && BEM.MODEL.getOne({ name: params.dmParams.name, id: params.dmParams.id }),
            modelGroup = params.dmParams &&
                BEM.MODEL.getOne({ name: params.dmParams.parentName, id: params.dmParams.parentId }),
            popupDomElem = this._dropdown.getPopup().domElem,
            activeItems = items || u['b-image-add-loader'].getActiveItems({
                image: params.modelData.image,
                creative: params.modelData.creative,

                isNewBanner: params.isNewBanner,
                adType: modelBanner && modelBanner.get('ad_type'),

                mediaType: params.mediaType,
                cpmGroupType: modelGroup && modelGroup.get('cpmGroupType')
            });

        activeItems.length && this._getDefaultItems().forEach(function(item) {
            this.toggleMod(
                this.findElem(popupDomElem, 'chooser-group-title chooser-item', 'position', item),
                'visibility', 'visible', 'hidden',
                activeItems.indexOf(item) !== -1
            );
        }, this);
    },

    /**
     * Событие срабатывает на выбор в селекте одного из вариантов загрузки изображения
     * @param {Event} e объект события
     * @private
     */
    _onFileSelectItemClick: function(e) {
        if (this.getMod(e.data.domElem, 'disabled', 'yes')) {
            return;
        }

        this._openFileLoaderDialog(this.getMod(e.data.domElem, 'type'));

        this._dropdown.getPopup().delMod('visibility');
    },

    /**
     * Открывает диалоговое окно для загрузки файла с интерфейсом, соответствующим выбранному способу загрузки
     * @param {String} dialogType тип загрузки файла
     * @private
     */
    _openFileLoaderDialog: function(dialogType) {
        switch (dialogType) {
            case 'local':
                this._getLocalFileUploader().openFileDialog();
                break;

            case 'local-multiple':
                this._getLocalFileUploader(true).openFileDialog();
                break;

            case 'external':
                this._getExternalFileUploader().show();
                break;

            case 'gallery':
                this._getGalleryFileChooser().show();
                break;

            case 'constructor-create':
                this._getConstructorFrame()
                    .show('presets', this.params.clientId, this._getCreativeValidationRules());
                break;

            case 'constructor-create-adaptive-image-ad':
                this._getConstructorFrame()
                    .show('add', this.params.clientId, this._getCreativeValidationRules(), null, null, true);
                break;

            case 'constructor-ideas':
                this._getConstructorFrame()
                    .show('ideas', this.params.clientId, this._getCreativeValidationRules());
                break;

            case 'constructor-select':
                this._getConstructorFrame()
                    .show('creatives', this.params.clientId, this._getCreativeValidationRules());
                break;

            case 'constructor-html5-add':
                this._getConstructorFrame()
                    .show('add', this.params.clientId, this._getCreativeValidationRules(), true, this._getProductType());
                break;

            case 'constructor-html5-choose-from-uploaded':
                this._getConstructorFrame()
                    .show('', this.params.clientId, this._getCreativeValidationRules(), true, this._getProductType());
                break;

            case 'video-addition-create':
                this._getConstructorFrame({ type: 'video-creative' })
                    .show('add', this.params.clientId);
                break;

            case 'audio-addition-create':
                this._getConstructorFrame({ type: 'video-creative' })
                    .show('add', this.params.clientId, 'cpm-audio');
                break;

            case 'audio-addition-gallery':
                this._getConstructorFrame({ type: 'video-creative' })
                    .show('creatives', this.params.clientId, 'cpm-audio');
                break;

            case 'video-addition-gallery':
                this._getConstructorFrame({ type: 'video-creative' })
                    .show('creatives', this.params.clientId);
                break;

            case 'video-outdoor-create':
                this._getConstructorFrame({ type: 'video-creative' })
                    .show('add', this.params.clientId, 'cpm-outdoor');
                break;

            case 'video-outdoor-gallery':
                this._getConstructorFrame({ type: 'video-creative' })
                    .show('creatives', this.params.clientId, 'cpm-outdoor');
                break;

            case 'video-indoor-create':
                this._getConstructorFrame({ type: 'video-creative' })
                    .show('add', this.params.clientId, 'cpm-indoor');
                break;

            case 'video-indoor-gallery':
                this._getConstructorFrame({ type: 'video-creative' })
                    .show('creatives', this.params.clientId, 'cpm-indoor');
                break;

            case 'cpc-video-create':
                this._getConstructorFrame({ type: 'cpc-video' })
                    .show('presets', this.params.clientId);
                break;

            case 'cpc-video-gallery':
                this._getConstructorFrame({ type: 'cpc-video' })
                    .show('creatives', this.params.clientId);
                break;

            case 'cpm-geoproduct-canvas-create':
                this._getConstructorFrame({ type: 'cpm-geoproduct' })
                    .show('presets', this.params.clientId, this._getCreativeValidationRules());
                break;

            case 'cpm-geoproduct-canvas-select':
                this._getConstructorFrame({ type: 'cpm-geoproduct' })
                    .show('creatives', this.params.clientId, this._getCreativeValidationRules());
                break;

            case 'mobile-content-video-create':
                this._getConstructorFrame({ type: 'mobile-content-video' })
                    .show('presets', this.params.clientId);
                break;

            case 'mobile-content-video-creatives':
                this._getConstructorFrame({ type: 'mobile-content-video' })
                    .show('creatives', this.params.clientId);
                break;

            case 'mobile-content-video-gallery':
                this._getConstructorFrame({ type: 'mobile-content-video' })
                    .show('creatives', this.params.clientId);
                break;

            case 'cpm-geoproduct-html5-add':
                this._getConstructorFrame({ type: 'cpm-geoproduct' })
                    .show('add', this.params.clientId, this._getCreativeValidationRules(), true, 'CPM_GEOPRODUCT');
                break;

            case 'cpm-geoproduct-html5-choose-from-uploaded':
                this._getConstructorFrame({ type: 'cpm-geoproduct' })
                    .show('', this.params.clientId, this._getCreativeValidationRules(), true, 'CPM_GEOPRODUCT');
                break;
        }
    },

    /**
     * Инициализирует и возвращает блок для загрузки изображения с компьютера
     * @param {Boolean} [multiple]
     * @return {BEM.DOM<i-local-file-uploader>} блок для загрузки файла изображения с компьютера
     * @private
     */
    _getLocalFileUploader: function(multiple) {
        var uploader = this._localFileUploader;

        if (!uploader) {
            this._localFileUploader = uploader = BEM.create(
                'i-local-file-uploader',
                {
                    name: 'picture',
                    mimeTypes: [
                        'image/jpg',
                        'image/jpeg',
                        'image/png',
                        'image/gif'
                    ]
                }
            );

            uploader.on('select', function(e, data) {
                this._onLocalFileSelect(data.blob);
            }, this);
        }

        multiple && uploader.setMultiple(!this.model.get('isImageLoaded'));

        return uploader;
    },

    /**
     * Событие, срабатывает на выбор файла пользователем из файловой системы.
     * Первично проверяет на ошибки.
     * Если ошибок нет, то делает запрос к серверу.
     * @param {Object} blob объект загруженного файла (https://developer.mozilla.org/ru/docs/Web/API/File)
     * @private
     */
    _onLocalFileSelect: function(blob) {
        var file,
            error,
            params = {},
            allowableNumberOfImages = this._availableBannerCount + 1;

        if (!blob) {
            error = 'upload';
        } else if (blob.files.length === 1) {
            file = blob && blob.files[0];

            if (file && (file.size > this._maxFileSize * 1024 * 1024)) {
                error = 'tooBig';
            } else if (file && file.type !== '' && !/^image\//.test(file.type)) {
                error = 'format';
            }
        } else if (blob.files.length > allowableNumberOfImages) {
            error = 'tooMuch';
            params.count = allowableNumberOfImages;
        }

        if (error) {
            this._showError(this._getErrorText(error, params));
            return;
        }

        this.setMod('loading', 'yes');

        this._ajaxSend('picture',
           this._getRequestParams({
               picture: blob,
               banner_type: this.params.isMcbanner ? 'mcbanner' : 'image_ad'
           }),
           this._onLocalFileUpload.bind(this),
           this._onLocalFileUpload.bind(this)
        );
    },

    /**
     * Событие, срабатывающее на возврат ответа от сервера, после загрузки изображения из файловой системы
     * @param {Object} data ответ сервера
     * @private
     */
    _onLocalFileUpload: function(data) {
        var filteredData;

        if (!data) {
            return this._showError(this._getErrorText('upload'));
        } else if (data.status) {
            return this._showError(iget2('b-image-add-loader', 'oshibka-servera', 'Ошибка сервера. Попробуйте, пожалуйста, позже'));
        } else if (data.error) {
            return this._showError(data.error);
        } else if (!data.result) {
            return this._showError(this._getErrorText('upload'));
        }

        if (u._.isArray(data.result)) {
            filteredData = data.result.reduce(function(res, image) {
                if (image.error) {
                    res.errors || (res.errors = []);
                    res.errors.push(image.client_filename + ' &nbsp;&mdash;&nbsp;  ' + image.error);
                } else {
                    res.images.push(image);
                }

                return res;
            }, { images: [] });
        } else {
            filteredData = { images: [data] };
        }

        this._onCreativeSelect(filteredData);
    },

    /**
     * Инициализирует и возвращает блок для загрузки изображения по ссылке
     * @return {BEM.DOM<b-external-file-uploader>} блок для загрузки файла изображения по ссылке
     * @private
     */
    _getExternalFileUploader: function() {
        var uploader = this._externalFileUploader;

        if (!uploader) {
            uploader = BEM.DOM.blocks['b-external-file-uploader'].create({
                title: iget2('b-image-add-loader', 'zagruzka-kreativa-po-ssylke', 'Загрузка креатива по ссылке'),
                placeholder: iget2('b-image-add-loader', 'adres-kreativa', 'Адрес креатива')
            });

            uploader.on('save', function(e, data) {
                uploader.setMod('loading', 'yes');

                this._ajaxSend('url',
                    this._getRequestParams({
                        url: data.value,
                        banner_type: this.params.isMcbanner ? 'mcbanner' : 'image_ad'
                    }),
                    this._onExternalFileUpload.bind(this),
                    this._onExternalFileUpload.bind(this));
            }, this);

            this._externalFileUploader = uploader;
        }

        return uploader;
    },

    /**
     * Событие, срабатывающее на возврат ответа от сервера, после загрузки изображения по ссылке
     * @param {Object} data ответ сервера
     * @private
     */
    _onExternalFileUpload: function(data) {
        var uploader = this._externalFileUploader;

        uploader.setMod('loading', '');

        if (!data) {
            uploader.showErrors(this._getErrorText('upload'));
            return;
        }

        // обработка падений Zora, когда присылается честный xhr ответ
        // 304 = Not modified
        if ((data.status < 200 || data.status >= 300) && data.status !== 304) {
            uploader.showErrors(this._getErrorText('server'));
            return;
        }

        if (data.error) {
            uploader.showErrors(data.error);
            return;
        }

        uploader.hide();

        this._changeCreatives({ images: [data] });
    },

    /**
     * Инициализирует и возвращает блок для выбора изображения из галереи ранее загруженных
     * @return {BEM.DOM<b-external-file-uploader>} блок выбора файла из ранее загруженных
     * @private
     */
    _getGalleryFileChooser: function() {
        if (!this._galleryFileChooser) {
            this._galleryFileChooser = BEM.DOM.blocks['b-gallery-file-chooser'].create({
                title: iget2('b-image-add-loader', 'ranee-zagruzhennye-kreativy', 'Ранее загруженные креативы'),
                galleryItems: this.params.gallery
            });

            this._galleryFileChooser.on('save', function(e, data) {
                this._changeCreatives({ images: [data.item] });
            }, this);
        }

        return this._galleryFileChooser;
    },

    //<editor-fold desc="{creatives...}">

    /**
     * Формирование правил валидации для КК
     * @returns {Object}
     * @private
     */
    _getCreativeValidationRules: function() {
        var creative = this.model.get('creative'),
            single = creative && (
                creative.is_adaptive === '1' ?
                    'adaptive' :
                    { width: creative.width, height: creative.height }
                );

        return !this.params.isNewBanner && single ?
            { single: single } :
            { maxCount: this._availableBannerCount + 1 }
    },

    /**
     * Возвращает тип продукта в зависимости от mediaType
     * @private
     */
    _getProductType: function() {
        switch (this.params.mediaType) {
            case 'cpm_yndx_frontpage':
                return 'CPM_YNDX_FRONTPAGE';

            default:
                return 'GENERAL';
        }
    },

    /**
     * Инициализирует и возвращает блок для создания/выбора креативов через конструктор
     * @param {Object} mods - модификатор для "b-banner-storage-frame"
     * @return {BEM.DOM<b-banner-storage-selector>} блок для создания/выбора креативов через конструктор
     * @private
     */
    _getConstructorFrame: function(mods) {
        var uploader = BEM.DOM.blocks['b-banner-storage-frame'].create(mods);
        this._constructorType = mods ? mods.type : undefined;

        uploader.on('select', function(e, creatives) {
            this._onCreativeSelect({ creatives: creatives });
        }, this);

        uploader.on('close', function() {
            uploader.destruct();
        });

        uploader.on('ready', function(e, data) {
            this.trigger('ready', { bannerStorage: e.block });
        }, this);

        return uploader;
    },

    /**
     * Возвращает первый валидный href
     * @returns {Object}
     * @private
     */
    _getValidHrefData: function() {
        var isMobileCampaign = this.params.isMobileCampaign,
            dmParams = this.params.dmParams,
            banners = BEM.MODEL.getOne({ name: dmParams.parentName, id: dmParams.parentId }).getBanners();

        if (isMobileCampaign) {
            return this._getValidMobileCampaignHrefData(banners)
        }

        var bannerWithValidHref = u._.find(banners, function(b) {
            return b.get('href_model') && b.validate('href_model').valid;
        });

        if (bannerWithValidHref) {
            return {
                href: bannerWithValidHref.get('href_model').get('href'),
                protocol: bannerWithValidHref.get('href_model').get('url_protocol')
            };
        }

        return {};
    },

    /**
     * Возвращает последний валидный href мобильной компании.
     * @param {BannerDataModel[]} banners
     * @returns {Object}
     * @private
     */
    _getValidMobileCampaignHrefData: function(banners) {
        var validHrefBanners = banners
            .filter(function(b) { return b.get('href') && b.validate('href').valid; }),
            lastBanner = u._.last(validHrefBanners);

        if (lastBanner) {
            return {
                href: lastBanner.get('href'),
                protocol: lastBanner.get('url_protocol')
            }
        }

        return {}
    },

    /**
     * Обработчик события "выбран креатив в КК или картинки"
     * @private
     */
    _onCreativeSelect: function(data) {
        var hrefPopup,
            creatives = data.creatives || data.images,
            hrefData;

	// для "баннера в метро" (cpm-geoproduct) отсутствует сайт
        if (creatives.length > 1 && this.params.askGroupLink && this._constructorType !== 'cpm-geoproduct') {
            hrefData = this._getValidHrefData();

            hrefPopup = BEM.DOM.blocks['b-href-popup'].create(function(href) {
                data.href = href;
                this._changeCreatives(data);
            }, this, {
                href: hrefData.href,
                protocol: hrefData.protocol,
                isMobileCampaign: this.params.isMobileCampaign,
                btnOkText: iget2('b-image-add-loader', 'dobavit', 'Добавить'),
                btnCancelText: iget2('b-image-add-loader', 'sozdat-bez-ssylki', 'Создать без ссылки')
            });

            hrefPopup.show();
        } else {
            this._changeCreatives(data);
        }
    },

    //</editor-fold>

    /**
     * Возвращает текст ошибки по коду.
     * @param {String} errorCode код ошибки
     * @param {Object} [params]
     * @return {String}
     * @private
     */
    _getErrorText: function(errorCode, params) {
        return this._errors(params)[errorCode] || '';
    },

    /**
     * Показывает ошибку в модальном окне.
     * @param {String} error текст ошибки
     * @private
     */
    _showError: function(error) {
        BEM.blocks['b-confirm'].open({
            message: error,
            type: 'alert',
            onYes: function() { this.setMod('loading', ''); }
        }, this);
    },

    /**
     * Возвращает хеш данных, необходимых для отправки изображения на сервер
     * @param {Object} params дополнительные параметры
     * @returns {Object}
     * @private
     */
    _getRequestParams: function(params) {
        var defaultParams = {
            csrf_token: BEM.blocks['i-global'].param('csrf_token'),
            ulogin: u.consts('ulogin') || '',
            cmd: (params.url || params.picture.files.length === 1) ? 'uploadImage' : 'uploadMultipleAdImageFiles'
        };

        this.params.bannerId && (defaultParams.bid = this.params.bannerId);

        return u._.extend(defaultParams, params);
    },

    /**
     * Делает запрос к серверу для загрузки изображения
     * @param {('url'|'picture')} type определяет как загружаем файл (по ссылке или файлом с компьютера)
     * @param {Object} params данные для отправки на сервер
     * @param {Function} onSuccess callback на удачное завершение запроса
     * @param {Function} onError callback на неудачное завершение запроса
     * @private
     */
    _ajaxSend: function(type, params, onSuccess, onError) {
        var request = BEM.create(type === 'picture' ? 'i-request_type_file' : 'i-request_type_ajax', {
            url: u.consts('SCRIPT'),
            dataType: 'json',
            type: 'POST',
            timeout: type === 'picture' ? this._fileUploadTimeout : this._ajaxTimeout
        });

        request.get(params, onSuccess, onError);
    },

    /**
     * Меняет выбранный креатив на новый
     * @param {Object} data
     * @param {Array} [data.images] — выбранные картинки
     * @param {Array} [data.errors] — ошибки выбранных картинок
     * @param {Array} [data.creatives] — выбранные креативы
     * @param {Object} data.href — объект с данными по общей ссылке
     * @param {String} data.href.protocol — протокол
     * @param {String} data.href.href — ссылка
     * @param {String} data.href.href_domain — домен ссылки
     * @param {String} data.href.url — полный url
     * @private
     */
    _changeCreatives: function(data) {
        var event = $.Event('change:creatives');

        this.setMod('loading', '');

        this.trigger(event, data);

        // если один из подписчиков события change:creatives вернет false, то event.isDefaultPrevented вернет true
        if (!event.isDefaultPrevented()) {
            data.images && this.model.setImage(data.images[0]);
            data.creatives && this.model.setCreative(data.creatives[0]);
        }
    },

    _onMouseOverOutHelp: function(e) {
        if (e.type === 'pointerover') {
            this._tipman || (this._tipman = BEM.create('tipman', {
                tipMods: { theme: 'white', size: 's', target: 'anchor' },
                tipMix: { block: 'b-image-add-loader', elem: 'tooltip-content' },
                tipJs: { to: ['right'] },
                popupMods: { 'on-scroll': 'close' }
            }));

            this._tipman.show({
                owner: e.data.domElem,
                content: this.elemParams(e.data.domElem)['tooltip-content']
            });
        } else {
            this._tipman && this._tipman.hide();
        }
    },

    destruct: function() {
        this._subscriptionManager.dispose();
        this._subscriptionManager.destruct();
        this._tipman && BEM.DOM.destruct(this._tipman);

        return this.__base.apply(this, arguments);
    }

}, {

    live: function() {
        this.liveBindTo('tooltip-hover', 'pointerover pointerout', function(e) { this._onMouseOverOutHelp(e) });

        return false;
    }

});
