BEM.DOM.decl('b-manage-vcards', {

    onSetMod: {
        js: function() {
            this
                ._createModels('m-vcard', 'vcard_id', this.params.vcards)
                ._createModels('m-banner', 'bid', this.params.banners);

            u.graspSelf.call(this, {
                banners: 'b-vcard-banners on vcard-banners',
                vcards: 'b-vcards on vcards'
            });

            this._initEvent();
        }
    },

    /**
     * Создаёт модели
     * @param {String} modelName название модели
     * @param {String} idKey ключ id параметра модели в объекте данных, которыми инициализируется модель
     * @param {Array} dataList список объектов данных
     * @private
     */
    _createModels: function(modelName, idKey, dataList) {
        dataList.forEach(function(modelData) {
            BEM.MODEL.create({ name: modelName, id: modelData[idKey] }, modelData);
        });

        return this;
    },

    /**
     * Инициализация событий моделей
     * @private
     */
    _initEvent: function() {
        this.vcards
            // событие запроса на показ баннеров, запрашиваемое из вне
            .on('request-show-banners', function(e, vcardId) {
                this.banners.showVCardBanners(vcardId);
            }, this)

            // событие запроса на привязывание выбранных баннеров с визиткой, запрашиваемое из вне
            .on('request-assign-vcard', function(e, vcardId) {
                var _this = this,
                    // убираем те баннеры, которые уже привязаны к этой визитке
                    bids = this.banners.getCheckedBids().filter(function(bid) {
                        var banner = BEM.MODEL.getOne({ name: 'm-banner', id: bid });

                        return !(banner.get('vcard') && banner.get('vcard').vcard_id === vcardId);
                    });

                bids.length && this._confirmUnlinkScenario('assign-vcard', bids).done(function(allow) {
                    allow && _this._doAssignVCard(bids, vcardId);
                });
            }, this)

            // событие показа визитки в iframe
            .on('show-vcard', function(e, vcardId) {
                this.vcards.setAssignableState(this.banners.getCheckedBids().length);
            }, this)

            // событие создания визитки, после которого нужно привязать её к текущему открытому на редактирование связи баннеру
            // или привязать от старой визитки к новой её баннеры
            .on('create-vcard', function(e, data) {
                var bids,
                    vcardId = data.vcardId,
                    oldVCardId = data.oldVCardId;

                if (this._editingModel) {
                    this._resetAssignment(this._editingModel.get('bid'), vcardId);

                    this._editingModel = null;
                }

                if (oldVCardId) {
                    // выбираем баннеры, которые привязаны к старой визитке
                    bids = BEM.MODEL
                        .get('m-banner')
                        .filter(function(banner) {
                            return banner.get('vcard') && banner.get('vcard').vcard_id === oldVCardId;
                        })
                        .map(function(banner) {
                            return banner.get('bid');
                        });

                    bids.length && this._resetAssignment(bids, vcardId);
                }

                this.vcards.afterCurrentEvent(function() {
                    this._scrollToVCard(
                        this.findElem(this.vcards.domElem, 'scrollable-box'),
                        vcardId);
                }, this);
            }, this)

            // события открытия/закрытия редактора визитки
            .on('frame-opening', this._onFrameOpenOrClose, this)
            .on('frame-close', this._onFrameOpenOrClose, this);

        this.banners
            // при выборе баннеров необходимо давать знать о возможности привязки визиток к баннерам
            .on('check', function(e, bids) {
                this.vcards.setAssignableState(bids.length > 0);
            }, this)

            // событие запроса на редактирование или создание визитки, запрашиваемое из вне
            .on('request-edit-vcard', function(e, bid) {
                var bannerModel = BEM.MODEL.getOne({ name: 'm-banner', id: bid }),
                    vcard = bannerModel.get('vcard');

                this._editingModel &&
                    this.banners.toggleDisabled(this._editingModel, false);

                this.banners.toggleDisabled(bannerModel, true);

                this._editingModel = bannerModel;

                this.vcards.openEditor(vcard && vcard.vcard_id, bid);
            }, this)

            // событие запроса на добавление визитки, запрашиваемое из вне
            .on('request-unlink-vcard', function(e, bid) {
                var _this = this;

                this._confirmUnlinkScenario('unlink-vcard', bid).done(function(allow) {
                    allow && _this._doUnlinkVCard(bid);
                });
            }, this);

        this.bindTo(this.findElem(this.banners.domElem, 'scrollable-box'), 'scroll', $.throttle(function() {
            this.banners.togglePhrasesPopup();
        }, 500, this));

        [this.vcards, this.banners].forEach(function(block) {
            block.getFilterModel().on('change', function(e, data) {
                data.field === 'archived' || this.findElem(block.domElem, 'scrollable-box').scrollTop(0);
            }, this);
        }, this);
    },

    _onVcardAssigned: function(bids, vcardId, data) {
        var model,
            vcard;

        if (data && data.success) {

            if (data.result.new_vcard) {
                vcard = data.result.new_vcard;

                vcard.uid += '';
                vcard.cid += '';
                vcard.vcard_id += '';

                vcard.vcard_id &&
                    (model = BEM.MODEL.getOne({ name: 'm-vcard', id: vcard.vcard_id }));

                model ||
                    (model = this.vcards.addCreatedVCard(vcard));

                model.update(vcard);

                vcardId = model.get('vcard_id');

                this._resetAssignment(bids, vcardId);

                this.vcards.afterCurrentEvent(function() {
                    this._scrollToVCard(
                        this.findElem(this.vcards.domElem, 'scrollable-box'),
                        vcardId);
                }, this);
            } else {
                this._resetAssignment(bids, vcardId);
            }
        }
    },

    /**
     * Делает запрос на сервер для привязывания баннеров и визитки
     * @param {String[]} bids список идентификаторов баннеров
     * @param {String} vcardId идентификатор визитки
     * @private
     */
    _doAssignVCard: function(bids, vcardId) {
        this.vcards.setAssignableState(false);

        this.setMod(this.elem('vcards'), 'cursor', 'wait');

        var done = (function() {
            this.delMod(this.elem('vcards'), 'cursor');
        }).bind(this);

        BEM.blocks['i-web-api-request'].vcards
            .assignVcard(u.consts('ulogin'), bids, this.params.cid , vcardId)
            .then(this._onVcardAssigned.bind(this, bids, vcardId))
            .then(done, done);
    },

    /**
     * Делает запрос к серверу для отвязки визитки от баннера
     * @param {String} bid идентификатор баннера
     * @private
     */
    _doUnlinkVCard: function(bid) {
        var bannerModel = BEM.MODEL.getOne({ name: 'm-banner', id: bid });

        this.banners.toggleActions(bannerModel, true);

        this.setMod(this.elem('vcard-banners'), 'cursor', 'wait');

        var done = (function() {
            this.banners.toggleActions(bannerModel, false);

            this.delMod(this.elem('vcard-banners'), 'cursor');
        }).bind(this);

        BEM.blocks['i-web-api-request'].vcards
            .unassignVcard(u.consts('ulogin'), bid, this.params.cid)
            .then((function(data) {
                data && data.success && this._resetAssignment(bid);
            }).bind(this))
            .then(done, done);
    },

    /**
     * Вызывает диалог подтверждения следования сценарию при отвязке визитки от баннера или привязки (проверка на смену баннером визитки).
     * @param {String} type тип события сценария (привязка/отвязка визитки...)
     * @param {String|String[]} bids идентификатор или список идентификаторов баннеров, от которых будут отвязаны их визитки
     * @returns {$.Deferred}
     * @private
     */
    _confirmUnlinkScenario: function(type, bids) {
        var confirmMessage,
            deleteList,
            deleteCount,
            names,
            dfd = $.Deferred(),
            affectedMap = {},
            affectedItems = [];

        bids = u._.isArray(bids) ? bids : [bids];

        bids.forEach(function(bid) {
            var affected,
                vcard,
                vcardId,
                banner = BEM.MODEL.getOne({ name: 'm-banner', id: bid });

            if (!banner.get('vcard')) {
                return;
            }

            vcardId = banner.get('vcard').vcard_id;

            affected = affectedMap[vcardId];

            if (!affected) {
                vcard = BEM.MODEL.getOne({ name: 'm-vcard', id: vcardId });

                affected = affectedMap[vcardId] = {
                    total: vcard.get('bannersCount').total
                };

                affectedItems.push(vcard);
            }

            affected.total && affected .total--;
        });

        deleteList = affectedItems.filter(function(vcard) {
            var vcardId = vcard.get('vcard_id'),
                affected = affectedMap[vcardId];

            return !affected.total;
        });

        deleteCount = deleteList.length;

        if (deleteCount) {
            names = deleteList.map(function(vcard) {
                return ['«', '»'].join(vcard.get('name'));
            }).join(', ');

            confirmMessage = [
                u.pluralizeWord([
                    iget2('b-manage-vcards', 'vizitka-name-nazvanie-bolshe', 'Визитка {name} больше не используется и будет удалена.', {
                        name: names,
                        context: 'name - «Название»'
                    }),
                    iget2('b-manage-vcards', 'vizitki-names-nazvanie-1', 'Визитки {names} больше не используются и будут удалены.', {
                        names: names,
                        context: 'names - «Название 1», «Название 2»'
                    })
                ], deleteCount),
                ({
                    'assign-vcard': iget2('b-manage-vcards', 'naznachit', 'Назначить?'),
                    'unlink-vcard': u.pluralizeWord([
                        u.pluralizeWord([
                            iget2('b-manage-vcards', 'vy-tochno-hotite-udalit', 'Вы точно хотите удалить визитку из объявления?'),
                            iget2('b-manage-vcards', 'vy-tochno-hotite-udalit-100', 'Вы точно хотите удалить визитку из объявлений?')
                        ], bids.length),
                        iget2('b-manage-vcards', 'vy-tochno-hotite-udalit-101', 'Вы точно хотите удалить визитки из объявлений?')
                    ], deleteCount)
                })[type] || ''
            ].join(' ');
        }

        type === 'unlink-vcard' && !confirmMessage &&
            (confirmMessage = u.pluralizeWord([
                iget2('b-manage-vcards', 'vy-tochno-hotite-udalit', 'Вы точно хотите удалить визитку из объявления?'),
                iget2('b-manage-vcards', 'vy-tochno-hotite-udalit-100', 'Вы точно хотите удалить визитку из объявлений?')
            ], bids.length));

        if (confirmMessage) {
            BEM.blocks['b-confirm'].open({
                message: confirmMessage,
                onYes: function() { dfd.resolve(true) },
                onNo: function() { dfd.resolve(false) }
            });
        } else {
            dfd.resolve(true);
        }

        return dfd;
    },

    /**
     * Переназначает связи баннеров и их визиток, если не передан баннер, то для указанных баннеров визитки
     * будут отвязаны, иначе все баннеры будут связаны с указанной визиткой
     * @param {String|String[]} bids идентификатор или список идентификаторов баннеров
     * @param {String} [newVCardId] идентификатор визитки, которая будет привязана к баннерам
     * @returns {BEM.MODEL[]} вернёт список моделей затронутых баннеров
     * @private
     */
    _resetAssignment: function(bids, newVCardId) {
        var affectedMap = {},
            vcards = [],
            updatePatch = { vcard: null },
            newVCard = newVCardId && BEM.MODEL.getOne({ name: 'm-vcard', id: newVCardId }),
            isLinking = !!newVCard,
            active,
            banners;

        if (isLinking) {
            // для учета не архивных баннеров при привязывании визитки
            active = 0;

            // сбор моделей затронутых визиток
            vcards.push(newVCard);

            updatePatch.vcard = { vcard_id: newVCard.get('vcard_id') };
        }

        bids = [].concat(bids || []);

        banners = bids.map(function(bid) {
            var banner = BEM.MODEL.getOne({ name: 'm-banner', id: bid }),
                vcard = banner.get('vcard'),
                isArchived = banner.get('isArchived'),
                oldVCard,
                vcardId;

            // привязка/отвязка баннера с визиткой/от визитки
            banner.update(updatePatch);

            isLinking && (isArchived || active++);

            if (vcard) {
                vcardId = vcard.vcard_id;

                oldVCard = BEM.MODEL.getOne({ name: 'm-vcard', id: vcardId });

                // обновление информации по счётчикам для отвязываемой от баннера визитки
                this._updateBannersCount(oldVCard, {
                    total: -1,
                    active: isArchived ? 0 : -1
                });

                // сбор моделей затронутых визиток
                // FYI: используем хэш, чтобы не искать по массиву
                if (!affectedMap[vcardId]) {
                    affectedMap[vcardId] = true;

                    vcards.push(oldVCard);
                }
            }

            return banner;
        }, this);

        // обновление информации по счётчикам для привязываемой к баннерам визитки
        isLinking && this._updateBannersCount(newVCard, {
            total: bids.length,
            active: active
        });

        this.vcards.refresh(vcards);

        this.banners.refresh(banners);

        this.banners.unCheckAllSelected();
    },

    /**
     * Обновляет счётчики баннеров у визитки
     * @param {BEM.MODEL} vcardModel модели визитки
     * @param {Object} countObj объект со свойствами поля bannersCount визитки, значениями которого инкрементируются счётчики
     * @param {Number} countObj.total +/- общее количество баннеров у визитки
     * @param {Number} countObj.active +/- количество активных баннеров у визитки
     * @private
     */
    _updateBannersCount: function(vcardModel, countObj) {
        var bannersCount = vcardModel.get('bannersCount');

        vcardModel.set('bannersCount', {
            total: bannersCount.total + countObj.total,
            active: bannersCount.active + countObj.active
        });
    },

    _scrollToVCard: function(scrollable, vcardId) {
        var _this = this,
            elem = this.vcards.elem('item', 'vcard-id', vcardId);

        // если создали новую визитки, то скролимся к ней и подсвечиваем
        elem.length && scrollable
            .animate(
                { scrollTop: elem.position().top, duration: 100 },
                function() { _this.vcards.highlight(vcardId, 3) });
    },

    /**
     * Обработчик открытия/закрытия редактирования/просмотра визитки
     * @param {Object} e
     * @private
     */
    _onFrameOpenOrClose: function(e) {
        var opening = e.type === 'frame-opening',
            scrollable = this.findElem(this.vcards.domElem, 'scrollable-box'),
            scrollTop = scrollable.data('scrollTop') || 0;

        opening && scrollTop &&
            (scrollTop = scrollable.get(0).scrollTop);

        scrollable.data('scrollTop', scrollTop);

        scrollable.get(0).scrollTop = opening ? 0 : scrollTop;

        this
            .toggleMod(scrollable, 'prevent', 'scrolling', opening)
            .toggleMod(
                this.elem('vcard-banners'),
                'no-checkboxes',
                'yes',
                opening && this.vcards.getMod('frame-type') !== 'view'
            );

        !opening && this._editingModel &&
            this.banners.toggleDisabled(this._editingModel, false);
    }

});
