BEM.DOM.decl({ block: 'b-metrika-counters' }, {

    onSetMod: {
        js: function() {
            this._limit = this.params.limit;

            this._input = this.findBlockInside('input');

            this._model = this.params.dataModel ? BEM.MODEL.getOne(this.params.dataModel) : undefined;

            this._input.on('change', function() {
                var errorKey = this._hasValidationError(),
                    hasValue = ((this._model && this._model.get('metrika_counters')) ||
                        this._input.val().trim()) !== '';

                this._counters = this._input.val().trim();

                errorKey ?
                    this._updateErrors(this._getErrorText(errorKey)) :
                    this._updateErrors();

                this.trigger('state', { canSave: hasValue && !errorKey });
            }, this);
        },

        'check-request': function(modName, modVal) {
            if (modVal) {
                // при запросе очищаем сообщения об ошибках
                this._updateErrors();

                this
                    .trigger('check-request', modVal)
                    .trigger('state', { canSave: false });
            } else {
                this
                    .trigger('check-request')
                    .trigger('state', { canSave: !this._hasValidationError() });
            }
        }
    },

    /**
     * @private
     * @type {String}
     */
    _counters: '',

    /**
     * Возвращает список счетчиков
     * @returns {String}
     */
    getCounters: function() {
        return this._counters;
    },

    /**
     * Метод подготовки формы. Должен быть вызван при каждом показе формы
     */
    prepareToShow: function() {
        var counters;
        // очищаем блок ошибок перед началом работы блока
        this._updateErrors();

        if (this._model) {
            counters = this._model.get('metrika_counters');
            counters = this._getCountersString(counters);

            this._input.val(counters);
            this._counters = counters;
        } else if (this.params.counters) {
            counters = this._getCountersString(this.params.counters);
            this._counters = counters;
            this._input.val(counters);
        } else {
            this._input.val('');
        }

        // при неуказанных счётчиках триггерим событие о неготовности сохраняться
        counters || this.trigger('state', { canSave: false });
    },

    /**
     * Осуществляет дополнительную процедуру валидации
     * @returns {Boolean|$.Deferred}
     */
    validate: function() {

        // проверка от умников
        if (this._hasValidationError()) {
            return false;
        }

        // для ДМО кампаний или на странице Настройки пользователя
        if (this.params.needCheck) {
            // проверка прав на счётчики посредством обращения к серверу
            return this._checkUserCounters(this._input.val().trim());
        }

        return true;
    },

    /**
     * Метод, вызываемый, когда будет нажата кнопка ОК
     */
    updateModel: function() {
        if (!this._model) return;

        this._model
            .set('metrika_counters', this._input.val().trim())
            .fix();

        return true;
    },

    /**
     * Получение лимита счетчиков
     * @returns {Number}
     */
    getLimit: function() {
        return this._limit;
    },

    /**
     * Проверяет используьется ли счетчик
     * @return {boolean}
     * @private
     */
    isCountersUsed: function() {
        var usedCounters = this._model.get('counters_in_experiments') || {},
            oldCountersArray = this._model.get('metrika_counters').split(/[,\s]+/g),
            countersArray = this._input.val().trim().split(/[,\s]+/g);

        u._.remove(oldCountersArray, function(id) {
            return countersArray.indexOf(id) !== -1;
        });

        return Object.keys(usedCounters).some(function(counter_id) {
            return oldCountersArray.indexOf(counter_id) !== -1;
        });
    },

    /**
     * Вовзращает строку счетчиков, разделенную запятой И пробелом
     * @param {String} countersStr - исходная строка
     * @returns {String}
     * @private
     */
    _getCountersString: function(countersStr) {
        var countersArray = countersStr.trim() ?
            countersStr.split(/[,\s]+/g) :
            [];

        countersArray = u._.filter(countersArray, function(counter) {
            return counter !== '';
        });

        return countersArray.join(', ');
    },

    /**
     * Осуществляет процедуру валидации введенного значения
     * @returns {String|Null} если есть ошибка, возвращает текстовый код ошибки
     */
    _hasValidationError: function() {
        var counters = this._input.val().trim(),
            countersArray;

        // проверка на некорректные символы
        if (!(/^[0-9\s,]*$/g).test(counters)) {
            return 'invalid';
        }

        countersArray = counters.split(/[,\s]+/g);

        // проверяем количество счётчиков
        if (countersArray.length > this._limit) {
            return 'limit';
        }

        return null;
    },

    /**
     * Возвращает текст ошибки по текстовому коду ошибки
     * @param {String<serverCheck|invalid|length|limit>} key текстовый код ошибки
     * @returns {String}
     * @private
     */
    _getErrorText: function(key) {
        var errors = {
            serverCheck: iget2('b-metrika-counters', 'oshibka-pri-proverke-vozmozhnosti', 'Ошибка при проверке возможности указания счётчика метрики')
        };

        if (this._limit > 1) {
            u._.assign(errors, {
                invalid: iget2('b-metrika-counters', 'neverno-ukazany-dopolnitelnye-schyotchiki', 'Неверно указаны дополнительные счётчики Метрики'),
                limit: iget2('b-metrika-counters', 'mozhno-ukazat-ne-bolee', 'Можно указать не более {foo} дополнительных счётчиков Метрики', {
                    foo: this._limit
                })
            });
        } else {
            u._.assign(errors, {
                invalid: iget2('b-metrika-counters', 'nomer-schyotchika-mozhet-sostoyat', 'Номер счётчика может состоять только из цифр'),
                limit: iget2('b-metrika-counters', 'mozhno-ukazat-tolko-odin', 'Можно указать только один дополнительный счётчик Метрики')
            });
        }

        return errors[key];
    },

    /**
     * Осуществляет процедуру валидации и проверки прав на счётчики для пользователя
     * @param {String} counters
     * @returns {$.Deferred}
     * @private
     */
    _checkUserCounters: function(counters) {
        var deferred = $.Deferred(),
            request = BEM.create('i-request_type_ajax', {
                url: '/registered/main.pl',
                type: 'GET',
                cache: false,
                dataType: 'json',
                callbackCtx: this
            });

        this.setMod('check-request', 'yes');

        request.get(
            { cmd: 'ajaxCheckUserCounters', counter: counters, ulogin: u.consts('ulogin') },
            function(response) {
                if (response.error || !response.result) {
                    this._processCheckUserCountersError(response.error);

                    deferred.reject();
                } else if (response.result) {
                    deferred.resolve(this._processCheckUserCountersResult(response.result,
                        counters.trim() ? counters.split(/[,\s]+/g) : []));
                }

                this.delMod('check-request');
            },
            function() {
                this._processCheckUserCountersError();

                deferred.reject();

                this.delMod('check-request');
            },
            { callbackCtx: this });

        return deferred;
    },

    /**
     * Обработка ошибок при ответе сервера
     * Если ошибка не указана, то будет отображено сообщение об ошибке по умолчанию
     * @param {Object|Error} [error] объект ошибки
     * @private
     */
    _processCheckUserCountersError: function(error) {
        error || (error = { message: this._getErrorText('serverCheck') });

        this._updateErrors(error.message);
    },

    /**
     * Обработка результатов проверки при ответе сервера
     * Если имеются счётчики, прав на которые у пользователя нет, будет выведено сообщение об ошибке прав,
     * Иначе обновит модель кампании информацией о счетчике
     * @param {Object} result хэш вида { Number counter: Boolean hasRules }
     * @param {String[]} counters список сохраняемых счётчиков
     * @returns {Boolean} вернёт true, если есть права, или false, если нет прав хотя бы на один из проверяемых
     *     счётчиков
     * @private
     */
    _processCheckUserCountersResult: function(result, counters) {
        var key = counters.length === 1 ? 'one-' : 'many-',
            labels,
            deniedCounters = Object.keys(result).reduce(function(acc, counter) {
                var available = result[counter].allow;

                available || acc.push(counter);

                return acc;
            }, []);

        if (deniedCounters.length) {
            labels = {
                'many-all': iget2(
                    'b-metrika-counters',
                    'prava-na-ukazannye-schyotchiki',
                    'Права на указанные счётчики Яндекс.Метрики отсутствуют. Если счётчики принадлежат другому логину, пожалуйста, {access}.',
                    {
                        access: function(text) {
                            return {
                                block: 'link',
                                target: 'blank',
                                url: u.getUrl('requestAccessToMetrikaCounters', { ulogin: u.consts('ulogin') }),
                                content: text
                            };
                        }(iget2('b-metrika-counters', 'prava-na-ukazannye-schyotchiki-param-access', 'получите к ним доступ')),

                        counters: deniedCounters.join(', '),
                        context: 'access - получите к ним доступ'
                    }
                ),
                'many-some': iget2(
                    'b-metrika-counters',
                    'prava-na-chast-ukazannyh',
                    'Права на часть указанных счётчиков Яндекс.Метрики отсутствуют. Если счётчики {counters} принадлежат другому логину, пожалуйста, {access}.',
                    {
                        access: function(text) {
                            return {
                                block: 'link',
                                target: 'blank',
                                url: u.getUrl('requestAccessToMetrikaCounters', { ulogin: u.consts('ulogin') }),
                                content: text
                            };
                        }(iget2('b-metrika-counters', 'prava-na-chast-ukazannyh-param-access', 'получите к ним доступ')),

                        counters: deniedCounters.join(', '),
                        context: 'access - получите к ним доступ'
                    }
                ),
                'many-one': iget2(
                    'b-metrika-counters',
                    'prava-na-odin-iz',
                    'Права на один из указанных счётчиков Яндекс.Метрики отсутствуют. Если счётчик {counters} принадлежит другому логину, пожалуйста, {access}.',
                    {
                        access: function(text) {
                            return {
                                block: 'link',
                                target: 'blank',
                                url: u.getUrl('requestAccessToMetrikaCounters', { ulogin: u.consts('ulogin') }),
                                content: text
                            };
                        }(iget2('b-metrika-counters', 'prava-na-odin-iz-param-access', 'получите к нему доступ')),

                        counters: deniedCounters.join(', '),
                        context: 'access - получите к нему доступ'
                    }
                ),
                'one-all': iget2(
                    'b-metrika-counters',
                    'prava-na-ukazannyy-schyotchik',
                    'Права на указанный счётчик Яндекс.Метрики отсутствуют. Если счётчик принадлежит другому логину, пожалуйста, {access}.',
                    {
                        access: function(text) {
                            return {
                                block: 'link',
                                target: 'blank',
                                url: u.getUrl('requestAccessToMetrikaCounters', { ulogin: u.consts('ulogin') }),
                                content: text
                            };
                        }(iget2('b-metrika-counters', 'prava-na-ukazannyy-schyotchik-param-access', 'получите к нему доступ')),

                        counters: deniedCounters.join(', '),
                        context: 'access - получите к нему доступ'
                    }
                )
            };
            key += deniedCounters.length === counters.length ?
                'all' :
                deniedCounters.length === 1 ?
                    'one' :
                    'some';

            this._updateErrors(labels[key]);
        } else {
            var counterId = Object.keys(result)[0];

            counterId && this._updateMetrikaCounterData(result[counterId]);
        }

        return !deniedCounters.length;
    },

    /**
     * Обновляет сообщение об ошибке
     * В зависимости от наличия текста снимается/выставляется модификатор _with-errors_no
     * @param {String|Array} [errorText] текст ошибки или массив с ошибками
     * @private
     */
    _updateErrors: function(errorText) {
        errorText && (errorText = BEMHTML.apply({
            block: 'b-metrika-counters',
            elem: 'error',
            content: errorText
        }));

        BEM.DOM.update(this.elem('errors'), errorText || '');

        this.toggleMod('with-errors', 'no', !errorText);
    },

    /**
     * Обновляет модель метрики в модели кампании данными из ответа аякс проверки
     * @param {Object} data - данные по счетчику
     * @private
     */
    _updateMetrikaCounterData: function(data) {
        this._model && this._model.update({
            metrika: {
                campaign_goals: data.goals
            }
        });
    }

}, {

    live: true

});
