/**
 * Метрика
 * @event b-metrika2#metrika-goal генерируется при достижении цели
 * @event b-metrika2#metrika-param генерируется при отправки параметров визита
 *
 * @fires b-metrika2#metrika-goal
 * @fires b-metrika2#metrika-param
 */
BEM.DOM.decl('b-metrika2', {

    onSetMod: {
        /**
         * Инициализация метрики
         */
        js: function() {
            var params = this.params;
            params.event && this._subscribe(params);
            Array.isArray(params.items) && params.items
                .map(function(item) {
                    if (!item.blockName) {
                        item.blockName = params.blockName;
                    }
                    return item;
                })
                .forEach(this._subscribe.bind(this));

            if (params.immediate) {
                this._onParam({
                    params: params.immediate
                });
            }

            if (params.nativeEvent) {
                this.bindTo(this.domElem, params.nativeEvent, function() {
                    if (params.params) {
                        this._onParam({
                            params: params.params
                        });
                    }
                    if (params.goal) {
                        this._onGoal(params.goal);
                    }
                }.bind(this));
            }
        }
    },

    /**
     * Подписка на события блока с метрикой
     * @param {{ event: String, target: String|Array, targetParams: Object, counter: String|Number|Array, params: *, [blockName]: String}} prop
     * @private
     */
    _subscribe: function(prop) {
        var block = BEM.blocks[prop.blockName];

        if (!block) {
            throw new Error('Invalid metrika2 params: ' + JSON.stringify(this.params));
        }

        prop.target && block.on(this.domElem, prop.event, this._onGoal.bind(
            this, {
                target: prop.target,
                id: prop.counter,
                params: prop.targetParams
            })
        );

        prop.params && block.on(this.domElem, prop.event, this._onParam.bind(
            this, {
                id: prop.counter,
                params: prop.params
            })
        );
    },

    /**
     * Обработчик события достижения цели
     * @param {{ target: String|Array, counter: String|Array }} prop
     * @private
     */
    _onGoal: function(prop) {
        this.__self.reachGoal(prop);
        this.trigger('metrika-goal', prop);
    },

    /**
     * Обработчик события отправки параметров визита
     * @param {Object} params
     * @private
     */
    _onParam: function(params) {
        this.__self.params(params);
        this.trigger('metrika-param', params);
    }
}, {

    /**
     * @type {Number} время(ms), по истечение которого считается, что счетчик метрики уже не будет создан
     */
    RESET_TIMEOUT: 10000,

    /**
     * @type {String} префикс с помощью которого формируется имя счетчика в window
     */
    _counterPrefix: 'yaCounter',

    /**
     * @type {Number} id счетчика метрики по умолчанию
     */
    _defaultCounterId: 34,

    /**
     * Возвращает id счетчика метрики, используемого по умолчанию
     * @returns {Number}
     */
    getDefaultCounterId: function() {
        return this._defaultCounterId;
    },

    /**
     * Возвращает имя счетчика метрики в глобальной области по его id
     * @sync
     * @param {Number} id
     * @returns {String}
     */
    getGlobalName: function(id) {
        return this._counterPrefix + id;
    },

    /**
     * Достижение цели
     * @async
     * @param {String|{id: Number, target: String, callback: Function, ctx: Object params: Object}} goal
     * @desc сигнатура соответствует документации метрики - только передается в объекте
     * @link https://yandex.ru/support/metrika/objects/reachgoal.xml
     * @returns {Object}
     */
    reachGoal: function(goal) {
        var id,
            target,
            callback,
            ctx,
            params;

        switch (typeof goal) {
            case 'string':
                id = this.getDefaultCounterId();
                target = goal;
                break;
            default:
                id = goal.id || this.getDefaultCounterId();
                target = goal.target;
                callback = goal.callback;
                ctx = goal.ctx;
                params = goal.params;
        }

        this._getCounter(id)
            .done(function(counter) {
                counter.reachGoal(target, params, callback, ctx);
                params && counter.params(params);
            });

        return this;
    },

    /**
     * Передача параметров визита
     * @data {{ id: String|Number, params: Object}} data данные для метрики
     * @returns {Object}
     */
    params: function(data) {
        this._getCounter(data.id || this.getDefaultCounterId())
            .done(function(counter) {
                counter.params(data.params);
            });
        return this;
    },

    /**
     * Устанавливает id клиента в метрике
     * https://yandex.ru/support/metrika/objects/set-user-id.xml
     * @param {String|Number|{id: Number, userID: String|Number}} data
     * @returns {Object}
     */
    setUserID: function(data) {
        var id,
            userID;

        switch (typeof data) {
            case 'string':
                id = this.getDefaultCounterId();
                userID = data;
                break;
            default:
                id = data.id || this.getDefaultCounterId();
                userID = data.userID;
        }

        this._getCounter(id)
            .done(function(counter) {
                counter.setUserID(userID);
            });

        return this;
    },

    getCounter: function(id) {
        return this._getCounter(id || this.getDefaultCounterId());
    },

    /**
     * Получить объект счетчика из глобальной области по его id
     * @param {Number} id счетчика
     * @returns {Promise.<Ya.Metrika>}
     */
    _getCounter: function(id) {
        var globalName = this.getGlobalName(id),
            defer = $.Deferred();

        if (this._validateCounter(globalName)) {
            defer.resolve(window[globalName]);
        } else {
            setTimeout(
                defer.reject.bind(defer, 'problem with id counter: ' + id),
                this.RESET_TIMEOUT
            );

            $(document).on(
                'yacounter' + id + 'inited',
                function() {
                    if (this._validateCounter(globalName)) {
                        defer.resolve(window[globalName]);
                    } else {
                        defer.reject(globalName)
                    }
                }.bind(this)
            );
        }

        return defer.promise();
    },

    /**
     * Валидация объекта счетчика
     * @param {String} globalName
     * @returns {Boolean}
     * @private
     */
    _validateCounter: function(globalName) {
        var counter = window[globalName],
            isYaMetrika = window.Ya && window.Ya.Metrika && (counter instanceof window.Ya.Metrika),
            isYaMetrika2 = window.Ya && window.Ya.Metrika2 && (counter instanceof window.Ya.Metrika2);

        return isYaMetrika || isYaMetrika2;
    }

});
