/**
 * @param {Object} this.params.ctxData Объект с входными данными блока
 * @param {{cid: Number, campName: String, adgroupId: Number, groupName: String, bid: Number, text: String}[]} this.params.ctxData.params Добавляемые минус-фразы
 * @param {Object.<string, number>} this.params.ctxData.groupsSymbolsCounter Общая длина символов текущих минус-слов в группах
 * @param {Object.<string, number>} this.params.ctxData.campaignsSymbolsCounter Общая длина символов текущих минус-слов в кампаниях
 * @param {Object} this.params.scrollableElemMix BEMJSON, который миксуется к элементу со полосой прокрутки
 */
BEM.DOM.decl('b-minus-phrases-manager', {

    onSetMod: {

        js: function() {
            var ctxData = this.params.ctxData;

            this.setMod('progress', 'yes');

            this._subMan = BEM.create('i-subscription-manager');

            this._vm = BEM.MODEL.create('b-minus-phrases-manager', {
                phrases: this._preparePhrases(ctxData.phrases),
                groupsSymbolsCounter: ctxData.groupsSymbolsCounter,
                campaignsSymbolsCounter: ctxData.campaignsSymbolsCounter
            });

            this._vmEmitter = this._subMan.wrap(this._vm);

            this._vmEmitter
                .on('phrases', 'remove', this._onPhraseItemVmRemove, this)
                .on('phrases', 'change', this._onPhraseChange, this);

            // cyn: onFirst почему-то не работает, поэтому без `i-subscription-manager` и отписываемся в обработчике
            this._vm.on('symbols-counters-calculated', this._onVmInit, this);

            this._vm.init();

            this._render();
        },

        progress: function(modName, modVal) {
            this.trigger(modVal ? 'progress-start' : 'progress-end');
        }
    },

    onElemSetMod: {

        error: {

            showed: function(elem, modName, modVal) {
                var inputInside = this.findBlockInside(elem, 'input');

                if (inputInside) {
                    inputInside.setMod('highlight-border', modVal ? 'red' : '');
                }
            }

        }

    },

    /**
     * Сообщает были ли изменения в блоке
     * @returns {Boolean}
     */
    isChanged: function() {
        return this._vm.isChanged();
    },

    /**
     * Сохраняет фразы, обрабатывает ответ
     * @returns {$.Deferred}
     */
    save: function() {
        var deferred = $.Deferred();

        if (this.findElem('error', 'showed', 'yes').length) {
            this._showErrors();

            deferred.reject('validationErrors');
        } else {
            this.setMod('progress', 'yes');

            this._save().then(
                function(data) {
                    this.delMod('progress');

                    deferred.resolve(data);
                }.bind(this),
                function(error) {
                    this.delMod('progress');

                    deferred.reject(error);
                }.bind(this)
            );
        }

        return deferred.promise();
    },

    /**
     * Обработчик инициализации view-модели
     * @private
     */
    _onVmInit: function() {
        this._vm.un('symbols-counters-calculated', this._onVmInit, this);

        this._vm.fix();

        this.delMod('progress');
    },

    /**
     * Отрисовка блока
     * @private
     */
    _render: function() {
        var data = this._vm.toJSON(),
            blockName = this.__self.getName(),
            mods = {
                'stat-type': this.getMod('stat-type')
            },
            composite = $(BEMHTML.apply([
                {
                    block: blockName,
                    mods: mods,
                    elem: 'header',
                    itemData: {
                        id: this._vm.id,
                        target: data.target
                    }
                },
                {
                    block: 'composite',
                    mix: [
                        { block: blockName, elem: 'phrases-list' },
                        this.params.scrollableElemMix
                    ],
                    itemView: {
                        block: blockName,
                        mods: mods,
                        elem: 'phrase-item'
                    },
                    collection: data.phrases
                }
            ]));

        BEM.DOM.update(this.domElem, composite);

        this._composite = this.findBlockInside('composite');
    },

    /**
     * Отображает подсказку с ошибкой
     * @param {jQuery} errorDomElem
     * @private
     */
    _showErrorTooltip: function(errorDomElem) {
        this._tipman || (this._tipman = BEM.create('tipman', {
            tipMods: { theme: 'normal' },
            popupDirections: ['bottom', 'top'],
            delay: 50
        }));

        this._tipman.show({
            owner: errorDomElem,
            content: function() {
                var elemParams = this.elemParams(errorDomElem),
                    phraseItemVM = this._getPhraseItemVM(elemParams.itemId);

                return BEMHTML.apply({
                    block: this.__self.getName(),
                    elem: 'tooltip-message',
                    content: phraseItemVM.get('error')
                });
            }.bind(this)
        });
    },

    /**
     * Прячет подсказку
     * @private
     */
    _hideErrorTooltip: function() {
        this._tipman && this._tipman.hide();
    },

    /**
     * Обработчик наведения курсора мышки на элемент ошибки
     * @param {jQuery.Event} e
     * @private
     */
    _onMouseOverOutError: function(e) {
        if (e.type === 'pointerover') {
            this._showErrorTooltip(e.data.domElem);
        } else {
            this._hideErrorTooltip();
        }
    },

    /**
     * Возвращает vm фразы по её id
     * @param {String} id view-модели фразы
     * @return {BEM.MODEL}
     * @private
     */
    _getPhraseItemVM: function(id) {
        return this._vm.get('phrases').getById(id);
    },

    /**
     * Удаляет vm-фразы
     * @param {String} id view-модели фразы
     * @private
     */
    _removePhraseItem: function(id) {
        this._getPhraseItemVM(id).destruct();
    },

    /**
     * Обработчик удаления view-модели фразы
     * @param {jQuery.Event} e
     * @param {Object} data
     * @private
     */
    _onPhraseItemVmRemove: function(e, data) {
        this._composite.remove(data.model.id);
    },

    /**
     * Обработчик клика по корзине
     * @param {jQuery.Event} e
     * @param {Object} data
     * @private
     */
    _onActionRequested: function(e, data) {
        if (data.action === 'remove') {
            this._removePhraseItem(data.data.itemId);
        }
    },

    _onPhraseChange: function(e, data) {
        switch (data.innerField) {
            case 'text':
            case 'target':
                return this._validatePhrase(data.model);
        }
    },

    /**
     * Записывает ошибки пришедшие с сервера в модель фразы
     * @param {Object} tree - смотри https://wiki.yandex-team.ru/Direkt/Development/Java/web-api/front-integration/
     * @param {Object} clientData
     * @returns {BEM}
     * @private
     */
    _setServerErrors: function(tree, clientData) {
        tree.errors.forEach(function(error) {
            var phrasesVM = u._.get(clientData, error.path);

            // если phrasesVM массив, то это ошибки на кампанию/группу
            // иначе phrasesVM модель фразы
            if (!u._.isArray(phrasesVM)) {
                phrasesVM = [phrasesVM];
            }

            phrasesVM.forEach(function(phraseItemVM) {
                if (phraseItemVM instanceof BEM.MODEL) {
                    phraseItemVM.set('errorFromServer', error.description);
                }
            });
        });

        return this;
    },

    _validatePhrase: function(phraseItemVM) {
        var errors = phraseItemVM.validate('text').errors,
            errorFromServer = '';

        if (errors) {
            errorFromServer = errors.map(u._.property('text')).join('\n');
        }

        phraseItemVM.set('errorFromServer', errorFromServer);

        return this;
    },

    /**
     * Возвращает первую ошибку, которая попадает в видимую область или false
     * @param {jQuery} errors
     * @param {jQuery} viewPort
     * @returns {jQuery|boolean}
     * @private
     */
    _getfirstErrorInViewPort: function(errors, viewPort) {
        var errorsLength = errors.length,
            i;

        for (i = 0; i < errorsLength; i++) {
            if (u.inviewport(errors.eq(i), viewPort)) {
                return errors.eq(i);
            }
        }

        return false;
    },

    /**
     * Отображает ошибки, при необходимости подкручивает полосу прокрутки к нужному месту
     * @private
     */
    _showErrors: function() {
        var errors = this.findElem('error', 'showed', 'yes'),
            viewPort = this.elem('phrases-list'),
            firstError;

        if (errors.length) {
            firstError = this._getfirstErrorInViewPort(errors, viewPort);

            if (firstError) {
                this._showErrorTooltip(firstError);
            } else {
                firstError = errors.eq(0);

                u.scrollNodeTo(
                    firstError,
                    viewPort,
                    { over: 14 },
                    function() {
                        this._showErrorTooltip(firstError);
                    }.bind(this)
                );
            }
        }
    },

    /**
     * Группирует фразы по идентификатору группы/кампании
     * @param {'group'|'campaign'} target
     * @returns {{id: String, phrases: {}[]}[]}
     * @private
     */
    _groupPhrasesByTargetId: function(target) {
        var phrasesModelByTarget = this._vm.get('phrases').where({ target: target }),
            uidFieldName = target === 'group' ? 'adgroupId' : 'cid',
            groupedByIdHash = u._.groupBy(phrasesModelByTarget, function(model) {
                return model.get(uidFieldName);
            });

        return u._.map(groupedByIdHash, function(phrases, id) {
            return {
                id: id,
                phrases: phrases
            };
        });
    },

    /**
     * Преобразует данные для сохранения на сервере
     * @param {{id: String, phrases: {}[]}[]} phrases
     * @private
     */
    _transformToServerData: function(phrases) {
        return phrases.map(function(phraseItem) {
            return {
                id: phraseItem.id,
                minus_keywords: phraseItem.phrases.map(function(vm) {
                    return u.minusWords.preprocessingStatPhrases(vm.get('text'));
                }),
                // порядок соответствует minus_keywords
                report_row_hash: phraseItem.phrases.map(function(vm) {
                    return vm.get('report_row_hash');
                })
            };
        })
    },

    /**
     * Преобразует данные для нахождения модели фразы по результатам валидации сервера
     * @param {{id: String, phrases: {}[]}[]} phrases
     * @private
     */
    _transformToClientData: function(phrases) {
        return phrases.map(function(phraseItem) {
            return {
                id: phraseItem.id,
                minus_keywords: phraseItem.phrases
            };
        })
    },

    /**
     * Формирует данные и делает запросы на валидацию и сохранения
     * @private
     */
    _save: function() {
        var deferred = $.Deferred(),
            groupedPhrases = {
                groups: this._groupPhrasesByTargetId('group'),
                campaigns: this._groupPhrasesByTargetId('campaign')
            },
            serverData = {},
            clientData = {};

        if (groupedPhrases.groups.length) {
            serverData.ad_group_minus_keywords = this._transformToServerData(groupedPhrases.groups);
            clientData.ad_group_minus_keywords = this._transformToClientData(groupedPhrases.groups);
        }

        if (groupedPhrases.campaigns.length) {
            serverData.campaign_minus_keywords = this._transformToServerData(groupedPhrases.campaigns);
            clientData.campaign_minus_keywords = this._transformToClientData(groupedPhrases.campaigns);
        }

        this._checkKeywordInclusion(serverData)
            .then(function(response) {
                if (response.success) {
                    new Promise(function(resolve, reject) {
                        if (response.result.length) {
                            BEM.blocks['b-confirm'].open({
                                message: {
                                    block: 'b-minus-phrases-manager',
                                    elem: 'confirm-message',
                                    content: iget2(
                                        'b-minus-phrases-manager',
                                        'sleduyushchie-minus-frazy-peresekayutsya',
                                        'Следующие минус-фразы пересекаются с ключевыми фразами: {foo}. Данные минус-фразы не будут учитываться при показах по соответствующим ключевым фразам. Нажмите "ОК", чтобы сохранить минус-фразы.',
                                        {
                                            foo: response.result.map(function(inclusion) {
                                                var phrasesVM = u._.get(clientData, inclusion.path);

                                                return phrasesVM && phrasesVM.get('text');
                                            }).join(', ')
                                        }
                                    )
                                },
                                onYes: resolve,
                                onNo: reject,
                                fromPopup: this.findBlockOutside('popup'),
                                textYes: iget2('b-minus-phrases-manager', 'ok-123', 'ОК'),
                                textNo: iget2('b-minus-phrases-manager', 'otmena', 'Отмена')
                            });
                        } else {
                            resolve();
                        }
                    }.bind(this))
                        .then(function() {
                            this._addMinusKeywords(serverData)
                                .then(function(data) {
                                    if (data.success === false) {
                                        // ошибки добавления
                                        this._setServerErrors(data.validation_result, clientData);

                                        this._showErrors();

                                        deferred.reject('validationErrors');
                                    } else {
                                        deferred.resolve(this._exportData(data.result));
                                    }
                                }.bind(this))
                                .fail(function(error) {
                                    deferred.reject(error);
                                });
                        }.bind(this))
                        .catch(function() {
                            deferred.reject('canceledByUser')
                        }.bind(this));
                } else {
                    this._setServerErrors(response.validation_result, clientData);
                    this._showErrors();

                    deferred.reject('validationErrors');
                }
            }.bind(this))
            .fail(function(error) {
                deferred.reject(error);
            }.bind(this));

        return deferred.promise();
    },

    /**
     * Возвращает количество добавленных минус-фраз и обновленные значения счетчиков для групп/кампаний
     * @param {Object} result
     * @returns {{newMinusWordsLength: {}, addedCount: Number}}
     * @private
     */
    _exportData: function(result) {
        var newMinusWordsLength = {},
            addedCount = 0;

        if (result.ad_group_minus_keywords) {
            newMinusWordsLength.groups = this._getNewMinusWordsLength(result.ad_group_minus_keywords);
            addedCount += this._caclAddedMinusWords(result.ad_group_minus_keywords);
        }

        if (result.campaign_minus_keywords) {
            newMinusWordsLength.campaigns = this._getNewMinusWordsLength(result.campaign_minus_keywords);
            addedCount += this._caclAddedMinusWords(result.campaign_minus_keywords);
        }

        return {
            newMinusWordsLength: newMinusWordsLength,
            addedCount: addedCount
        };
    },

    /**
     * Возвращает актуальные числа счетчиков
     * @param {Array} addedItems
     * @private
     */
    _getNewMinusWordsLength: function(addedItems) {
        return addedItems.map(function(item) {
            return {
                id: item.id,
                newLength: item.current_length
            }
        });
    },

    /**
     * Возвращает количество добавленных фраз
     * @param {Number} addedItems
     * @private
     */
    _caclAddedMinusWords: function(addedItems) {
        return addedItems.reduce(function(result, item) {
            return result += item.added_count || 0;
        }, 0);
    },

    /**
     * Отправляет запрос на добавление фраз
     * @param {Object} phrasesByTarget добавляемые минус-фразы
     * @returns {$.Deferred}
     * @private
     */
    _addMinusKeywords: function(phrasesByTarget) {
        return this._doRequest('add', phrasesByTarget);
    },

    /**
     * Отправляет запрос на вхождение добавляемых минус-фраз в ключевые фразы
     * @param {Object} phrasesByTarget добавляемые минус-фразы
     * @returns {$.Deferred}
     * @private
     */
    _checkKeywordInclusion: function(phrasesByTarget) {
        return this._doRequest('check_keyword_inclusion', phrasesByTarget)
            //cyn@TODO: нужно будет оторвать когда бэкенд изменит ручку, см. DIRECT-66334
            .then(function(response) {

                if (response.success) {
                    response.result = [];
                } else {
                    var keywordInclusionErrors = u._.get(response, 'validation_result.errors', [])
                        .filter(function(error) { return error.code === 5161 });

                    if (keywordInclusionErrors.length) {
                        response = {
                            success: true,
                            result: keywordInclusionErrors
                        };
                    }
                }

                return response;
            }.bind(this));
    },

    /**
     * Возвращает инстанс запроса
     * @returns {BEM}
     * @private
     */
    _doRequest: function(handleName, phrasesByTarget) {
        var deferred = $.Deferred(),
            url = '/web-api/minus_keyword/' + handleName;

        if (u.consts('ulogin')) {
            url += '?ulogin=' + u.consts('ulogin');
        }

        this._requestInstance = BEM.create('i-request_type_ajax', {
            url: url,
            type: 'POST',
            cache: false,
            dataType: 'json',
            headers: {
                'X-CSRF-TOKEN': u.consts('csrf_token'),
                'X-Detected-Locale': u.consts('lang')
            },
            contentType: 'application/json; charset=utf-8',
            paramsToSettings: ['contentType', 'headers'],
            timeout: 1200000,
            callbackCtx: this
        });

        this._requestInstance.get(
            JSON.stringify(phrasesByTarget),
            function(result) {
                deferred.resolve(result);
            },
            function(error) {
                deferred.reject(error)
            });

        return deferred.promise();
    },

    /**
     * Предобработка фраз: сортировка, установка флага `prevPhraseHaveAnotherCid`
     * @param {Object[]} phrases
     * @returns {Object[]}
     * @private
     */
    _preparePhrases: function(phrases) {
        var uniqCidsQueue = [],
            phrasesGroupedByCid = u._.groupBy(phrases, function(phrase) {
                var cid = phrase.cid;

                if (uniqCidsQueue.indexOf(cid) === -1) {
                    uniqCidsQueue.push(cid);
                }

                return phrase.cid;
            });

        return uniqCidsQueue.reduce(function(result, cid) {
            var phrasesSameCid = u._.sortBy(phrasesGroupedByCid[cid], 'adgroupId');

            phrasesSameCid[0].prevPhraseHaveAnotherCid = true;

            return result.concat(phrasesSameCid);
        }, []);
    },

    destruct: function() {
        this._requestInstance && this._requestInstance.abort();
        this._subMan.dispose();
        this._subMan.destruct();
        this._tipman && this._tipman.destruct();

        this._vm.un('symbols-counters-calculated', this._onVmInit, this);
        this._vm.destruct();

        return this.__base.apply(this, arguments);
    }

}, {

    live: function() {
        this
            .liveBindTo(
                { elem: 'error', modName: 'showed', modVal: 'yes' },
                'pointerover pointerout',
                function(e, data) {
                    this._onMouseOverOutError(e, data);
                }
            )
            .liveInitOnBlockInsideEvent('actionRequested', 'b-control', function(e, data) {
                this._onActionRequested(e, data)
            });
    }

});
