BEM.MODEL.decl('b-minus-phrases-manager', {

    phrases: {
        type: 'models-list',
        modelName: 'b-minus-phrases-manager__phrase-item',
        validation: {
            rules: {
                deep: true
            }
        }
    },

    campaignsSymbolsCounter: {
        internal: true,
        type: 'object'
    },

    symbolsCounter: {
        internal: true,
        type: 'object',
        dependsFrom: ['groupsSymbolsCounter', 'campaignsSymbolsCounter'],
        calculate: function(data) {
            var campaignsSymbolsCounter = data.campaignsSymbolsCounter,
                groupsSymbolsCounter = data.groupsSymbolsCounter,
                symbolsCounter = {};

            if (groupsSymbolsCounter && campaignsSymbolsCounter) {
                Object.keys(groupsSymbolsCounter).forEach(function(groupId) {
                    symbolsCounter['g' + groupId] = groupsSymbolsCounter[groupId];
                });

                Object.keys(campaignsSymbolsCounter).forEach(function(campaignId) {
                    symbolsCounter['c' + campaignId] = campaignsSymbolsCounter[campaignId];
                });
            }

            return symbolsCounter;
        }
    },

    target: {
        type: 'enum',
        enum: ['undefined', 'group', 'campaign'],
        default: 'group'
    },

    groupsSymbolsCounter: {
        internal: true,
        type: 'object'
    }

}, {

    /**
     * Инициализация модели
     */
    init: function() {
        var targetIds = [];

        this
            .on('target', 'change', this._onHeaderTagetChange, this)
            .on('phrases', 'remove', this._onPhraseRemove, this);

        this.get('phrases').forEach(function(phraseitem) {
            phraseitem
                .on('text', 'change', this._onPhraseTextChange, this)
                .on('targetId', 'change', this._onPhraseTargetChange, this);

            targetIds.push(phraseitem.get('targetId'));
        }, this);

        this._addToQueueToCalcCounters(targetIds);
    },

    /**
     * Обработчик изменения места добавления всех минус-фраз
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {'campaign'|'group'} data.value
     * @private
     */
    _onHeaderTagetChange: function(e, data) {
        var target = data.value;

        if (target !== 'undefined') {
            this.get('phrases').forEach(function(phraseItem) {
                phraseItem.set('target', target, { source: 'header' });
            });
        }
    },

    /**
     * Обработчик изменения текста фразы
     * @param {jQuery.Event} e
     * @private
     */
    _onPhraseTextChange: function(e) {
        this._addToQueueToCalcCounters([e.target.model.get('targetId')]);
    },

    /**
     /**
     * Обработчик изменения места добавления конкретной минус-фразы
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {'campaign'|'group'} data.value
     * @param {'header'|undefined} data.source Источник изменения
     * @private
     */
    _onPhraseTargetChange: function(e, data) {
        var phraseModel = e.target.model,
            source = data && data.source;

        this._addToQueueToCalcCounters([phraseModel.get('targetId'), phraseModel.get('prevTargetId')]);

        // Игнорируем изменения из группового контрола
        if (source !== 'header') {
            this._calcHeaderTarget();
        }
    },

    /**
     * Устанавливает значение для группового места добавления минус-фраз
     * @private
     */
    _calcHeaderTarget: function() {
        var phraseItems = this.get('phrases', 'raw'),
            phraseItemsCount = phraseItems.length,
            firstPhraseItem = phraseItems[0],
            newHeaderTarget = firstPhraseItem ? firstPhraseItem.get('target') : 'undefined',
            i;

        for (i = 1; i < phraseItemsCount; i++) {
            if (phraseItems[i].get('target') !== newHeaderTarget) {
                newHeaderTarget = 'undefined';
                break;
            }
        }

        this.set('target', newHeaderTarget);
    },

    /**
     * Обработчик удаления фразы
     * @param {jQuery.Event} e
     * @param {Object} data
     * @private
     */
    _onPhraseRemove: function(e, data) {
        if (data.model.get('prevPhraseHaveAnotherCid')) {
            var nextPhraseSameCid = this.get('phrases').where({ cid: data.model.get('cid') })[0];

            if (nextPhraseSameCid) {
                nextPhraseSameCid.set('prevPhraseHaveAnotherCid', true);
            }
        }
        this._addToQueueToCalcCounters([data.model.get('targetId')]);
    },

    /**
     * Добавляет фразы в очередь на пересчет свободных мест в группах/кампаниях
     * @param {String[]} targetIds
     * @private
     */
    _addToQueueToCalcCounters: function(targetIds) {
        this._timer && clearInterval(this._timer);

        this._queueTargetIds = (this._queueTargetIds || []).concat(targetIds);

        this._timer = setTimeout(function() {
            this._calcOverflowSymbolsCount(u._.uniq(this._queueTargetIds));
            this._queueTargetIds = [];
        }.bind(this), 500);
    },

    /**
     * Подсчитывает количество свободных мест в группах/кампаниях
     * @param {String[]} targetIds
     * @private
     */
    _calcOverflowSymbolsCount: function(targetIds) {
        targetIds.forEach(function(targetId) {
            var phrasesListByTargetId = this.get('phrases').where({ targetId: targetId });

            if (phrasesListByTargetId.length) {
                var addedSymbolsCount = phrasesListByTargetId.reduce(function(res, phraseItem) {
                        return res += u['b-minus-words'].countLength(phraseItem.get('text'));
                    }, 0),
                    currentSymbolsCount = this.get('symbolsCounter')[targetId],
                    maxSymbolsCount = phrasesListByTargetId[0].get('target') === 'group' ?
                        u.consts('GROUP_MINUS_WORDS_LIMIT') :
                        u.consts('CAMPAIGN_MINUS_WORDS_LIMIT'),
                    overflowSymbolsCount = currentSymbolsCount + addedSymbolsCount - maxSymbolsCount;

                phrasesListByTargetId.forEach(function(phraseItem) {
                    phraseItem.set('overflowSymbolsCount', overflowSymbolsCount);
                });
            }
        }, this);

        this.trigger('symbols-counters-calculated');
    }

});
