BEM.MODEL.decl({ model: 'b-mobile-content-tracking-href', baseModel: 'vm-sync-dm2' }, {

    _dmDecl: {
        type: 'object',
        internal: true,
        calculate: function() {
            return {
                banner: {
                    name: 'dm-mobile-content-banner',
                    id: this.id
                }
            };
        }
    },

    // Протокол
    protocol: {
        type: 'enum',
        enum: ['http://', 'https://'],
        default: 'https://',
        calculate: function(href) {
            return u.getUrlProtocol(href);
        },
        dependsFrom: ['href']
    },

    // Часть ссылки без протокола и с параметрами
    href: {
        type: 'string',
        preprocess: function(val) {
            if ((val || '').length > 0) {
                val = val.replace(/[\s\n]+/g,'');
            }

            return val;
        }
    },

    // Доп параметры трекинговой ссылки для баннера
    queryParams: {
        type: 'string',
        default: '',
        preprocess: function(val) {
            var mobileApp = this.model.get('mobileAppTrackerHref');

            val = val || '';
            val = val.replace(/(^[&=@+/,;:$]+)|([&=@+/,;:$]+$)|[ ]/g, '');
            val = val.replace(/[<>]/g, u.escapeHTMLSafeReplacer);
            val = u._.uniq(val.split('&')).join('&');

            var queryDiff = u.getDiffQueryParams('?' + val, mobileApp);
            return u.queryParamsToString(queryDiff);

        },
        dependsFrom: ['mobileAppTrackerHref']
    },

    queryParamsIsEmpty: {
        type: 'boolean',
        default: true,
        calculate: function(queryParams) {
            return (queryParams && queryParams.length === 0);
        },
        dependsFrom: ['queryParams']
    },

    mobileAppTrackerHref: {
        type: 'string',
        default: ''
    },

    // Количество доступных символов
    hrefLengthLeft: {
        type: 'number',
        precision: 0,
        calculate: function(href) {
            return u.consts('MAX_URL_LENGTH') - href.length;
        },
        dependsFrom: ['href']
    },

    // Состояние переполнения доступных символов
    lengthExceeded: {
        type: 'boolean',
        calculate: function(hrefLengthLeft) {
            return hrefLengthLeft < 0;
        },
        dependsFrom: ['hrefLengthLeft']
    },

}, {
    init: function() {
        var dm = this.getDM()['banner'];

        this._checkTrackingHrefDebounced = $.debounce(this._checkTrackingHref, 300, this);

        this.on('href', 'change', function(e, data) {
            dm.set('href', data.value, { fromVM: true });
            this._checkTrackingHrefDebounced(data.value);
        }, this);

        this.on('queryParams', 'change', function(e, data) {
            this.set('href', u._.compact([this.get('mobileAppTrackerHref'), data.value]).join('&'));
        }, this);

        // Выполняем проверку по загрузке страницы, для того чтобы отобразились предупреждения
        this._checkTrackingHrefDebounced(this.get('href'));
    },

    /**
     * Проверяет трекинговую ссылку на принадлежность к поддерживаемым нами трекерам через ajax
     * если проверка не успеет выполниться - отработает серверная валидация
     * @param {String} href
     * @private
     */
    _checkTrackingHref: function(href) {
        var protocol = u.getUrlProtocol(href),
            dm = this.getDM()['banner'],
            isHrefEqualsMobileAppHref = u.isEqualTrackerHref(href, this.get('mobileAppTrackerHref'));

        // отсутствие трекинговой ссылки - корректно
        if (!href.length || isHrefEqualsMobileAppHref) {
            dm.set('approvedTrackingSystem', true);

            this._reportHrefProblems(href);
        } else if (u.isUrl(href)) {
            u.urlCheckTracking({
                url: href,
                protocol: protocol,
                callback: function(data) {
                    var approvedTrackingSystem = !!data.is_valid_tracking_href;

                    dm.set('approvedTrackingSystem', approvedTrackingSystem);
                    dm.set('approvedTrackingSystemErrorType', data.error_type || '');

                    if (data.code === 1) {
                        this._reportHrefProblems(href);
                    } else {
                        this._reportHrefProblems(href, approvedTrackingSystem ?
                            data.text :
                            [
                                data.text,
                                ' ',
                                {
                                    block: 'b-help-link',
                                    mods: { type: 'modal', margin: 'no' },
                                    stretched: true,
                                    url: u.getHelpUrl('mobile-apps-tracking-url'),
                                    text: iget2('b-mobile-content-tracking-href', 'razreshennye-trekingovye-sistemy', 'Разрешенные трекинговые системы.')
                                }
                            ]);
                    }
                }.bind(this)
            });
        } else {
            this._reportHrefProblems(href, 'hrefFormat');
        }
    },

    /**
     * Сообщает о проблемах в трекинговой ссылке
     * @param {String} href ссылка
     * @param {String} [error] Код или текст ошибки в урле
     * @private
     */
    _reportHrefProblems: function(href, error) {
        // Событие `url-check-error` на модели баннера слушает блок `b-banner-preview2-warnings`
        this.getDM()['banner'].trigger('url-check-error', {
            messages: [
                error,
                // DIRECT-47929 Наличие шаблона не блокирует сохранение, но он будет проигнорирован
                u.hasTemplateSymbols(href) ? 'trackingHrefTemplate' : false
            ].filter(Boolean)
        });
    }

});
