BEM.DOM.decl('b-modal-popup-opener', {

    onSetMod: {
        js: function() {
            this.__self._initCommonEvents();

            this.getEventBus().on('need-close', this.hide, this);
        }
    },

    /**
     * Закрывает модальный попап или открытое окно
     */
    hide: function() {
        var win = this._getWindow(),
            popup = this.__self._popup;

        win && win.close();

        if (popup && popup.domElem) {
            popup.hide();
            BEM.DOM.destruct(popup.domElem);
        }

        BEM.channel('direct-tooltips').un('load', this._fitHeight);
    },

    /**
     * Показывает модальный попап.
     * Для открытия страницы через POST или GET запрос или просто для задания другого url
     * можно воспользоваться параметрами
     * @param {String|Object} [urlOrParams] url или параметры формы для POST|GET запроса
     * @param {String} [urlOrParams.action] url формы
     * @param {String} [urlOrParams.method] метод отправки формы
     * @param {Object[]} [urlOrParams.data] опционально список с набором полей (объекты {name, value}), отправляемых на сервер
     */
    show: function(urlOrParams) {
        var popup,
            params,
            url = this.getUrl();

        if (typeof urlOrParams === 'string') {
            url = urlOrParams;
        } else if (urlOrParams && urlOrParams.action && urlOrParams.method) {
            params = urlOrParams;
        }

        this.hide();

        if (this.params.useNativeWindow) {
            this._openWindow(params ? '' : url);
        } else {
            popup = this._getPopup(params ? '' : url);

            popup.show();

            // при открытии в новом окне, следует убрать попап
            // используется pointerclick для предотвращения блокирования окна
            this.params.noNewWindow ||
                popup.bindTo(popup.findElem('target-blank'), 'pointerclick', this.changeThis(function() {
                    // iframe и открываемое новое окно имеют одно и то же имя
                    // надо удалять содержимое попапа с iframe, чтобы при сабмите формы её отправка шла через нужный фрэйм

                    popup.hide().setContent('');

                    this._openWindow(params ? '' : url);

                    params &&
                        this.__self.submitForm(this._getEventBusName(), params);
                }, this));

            this.__self._fitPopupSize();

            popup.repaint();

            popup.findBlockInside('b-iframe').bindTo('load', this._onIframeLoad.bind(this));
        }

        params &&
            this.__self.submitForm(this._getEventBusName(), params);
    },

    /**
      *
      * @fires iframeLoaded
      * @private
      */
    _onIframeLoad: function() {
        this.trigger('iframe:loaded');
    },

    /**
     * Возвращает шину событий, инициируемых на загружаемых страницах
     * @return {BEM}
     */
    getEventBus: function() {
        return BEM.blocks['i-event-bus'].getEventBus(this._getEventBusName());
    },

    /**
     * Возвращает url страницы для попапа.
     * @returns {String|undefined} Если url не был задан, то будет возвращён undefined
     */
    getUrl: function() {
        var url;

        // если url был задан, важно чтобы он был строкой
        typeof this.params.url === 'string' && (url = this.params.url);

        return url;
    },

    /**
     * Открывает новое окно
     * @param {String} url ссылка на страницу
     * @private
     */
    _openWindow: function(url) {
        return this.__self.openWindow(url, this._getEventBusName(), this._getSettings());
    },

    /**
     * Возвращает ссылку на открытое окно, если оно существует
     * @returns {window}
     * @private
     */
    _getWindow: function() {
        return this.__self.getWindow(this._getEventBusName());
    },

    /**
     * Возвращает имя шины, используемое и для нативного окна
     * @return {String}
     * @private
     */
    _getEventBusName: function() {
        return this._uniqId;
    },

    /**
     * Возвращает и наполняет контент попапа.
     * Заменяет контент, даже если пользователь повторно открыли один и тот же url,
     * т.к. могли быть переходы внутри iframe
     * @param {String} url ссылка на страницу
     * @returns {BEM.DOM}
     * @private
     */
    _getPopup: function(url) {
        return this.__self._getPopup(url, this._getEventBusName(), this.params, this.findBlockOutside('popup'));
    },

    /**
     * Составляет параметры для нового окна
     * @returns {String}
     * @private
     */
    _getSettings: function() {
        return this.__self.getSettings($.extend({
            width: this.params.width,
            height: this.params.height
        }, this.params.windowParams));
    }

}, {

    /**
     * Возвращает и наполняет контент попапа.
     * Заменяет контент, даже если пользователь повторно открыли один и тот же url,
     * т.к. могли быть переходы внутри iframe
     * @param {String} url ссылка на страницу
     * @param {String} eventBusName имя шины событий
     * @param {Object} params параметры
     * @param {String|Number} [params.width] ширина попапа
     * @param {String|Number} [params.height] высота попапа
     * @param {Boolean} [params.noNewWindow] флаг о не добавлении ссылки открытия в новом окне
     * @param {BEM.DOM} [parent] родительский попап
     * @returns {BEM.DOM}
     * @private
     */
    _getPopup: function(url, eventBusName, params, parent) {
        var popup = this._popup,
            content = this._getContent(url, eventBusName, params),
            existing$ = window.top.$ || $;

        // для модальных попапов необходимо использовать окружение самого топового окна
        // так как может возникнуть ситуация открытия попапа в iframe документе
        // проверяем блок попапа на уничтожение
        popup && !popup._isDestructing || (popup = existing$(BEMHTML.apply({
            block: 'b-modal-popup-opener',
            elem: 'popup',
            content: content,
            params: params
        })).bem('popup'));

        // приходится задавать контент через API интерфейс блока
        popup.setContent(BEMHTML.apply(content));
        // лезем в кишки, так как в api попапа нет метода сброса родительского попапа
        // если этого не делать, то при удочерении попапа, он навеки останется в DOM дереве родителя
        popup._parent = null;
        popup._isParentFixed = false;
        if (params.height == null) {
            BEM.channel('direct-tooltips')
                .on('load', this._fitHeight, { popup: popup, windowHeight: this.getWindowSize().height });
        }

        // назначать родителя попапу имеет смысл только в текущем документе
        // если попап будет создан в родительском окне, то смысла в привязке нет
        window === window.top && parent && popup.setParent(parent);

        this._popup = popup;

        this._currentParams = params;

        this._fitPopupSize();

        return popup;
    },

    /**
     * Возвращает контент для попапа
     * @param {String} url ссылка на страницу
     * @param {String} eventBusName имя шины событий
     * @param {Object} [params] флаг о не добавлении ссылки открытия в новом окне
     * @param {Boolean} [params.noNewWindow] флаг о не добавлении ссылки открытия в новом окне
     * @param {String} [params.scrolling=no] значение атрибута scrolling для iframe
     * @returns {Array}
     * @private
     */
    _getContent: function(url, eventBusName, params) {
        return [
            {
                block: 'b-iframe',
                mix: [
                    { block: 'popup', elem: 'iframe' },
                    { block: 'b-modal-popup-opener', elem: 'iframe' }
                ],
                name: eventBusName,
                src: url,
                scrolling: params.scrolling || 'no'
            },
            params.noNewWindow ? '' : {
                block: 'b-modal-popup-opener',
                elem: 'target-blank'
            }
        ];
    },

    /**
     * @anyakey для DIRECT-56678
     * для тултипов tooltip.html - при загрузке iframe а тултипе дергает
     * событие load через канал и передает ширину
     * включается, если не передана эксплицитно высота
     * @param {event} e,
     * @param {integer} height - высота попапа
     * @private
     */
    _fitHeight: function(e, height) {
        this.popup.setMod('tooltip', 'yes');
        // высота элемента с паддингами
        var iframeHeight = height + 40,
            // максимальная высота попапа -  80% экрана
            maxHeight = this.windowHeight * 0.8,
            // насколько iframe больше попапа
            diff = iframeHeight - maxHeight,
            // по умолчанию ставим попап высотой с iframe
            popupHeight = iframeHeight;
        // если iframe сильно выше попапа - ставим попап в 80% экрана
        // не используем max-height, потому что
        // а) возникает бессмысленный микро-скроллбар если iframe немного выше попапа
        // б) нужно выставить b-tooltip-у правильную высоту
        if (diff > 50) {
            popupHeight = maxHeight;
        }
        this.popup.domElem.animate({ height: popupHeight }, 300);
        // выставляем высоту b-tooltip-у, чтобы был красивый скролл
        BEM.channel('direct-tooltips').trigger('resized', popupHeight);
    },

    /**
     * Подгоняет размеры попапа, учитывая случаи когда размер попапа больше
     * вьюпорта (тогда попап будет "сжиматься" вместе с вьюпортом)
     * @private
     */
    _fitPopupSize: function() {
        var popup = this._popup,
            params = this._currentParams,
            popupSize,
            viewport,
            viewportSize,
            size,
            check;

        if (popup && !params.useNativeWindow) {
            size = {
                height: params.height || '80%',
                width: params.width || '70%'
            };

            check = {
                height: !isNaN(size.height),
                width: !isNaN(size.width)
            };

            // имеет смысл только для показанного попапа с абсолютными размерами
            if (popup.isShown() && (check.height || check.width)) {
                popupSize = popup.getPopupSize();

                viewport = popup._viewport;

                viewportSize = {
                    width: viewport.width(),
                    height: viewport.height()
                };

                // если ширина и высота попапа не влезает по вьюпорт, задаём каждую из них по максимуму

                check.width &&
                    popupSize.width >= viewportSize.width &&
                        size.width >= viewportSize.width &&
                            (size.width = '100%');

                check.height &&
                    popupSize.height >= viewportSize.height &&
                        size.height >= viewportSize.height &&
                            (size.height = '100%');
            }

            popup.domElem.css(size);
        }

    },

    /**
     * Инициализация общих для всех блоков событий
     * @private
     */
    _initCommonEvents: function() {
        if (this._commonEventsInited) return;

        // пересчет позиции открытого попапа DIRECT-37884
        BEM.blocks['i-resize'].getInstance().on('start end', function(e) {
            if (this._popup && this._popup.isShown()) {
                this._popup.show();

                this._fitPopupSize();

                this._popup.repaint();
            }
        }, this);

        this._commonEventsInited = true;
    },

    /**
     * Отправляет форму через открытый фрэйм или окно
     * @param {String} target имя фрэйма/окна для осуществления запроса
     * @param {Object} params параметры формы
     * @param {String} params.action url формы
     * @param {String} params.method метод отправки формы
     * @param {Object[]} params.data опционально список с набором полей (объекты {name, value}), отправляемых на сервер
     * @static
     */
    submitForm: function(target, params) {
        $(BEMHTML.apply({
            block: 'b-modal-popup-opener',
            elem: 'content-form',
            tag: 'form',
            attrs: {
                action: params.action,
                method: params.method,
                target: target
            },
            content: params.data.map(function(param) {
                return {
                    tag: 'input',
                    attrs: {
                        type: 'hidden',
                        name: param.name,
                        value: param.value
                    }
                }
            })
        }))
            .appendTo(document.body)
            .submit()
            .remove();
    },

    /**
     * Открывает новое окно
     * @param {String} url ссылка на страницу
     * @param {String} uniqId идентификатор окна
     * @param {String} settings параметры окна
     * @param {Number} settings.width ширина
     * @param {Number} settings.height высота
     * @param {String} settings.resizable возможность изменять размеры
     * @param {String} settings.scrollbars наличие прокрутки
     * @param {Number} settings.status

     * @private
     */
    openWindow: function(url, uniqId, settings) {
        var name,
            win = this.getWindow(uniqId);

        if (win) {
            win.document.location.href != url && (win.document.location.href = url);

            win.focus();
        } else {
            name = uniqId;

            win = window.open(url, name, settings);

            if (win) {
                win.focus();

                try {
                    top[name] = win;
                } catch (e) {
                    /**
                     * этот блок try/catch - защита от ошибок при кросс-доменном открытии попапа в iframe'ах
                     * в кросс-доменных фреймах свойство top недоступно
                     */
                }
            }
        }
    },

    /**
     * Возвращает ссылку на открытое окно c указанным uniqId, если оно существует
     * @param {String} uniqId идентификатор окна
     * @returns {window}
     * @static
     */
    getWindow: function(uniqId) {
        var win;

        try {
            win = top[uniqId];
        } catch (e) {
            /**
             * этот блок try/catch - защита от ошибок при кросс-доменном открытии попапа в iframe'ах
             * в кросс-доменных фреймах свойство top недоступно
             */
            return null;
        }

        return win != null && typeof win == 'object' && !win.closed ?
            win :
            null;
    },

    /**
     * Составляет параметры для нового окна
     * @param {Object} params параметры окна
     * @returns {String}
     * @Static
     */
    getSettings: function(params) {
        var key,
            settings = $.extend({
                width: 750,
                height: 850,
                resizable: 'yes',
                scrollbars: 'yes',
                status: 0
            }, params),
            settingsPares = [],
            documentElement = document.documentElement,
            dualScreenLeft = typeof window.screenLeft !== 'undefined' ? window.screenLeft : screen.left,
            dualScreenTop = typeof window.screenTop !== 'undefined' ? window.screenTop : screen.top,
            width = window.innerWidth ?
                window.innerWidth :
                documentElement.clientWidth ?
                    documentElement.clientWidth :
                    screen.width,
            height = window.innerHeight ?
                window.innerHeight :
                documentElement.clientHeight ?
                    documentElement.clientHeight :
                    screen.height;

        settings.left = (settings.left || (width / 2) - (settings.width / 2)) + dualScreenLeft;
        settings.top = (settings.top || (height / 2) - (settings.height / 2)) + dualScreenTop;

        for (key in settings) {
            settingsPares.push(key + '=' + settings[key]);
        }

        return settingsPares.join(',');
    },

    live: function() {
        // используется pointerclick для предотвращения блокирования окна
        // при открытии нативного попапа
        this.liveBindTo('pointerclick', function(e) {
            this.trigger('click');

            // имеет смысл что-то показывать, только если есть url
            this.getUrl() && this.show();
        });
    }

});
