(function() {
    /**
     * Набор причин отклонения с общими ссылками
     * @typedef Object VerdictsWithCommonLinks
     * @property {Object} verdicts хэш с причинами отклонения
     * @property {BEMJSON[]} commonLinks массив общих ссылок
     */

    /**
     * Контактная информация
     * @typedef Object ModerationRejectContact
     * @property {String} email
     */

    /**
     * Контактная информация про менеджера
     * @typedef Object ModerationRejectManager
     * @property {String} email
     * @property {String} login
     */

    /**
     * Контактная информация про агентство
     * @typedef Object ModerationRejectAgency
     * @property {String} email
     * @property {String} country_region_id идентификатор страны, в которой находится агентство
     */

    var
        // исключения для менеджеров, согласно DIRECT-44696
        IGNORED_MANAGERS = {
            'yndx-sedletskaya-manager': true,
            'yndx-msidenko-direct-manager': true,
            'yndx-kozhanova-manager': true,
            'yndx-vecher-manager': true,
            'yndx-perishkova-manager': true,
            'yndx-chdv-manager': true
        },
        /**
         * Генерирует bemjson списка
         * @param {Object} [mods]
         * @param {Array} items
         * @returns {BEMJSON}
         */
        list = function(mods, items) {
            var lastIndex;

            if (arguments.length < 2) {
                items = mods;
                mods = {};
            }

            lastIndex = items.length - 1;

            return {
                block: 'b-moderation-reject',
                elem: 'list',
                elemMods: mods,
                content: items.map(function(item, i) {
                    var elemMods = {};

                    if (mods.markers == 'none') {
                        elemMods.margin = 'no';

                        i == lastIndex && (elemMods['pressed-to'] = 'top');
                    }

                    if (mods.listing == 'yes') {
                        item = [item, i == lastIndex ? '.' : ';'];
                    }

                    return {
                        elem: 'list-element',
                        elemMods: elemMods,
                        content: item
                    };
                })
            };
        },
        /**
         * Генерирует bemjson ссылки
         * @param {String} url
         * @param {String|BEMJSON} text
         * @param {Boolean} [withBorder=false]
         * @returns {BEMJSON}
         */
        link = function(url, text, withBorder) {
            return {
                block: 'link',
                mods: withBorder ? {} : { theme: 'without-border' },
                attrs: /^mailto:/.test(url) ? {} : { target: '_blank' },
                url: url,
                content: text
            };
        },
        /**
         * Генерирует bemjson кнопки скачивания гарантийного письма
         * @param {String|Number} bid
         * @param {Boolean} [withChoice=false]
         * @returns {BEMJSON}
         */
        guaranteeButton = function(bid, withChoice) {
            return {
                block: 'b-gurantee-letter',
                mods: withChoice ? { 'with-choice': 'yes' } : {},
                bid: bid,
                isAgencyEmployee: u.consts('role') === 'agency'
            };
        },
        /**
         * Генерирует bemjson текста с инфо-иконкой
         * @param {String|BEMJSON} text
         * @returns {BEMJSON}
         */
        info = function(text) {
            return notice(text, 'info');
        },
        /**
         * Генерирует bemjson текста с нотис-иконкой
         * @param {String|BEMJSON} text
         * @param {String} [iconType]
         * @returns {BEMJSON}
         */
        notice = function(text, iconType) {
            return {
                block: 'icon-text',
                mods: {
                    size: 'xs',
                    theme: iconType
                },
                text: text
            }
        },
        /**
         * Генерирует bemjson блока абзаца
         * @param {Object} [mods]
         * @param {String|BEMJSON} text
         * @returns {BEMJSON}
         */
        p = function(mods, text) {
            if (arguments.length < 2) {
                text = mods;
                mods = {};
            }

            return {
                block: 'b-moderation-reject',
                elem: 'paragraph',
                elemMods: mods,
                content: text
            };
        },
        /**
         * Генерирует тексты по ключу в зависимости от параметров
         * @param {String} key - ключ для текста
         * @param {String} type - вариация текста
         */
        getText = function(key, type) {
            var texts = {
                checkAgain: {
                    banner: iget2('b-moderation-reject', 'oshibki-ustraneny-nuzhna-povtornaya', 'Ошибки устранены, нужна повторная проверка баннера'),
                    common: iget2(
                        'b-moderation-reject',
                        'oshibki-ustraneny-nuzhna-povtornaya-100',
                        'Ошибки устранены, нужна повторная проверка объявления'
                    )
                },
                'russian-law': {
                    banner: iget2(
                        'b-moderation-reject',
                        'yazyk-sayta-i-nbsp',
                        'Язык сайта и&nbsp;текста креатива, а также стоимостные показатели (валюта) должны соответствовать стране размещения рекламы.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'yazyk-sayta-i-nbsp-101',
                        'Язык сайта и&nbsp;текста объявления, а также стоимостные показатели (валюта) должны соответствовать стране размещения рекламы.'
                    )
                },
                'yandex-policy-title': {
                    banner: iget2('b-moderation-reject', 'obekt-reklamirovaniya-chyotko-opredelyon', 'Объект рекламирования чётко определён.'),
                    common: iget2(
                        'b-moderation-reject',
                        'obekt-reklamirovaniya-chyotko-opredelyon-102',
                        'Объект рекламирования чётко определён в&nbsp;тексте объявления.'
                    )
                },
                'yandex-policy-site': {
                    banner: iget2(
                        'b-moderation-reject',
                        'stranica-perehoda-dolzhna-sootvetstvovat',
                        'Страница перехода должна соответствовать креативу (например, если в&nbsp;тексте креатива есть информация о&nbsp;конкретном товаре, акции, скидке и т.п., то эта же информация должна присутствовать на&nbsp;странице перехода).'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'stranica-perehoda-dolzhna-sootvetstvovat-103',
                        'Страница перехода должна соответствовать тексту объявления (например, если в&nbsp;объявлении есть информация о&nbsp;конкретном товаре, акции, скидке и т.п., то эта же информация должна присутствовать на&nbsp;странице перехода'
                    )
                },
                'yandex-policy-registration': {
                    banner: iget2(
                        'b-moderation-reject',
                        'v-tekste-kreativa-est',
                        'В тексте креатива есть предупреждение о&nbsp;том, что необходимо зарегистрироваться на&nbsp;сайте (если регистрация обязательна).'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'v-tekste-obyavleniya-est',
                        'В тексте объявления есть предупреждение о&nbsp;том, что необходимо зарегистрироваться на&nbsp;сайте (если регистрация обязательна).'
                    )
                },
                'site-requirements-info': {
                    banner: iget2(
                        'b-moderation-reject',
                        'na-stranice-perehoda-prisutstvuet',
                        'На странице перехода присутствует информация о&nbsp;скидке, акции или конкретном товаре (услуге), если она есть в&nbsp;тексте креатива.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'na-stranice-perehoda-prisutstvuet-104',
                        'На странице перехода присутствует информация о&nbsp;скидке, акции или конкретном товаре (услуге), если она есть в&nbsp;тексте объявления.'
                    )
                },
                'site-requirements-language': {
                    banner: iget2(
                        'b-moderation-reject',
                        'sayt-sostavlen-na-nbsp',
                        'Сайт составлен на&nbsp;языке креатива, или в&nbsp;тексте креатива указан язык сайта.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'sayt-sostavlen-na-nbsp-105',
                        'Сайт составлен на&nbsp;языке рекламного объявления, или в&nbsp;тексте объявления указан язык сайта.'
                    )
                },
                'news-requirements-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-v-nbsp',
                        'Убедитесь, что в&nbsp;тексте креатива есть указание на&nbsp;информационный ресурс'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-v-nbsp-106',
                        'Убедитесь, что в&nbsp;тексте объявления есть указание на&nbsp;информационный ресурс'
                    )
                },
                'news-requirements': {
                    banner: iget2(
                        'b-moderation-reject',
                        'esli-reklamiruetsya-novostnoy-resurs',
                        'Если рекламируется новостной ресурс, баннер должен вести на&nbsp;страницу именно с&nbsp;той статьёй, которая заявлена в&nbsp;рекламе.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'esli-reklamiruetsya-novostnoy-resurs-107',
                        'Если рекламируется новостной ресурс, объявление должно вести на&nbsp;страницу именно с&nbsp;той статьёй, которая заявлена в&nbsp;рекламе.'
                    )
                },
                'false-info-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'isklyuchite-iz-nbsp-teksta',
                        'Исключите из&nbsp;текста креатива слова в&nbsp;превосходной или сравнительной степени'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'isklyuchite-iz-nbsp-teksta-108',
                        'Исключите из&nbsp;текста объявления слова в&nbsp;превосходной или сравнительной степени'
                    )
                },
                'false-info': {
                    banner: iget2(
                        'b-moderation-reject',
                        'slova-v-nbsp-prevoshodnoy',
                        'Слова в&nbsp;превосходной или сравнительной степени можно использовать в&nbsp;тексте креатива, если на&nbsp;странице перехода размещено заключение третьих лиц о&nbsp;преимуществе вашего товара (услуги) перед другими товарами.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'slova-v-nbsp-prevoshodnoy-109',
                        'Слова в&nbsp;превосходной или сравнительной степени можно использовать в&nbsp;тексте объявления, если на&nbsp;странице перехода размещено заключение третьих лиц о&nbsp;преимуществе вашего товара (услуги) перед другими товарами.'
                    )
                },
                'errors-title': {
                    banner: iget2('b-moderation-reject', 'ubedites-chto-v-nbsp-110', 'Убедитесь, что в&nbsp;тексте креатива нет опечаток'),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-v-nbsp-111',
                        'Убедитесь, что в&nbsp;тексте нет опечаток, а шаблон используется корректно'
                    ),
                    video: iget2('b-moderation-reject', 'ubedites-chto-v-nbsp-112', 'Убедитесь, что в&nbsp;тексте нет опечаток')
                },
                'errors-subtitle': {
                    banner: iget2('b-moderation-reject', 'v-tekste-ne-dopuskaetsya', 'В тексте не допускается'),
                    common: iget2('b-moderation-reject', 'v-tekste-obyavleniya-ne', 'В тексте объявления не допускается'),
                    video: iget2('b-moderation-reject', 'v-tekste-video-obyavleniya-ne', 'В рекламном тексте не допускается')
                },
                'invalid-url': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-sayt-ispolzuemyy',
                        'Убедитесь, что сайт, используемый в&nbsp;рекламе, работает, а страница, перехода открывается корректно.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-sayt-ispolzuemyy-112',
                        'Убедитесь, что сайт, используемый в&nbsp;объявлении, работает, а страница, на&nbsp;которую ссылается объявление, открывается корректно.'
                    )
                },
                'irrelevant-keywords': {
                    banner: iget2(
                        'b-moderation-reject',
                        'klyuchevye-frazy-dolzhny-imet',
                        'Ключевые фразы должны иметь прямое отношение к&nbsp;теме рекламы.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'klyuchevye-frazy-dolzhny-imet-113',
                        'Ключевые фразы должны иметь прямое отношение к&nbsp;теме объявления.'
                    )
                },
                'yandex-policy-auto': {
                    banner: iget2(
                        'b-moderation-reject',
                        'vash-kreativ-ne-mozhet',
                        'Ваш креатив не может быть принят к размещению, поскольку рекламные материалы содержат информацию о запрещённых к рекламе товарах/услугах. '
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'vashe-reklamnoe-obyavlenie-ne',
                        'Ваше рекламное объявление не может быть принято к размещению, поскольку рекламные материалы содержат информацию о запрещённых к рекламе товарах/услугах. '
                    )
                },
                'irrelevant-image-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-izobrazhenie-sootvetstvuet',
                        'Убедитесь, что изображение соответствует тематике баннера'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-izobrazhenie-sootvetstvuet-114',
                        'Убедитесь, что изображение соответствует тематике объявления'
                    )
                },
                'irrelevant-image': {
                    banner: iget2(
                        'b-moderation-reject',
                        'dopustimo-ispolzovat-tolko-te',
                        'Допустимо использовать только те изображения, которые имеют непосредственное отношение к&nbsp;заголовку, тексту баннера и&nbsp;содержанию страницы сайта.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'dopustimo-ispolzovat-tolko-te-115',
                        'Допустимо использовать только те изображения, которые имеют непосредственное отношение к&nbsp;заголовку, тексту объявления и&nbsp;содержанию страницы сайта.'
                    )
                },
                'ad_image-bad-text': {
                    banner: iget2('b-moderation-reject', 'ubedites-chto-tekst-kreativa', 'Убедитесь, что текст креатива можно прочитать'),
                    common: iget2('b-moderation-reject', 'ubedites-chto-tekst-na', 'Убедитесь, что текст на графическом объявлении можно прочитать')
                },
                'ad_image-no-border-title': {
                    banner: iget2('b-moderation-reject', 'ubedites-chto-kreativ-imeet', 'Убедитесь, что креатив имеет видимые границы'),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie',
                        'Убедитесь, что графическое объявление имеет видимые границы'
                    )
                },
                'ad_image-no-border': {
                    banner: iget2(
                        'b-moderation-reject',
                        'kreativ-dolzhen-imet-chyotkie',
                        'Креатив должен иметь чёткие видимые границы и непрозрачный фон. Цвет рамки вокруг изображения не должен совпадать с цветом фона.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'izobrazhenie-obyavleniya-dolzhno-imet',
                        'Изображение объявления должно иметь чёткие видимые границы и непрозрачный фон. Цвет рамки вокруг изображения не должен совпадать с цветом фона.'
                    )
                },
                'ad_image-bad-quality': {
                    banner: iget2(
                        'b-moderation-reject',
                        'izobrazhenie-dolzhno-byt-kachestvennym',
                        'Изображение должно быть качественным и чётким, без артефактов сжатия.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'izobrazhenie-graficheskogo-obyavleniya-dolzhno',
                        'Изображение графического объявления должно быть качественным и чётким, без артефактов сжатия.'
                    )
                },
                'ad_image-no-source-title': {
                    banner: iget2('b-moderation-reject', 'ubedites-chto-v-kreative', 'Убедитесь, что в креативе указан источник информации'),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-v-graficheskom',
                        'Убедитесь, что в графическом объявлении указан источник информации'
                    )
                },
                'ad_image-no-source': {
                    banner: iget2(
                        'b-moderation-reject',
                        'izobrazhenie-dolzhno-davat-chyotkoe',
                        'Изображение должно давать чёткое представление об источнике содержащейся в нем информации. Например, на изображении может присутствовать логотип, название бренда, название организации или ФИО индивидуального предпринимателя.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'izobrazhenie-obyavleniya-dolzhno-davat',
                        'Изображение объявления должно давать чёткое представление об источнике содержащейся в нем информации. Например, на изображении может присутствовать логотип, название бренда, название организации или ФИО индивидуального предпринимателя.'
                    )
                },
                'ad_image-nasty-title': {
                    banner: iget2('b-moderation-reject', 'izmenite-kreativ', 'Измените креатив'),
                    common: iget2('b-moderation-reject', 'izmenite-izobrazhenie-graficheskogo-obyavleniya', 'Измените изображение графического объявления')
                },
                'ad_image-interface-element-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-kreativ-ne',
                        'Убедитесь, что креатив не содержит элементы пользовательского графического интерфейса'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie-116',
                        'Убедитесь, что графическое объявление не содержит элементы пользовательского графического интерфейса'
                    )
                },
                'ad_image-interface-element': {
                    banner: iget2('b-moderation-reject', 'kreativ-ne-dolzhen-soderzhat', 'Креатив не должен содержать:'),
                    common: iget2('b-moderation-reject', 'graficheskoe-obyavlenie-ne-dolzhno', 'Графическое объявление не должно содержать:')
                },
                'ad_image-finance-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-kreativ-soderzhit',
                        'Убедитесь, что креатив содержит достаточную информацию о предоставителе услуг и источник информации'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie-117',
                        'Убедитесь, что графическое объявление содержит достаточную информацию о предоставителе услуг и источник информации'
                    )
                },
                'ad_image-lottery-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-kreativ-soderzhit-118',
                        'Убедитесь, что креатив содержит необходимую информацию о мероприятии'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie-119',
                        'Убедитесь, что графическое объявление содержит необходимую информацию о мероприятии'
                    )
                },
                'ad_image-lottery': {
                    banner: iget2(
                        'b-moderation-reject',
                        'reklama-provedeniya-stimuliruyushchego-meropriyatiya',
                        'Реклама проведения стимулирующего мероприятия (например, лотереи или конкурса) должна содержать:'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'graficheskoe-obyavlenie-o-provedenii',
                        'Графическое объявление о проведении стимулирующего мероприятия (например, лотереи или конкурса) должно содержать:'
                    )
                },
                'ad_image-distance-title': {
                    banner: iget2('b-moderation-reject', 'ubedites-chto-kreativ-soderzhit-120', 'Убедитесь, что креатив содержит данные о продавце'),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie-121',
                        'Убедитесь, что графическое объявление содержит данные о продавце'
                    )
                },
                'ad_image-distance': {
                    banner: iget2(
                        'b-moderation-reject',
                        'reklama-distancionnoy-prodazhi-tovarov',
                        'Реклама дистанционной продажи товаров должна содержать наименование юридического лица/ФИО индивидуального предпринимателя, его адрес и ОГРН.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'graficheskoe-obyavlenie-o-distancionnoy',
                        'Графическое объявление о дистанционной продаже товаров должно содержать наименование юридического лица/ФИО индивидуального предпринимателя, его адрес и ОГРН.'
                    )
                },
                'ad_image-medicine-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-reklama-soprovozhdaetsya',
                        'Убедитесь, что реклама сопровождается необходимым предупреждением'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie-122',
                        'Убедитесь, что графическое объявление сопровождается необходимым предупреждением'
                    )
                },
                'ad_image-medicine-warning': {
                    banner: iget2(
                        'b-moderation-reject',
                        'reklama-lekarstvennyh-sredstv-medicinskoy',
                        'Реклама лекарственных средств, медицинской техники, медицинских услуг и методов лечения должна содержать предупреждение о наличии противопоказаний, необходимости ознакомления с инструкцией по применению или получения консультации специалиста. Предупреждение должно занимать не менее 5% площади креатива.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'graficheskoe-obyavlenie-o-lekarstvennyh',
                        'Графическое объявление о лекарственных средствах, медицинской технике, медицинских услугах и методах лечения должно содержать предупреждение о наличии противопоказаний, необходимости ознакомления с инструкцией по применению или получения консультации специалиста. Предупреждение должно занимать не менее 5% площади графического объявления.'
                    )
                },
                'ad_image-medicine-bad': {
                    banner: iget2(
                        'b-moderation-reject',
                        'reklama-bad-dolzhna-soderzhat',
                        'Реклама БАД должна содержать предупреждение о том, что товар не является лекарством. Информация должна занимать не менее 10% площади креатива.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'graficheskoe-obyavlenie-o-bad',
                        'Графическое объявление о БАД должно содержать предупреждение о том, что товар не является лекарством. Информация должна занимать не менее 10% площади графического объявления.'
                    )
                },
                'ad_image-smi-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-kreativ-soderzhit-123',
                        'Убедитесь, что креатив содержит категорию информационной продукции'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie-124',
                        'Убедитесь, что графическое объявление содержит категорию информационной продукции'
                    )
                },
                'ad_image-babyfood-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-kreativ-sootvetstvuet',
                        'Убедитесь, что креатив соответствует требованиям к рекламе детского питания'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie-125',
                        'Убедитесь, что графическое объявление соответствует требованиям к рекламе детского питания'
                    )
                },
                'ad_image-babyfood': {
                    banner: iget2(
                        'b-moderation-reject',
                        'reklama-produktov-vklyuchyonnyh-v',
                        'Реклама продуктов, включённых в рацион ребёнка в течение его первого года жизни, или заменителей женского молока должна содержать возрастное ограничение применения и предупреждение о необходимости консультации специалиста.'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'graficheskoe-obyavlenie-o-produktah',
                        'Графическое объявление о продуктах, включённых в рацион ребёнка в течение его первого года жизни, или заменителях женского молока должно содержать возрастное ограничение применения и предупреждение о необходимости консультации специалиста.'
                    )
                },
                'ad_image-house-cooperative-title': {
                    banner: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-kreativ-soderzhit-126',
                        'Убедитесь, что креатив содержит информацию о жилищно-накопительном кооперативе'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'ubedites-chto-graficheskoe-obyavlenie-127',
                        'Убедитесь, что графическое объявление содержит информацию о жилищно-накопительном кооперативе'
                    )
                },
                'ad_image-house-cooperative': {
                    banner: iget2(
                        'b-moderation-reject',
                        'reklama-zhilishchno-nakopitelnyh-kooperativov',
                        'Реклама жилищно-накопительных кооперативов (ЖНК) должна содержать:'
                    ),
                    common: iget2(
                        'b-moderation-reject',
                        'graficheskoe-obyavlenie-o-zhilishchno',
                        'Графическое объявление о жилищно-накопительном кооперативе (ЖНК) должно содержать:'
                    )
                }
            };

            return texts[key] ? (texts[key][type] || texts[key].common) : '';
        };

    u.register({

        'b-moderation-reject': {

            /**
             * Возвращает контакты по умолчанию для агентских кампаний DIRECT-51710 (кроме Украины)
             * @param {ModerationRejectAgency} agency объект с информацией об агентстве
             * @param {ModerationRejectManager} [agencyManager] объект с информацией о менеджере агентства
             * @returns {ModerationRejectContact}
             */
            getAgencyManagerContacts: function(agency, agencyManager) {

                var agencyCIS = { email: 'sngmod-agency@support.yandex.ru' },
                    agencyUA = { email: 'mod-agency@support.yandex.ua' },
                    agencyDefault = { email: 'mod-agency@support.yandex.ru' },
                    agencySelected = ({
                        // Беларусь
                        149: agencyCIS,
                        // Казахстан
                        159: agencyCIS,
                        // Украина
                        187: agencyUA //agencyManager || {}
                    })[agency.country_region_id];

                return agencySelected || agencyDefault;
            },

            /**
             * Вычисляет контактную информацию по переданным параметрам
             * @param {Object} data
             * @param {ModerationRejectManager} [data.manager] информация о менеджере
             * @param {ModerationRejectAgency} [data.agency] информация об агентстве
             * @param {ModerationRejectManager} [data.agencyManager] информация о менеджере агентства
             * @returns {ModerationRejectContact}
             */
            specifyContacts: function(data) {
                // при наличии информации о менеджере, используются контакты менеджера
                if (data.manager) return data.manager;

                // при наличии информации об агентстве
                if (data.agency) {
                    // для не агентств используются контакты агентства
                    if (u.consts('role') !== 'agency') return data.agency;

                    // для агентств, менеджеры которых не попали в исключение
                    // используются контакты по умолчанию
                    if (!IGNORED_MANAGERS[(data.agencyManager || {}).login]) {
                        return this.getAgencyManagerContacts(
                            data.agency,
                            data.agencyManager);
                    }
                }

                // при невозможности определить контактную информацию возвращаем пустые контакты
                return { email: null };
            },

            /**
             * Формирует объект с причинами отклонения и общими ссылками по переданным параметрам
             * @param {String|Number} bid идентификатор баннера
             * @param {Object} data
             * @param {'banner'|'video'|'common'} [type='common'] - Тип модерируемого объекта,
             * нужно чтобы выбрать корректную формулировку
             * (например, использовать слово "баннер"/"креатив" вместо "объявление", так как в объявлении нет текста)
             * @param {ModerationRejectManager} [data.manager] информация о менеджере
             * @param {ModerationRejectAgency} [data.agency] информация об агентстве
             * @param {ModerationRejectManager} [data.agencyManager] информация о менеджере агентства
             * @returns {VerdictsWithCommonLinks}
             */
            getVerdictsWithCommonLinks: function(bid, data, type) {
                if (!type) {
                    type = 'common';
                }

                var agencyManager = data.agencyManager || {},
                    contacts = this.specifyContacts(data),
                    dissentUrl = contacts.email ?
                        'mailto:' + contacts.email :
                        u.getCommonHelpUrl('/direct/troubleshooting/moderation.html#dissent'),
                    sendDocsUrl = contacts.email ?
                        'mailto:' + contacts.email :
                        u.getCommonHelpUrl('/direct/troubleshooting/moderation.html#document'),
                    licenseHelpText = iget2(
                        'b-moderation-reject',
                        'komplekt-dokumentov-zavisit-ot',
                        'Комплект документов зависит от тематики рекламы — найти перечни документов для самых распространённых тематик можно в {help}. ',
                        {
                            help: function(text) {
                                return link(u.getCommonHelpUrl('/direct/moderation/special-categories.html'), text);
                            }(iget2('b-moderation-reject', 'komplekt-dokumentov-zavisit-ot-param-help', 'помощи')),

                            context: 'help - помощи'
                        }
                    ),
                    documentsTimeHelp = info(iget2(
                        'b-moderation-reject',
                        'proverka-dokumentov-mozhet-zanimat',
                        'Проверка документов может занимать некоторое время. Пожалуйста, не высылайте документы повторно — как только проверка будет закончена, вы получите уведомление на адрес электронной почты, который вы указали при отправке'
                    )),
                    russianLawRec = [
                        iget2('b-moderation-reject', 'nelzya-reklamirovat', 'Нельзя рекламировать') + ':',
                        list({ listing: 'yes' }, [
                            iget2('b-moderation-reject', 'azartnye-igry-sportivnye-prognozy', 'азартные игры, спортивные прогнозы, букмекерские конторы'),
                            iget2('b-moderation-reject', 'alkogolnuyu-produkciyu', 'алкогольную продукцию'),
                            iget2('b-moderation-reject', 'tabachnuyu-produkciyu-i-nbsp', 'табачную продукцию и&nbsp;курительные принадлежности'),
                            iget2('b-moderation-reject', 'oruzhie', 'оружие'),
                            iget2('b-moderation-reject', 'recepturnye-preparaty', 'рецептурные препараты'),
                            iget2('b-moderation-reject', 'deystviya-pobuzhdayushchie-k-nbsp', 'действия, побуждающие к&nbsp;нарушению закона'),
                            iget2(
                                'b-moderation-reject',
                                'distancionnuyu-prodazhu-tovarov-zapreshchyonnyh',
                                'дистанционную продажу товаров, запрещённых к&nbsp;реализации дистанционным способом (ювелирные изделия, лекарственные средства и&nbsp;др.)'
                            ),
                            iget2('b-moderation-reject', 'prodazhu-i-nbsp-poddelku', 'продажу и&nbsp;подделку официальных документов'),
                            iget2(
                                'b-moderation-reject',
                                'drugie-link-tovary-i',
                                'другие {link}, реклама которых запрещена или ограничена в&nbsp;интернете',
                                {
                                    link: function(text) {
                                        return link(u.getLegalUrl('/adv_rules/'), text);
                                    }(iget2('b-moderation-reject', 'drugie-link-tovary-i-param-link', 'товары и&nbsp;услуги')),

                                    context: 'link - товары и&nbsp;услуги'
                                }
                            )
                        ])
                    ],
                    tmRec = [
                        p({ space: 'yes' }, iget2(
                            'b-moderation-reject',
                            'my-poluchili-oficialnuyu-zhalobu',
                            'Мы получили официальную жалобу на&nbsp;нарушение прав интеллектуальной собственности от&nbsp;правообладателя. Дальнейшее размещение вашей рекламы с&nbsp;упоминанием данного объекта интеллектуальной собственности (товарного знака, фирменного наименования и&nbsp;т.п.) будет возможно только после того, как вы предоставите нам оригинал гарантийного письма от&nbsp;плательщика.'
                        )),
                        p(iget2(
                            'b-moderation-reject',
                            'pered-zapolneniem-garantiynogo-pisma',
                            'Перед заполнением гарантийного письма, пожалуйста, ознакомьтесь с&nbsp;{rules}.',
                            {
                                rules: function(text) {
                                    return link(
                                        u.getCommonHelpUrl('/direct/moderation/letter-of-guarantee.html'),
                                        text
                                    );
                                }(iget2('b-moderation-reject', 'pered-zapolneniem-garantiynogo-pisma-param-rules', 'правилами его оформления')),

                                context: 'rules - правилами его оформления'
                            }
                        )),
                        guaranteeButton(bid),
                        info(iget2(
                            'b-moderation-reject',
                            'dlya-proverki-korrektnosti-zapolneniya',
                            'Для проверки корректности заполнения вы можете {send}.',
                            {
                                send: function(text) {
                                    return link(sendDocsUrl, text);
                                }(iget2(
                                    'b-moderation-reject',
                                    'dlya-proverki-korrektnosti-zapolneniya-param-send',
                                    'прислать электронную копию гарантийного письма'
                                )),

                                context: 'send - прислать электронную копию гарантийного письма'
                            }
                        ))
                    ],
                    guaranteeLetter = [
                        iget2(
                            'b-moderation-reject',
                            'esli-dlya-vashih-tovarov',
                            'Если для ваших товаров / услуг дополнительно требуется гарантийное письмо от плательщика, ниже вы можете скачать подготовленный шаблон, а затем заполнить недостающие поля в соответствии с {rules}',
                            {
                                rules: function(text) {
                                    return link(
                                        u.getCommonHelpUrl('/direct/moderation/letter-of-guarantee.html'),
                                        text
                                    );
                                }(iget2('b-moderation-reject', 'esli-dlya-vashih-tovarov-param-rules', 'правилами оформления гарантийного письма')),

                                context: 'rules - правилами оформления гарантийного письма'
                            }
                        ),
                        guaranteeButton(bid, true)
                    ],
                    reasonsQuestion = link(
                        // special_type - означает, что самоходные
                        agencyManager.special_type === 'self_agency' ?
                            u.getCommonHelpUrl('/direct/troubleshooting/moderation.html') :
                            dissentUrl,
                        iget2('b-moderation-reject', 'ostalis-voprosy-o-prichinah', 'Остались вопросы о причинах отклонения'),
                        true
                    ),
                    sendDocs = link(
                        // special_type - означает, что самоходные
                        agencyManager.special_type === 'self_agency' ?
                            u.getCommonHelpUrl('/direct/troubleshooting/moderation.html') :
                            sendDocsUrl,
                        iget2('b-moderation-reject', 'otpravit-dokumenty', 'Отправить документы'),
                        true
                    ),
                    html5RequirementsLink = link(
                        u.formatUrl({ host: u.url.getRegionalDomain(), path: '/adv/requirements/html5#direct' }),
                        iget2('b-moderation-reject', 'html5-requirements', 'Технические требования к баннеру HTML5 в Директе')
                    ),
                    fundReturn = link(u.getCommonHelpUrl('/direct/troubleshooting/payment.html#return'), iget2('b-moderation-reject', 'ostavit-zayavku-na-vozvrat', 'Оставить заявку на возврат средств'), true),
                    serviceQuestion = link(dissentUrl, iget2('b-moderation-reject', 'ostalis-drugie-voprosy-po', 'Остались другие вопросы по сервису'), true),
                    checkAgain = link(dissentUrl, getText('checkAgain', type), true),
                    falseInfoProofs = iget2(
                        'b-moderation-reject',
                        'esli-u-vas-est',
                        'Если у вас есть подобные заключения, добавьте ссылку на них на странице перехода и {link}.',
                        {
                            link: function(text) {
                                return link(dissentUrl, text);
                            }(iget2('b-moderation-reject', 'esli-u-vas-est-param-link', 'сообщите об этом нам')),

                            context: 'link - сообщите об этом нам'
                        }
                    ),
                    generalAdvRulesLink = link(u.getLegalUrl('/general_adv_rules/'), iget2('b-moderation-reject', 'polnyy-spisok-trebovaniy', 'Полный список требований')),
                    advRulesLink = link(u.getLegalUrl('/adv_rules/'), iget2('b-moderation-reject', 'pravila-razmeshcheniya-reklamy', 'Правила размещения рекламы')),
                    displayRulesLink = link(u.getLegalUrl('/direct_display_rules/'), iget2('b-moderation-reject', 'pravila-pokaza', 'Правила показа')),
                    verdicts = {
                        'kazakh-law': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-reklama-vashih',
                                'Убедитесь, что реклама ваших товаров и&nbsp;услуг разрешена законодательством Республики Казахстан'
                            ),
                            recommendation: [
                                iget2('b-moderation-reject', 'nelzya-reklamirovat', 'Нельзя рекламировать') + ':',
                                list({ listing: 'yes' }, [
                                    iget2('b-moderation-reject', 'recepturnye-preparaty', 'рецептурные препараты'),
                                    iget2('b-moderation-reject', 'tabak-i-nbsp-tabachnye', 'табак и&nbsp;табачные изделия'),
                                    iget2(
                                        'b-moderation-reject',
                                        'narkoticheskie-psihotropnye-veshchestva-i',
                                        'наркотические, психотропные вещества и&nbsp;прекурсоры'
                                    ),
                                    iget2('b-moderation-reject', 'azartnye-igry', 'азартные игры'),
                                    iget2('b-moderation-reject', 'oruzhie', 'оружие'),
                                    iget2(
                                        'b-moderation-reject',
                                        'drugie-other-tovary-i',
                                        'другие {other}, реклама которых запрещена или ограничена законодательством Республики Казахстан',
                                        {
                                            other: function(text) {
                                                return link(iget2('b-moderation-reject', 'https-legal-yandex-kz', 'https://legal.yandex.kz/adv_rules/?ncrnd=5933'), text);
                                            }(iget2('b-moderation-reject', 'drugie-other-tovary-i-param-other', 'товары и&nbsp;услуги')),

                                            context: 'other - товары и&nbsp;услуги'
                                        }
                                    )
                                ])
                            ],
                            links: [reasonsQuestion]
                        },
                        copyright: {
                            title: iget2('b-moderation-reject', 'predostavte-garantiynoe-pismo-ot', 'Предоставьте гарантийное письмо от&nbsp;плательщика'),
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'pri-reklame-rezultatov-intellektualnoy',
                                    'При рекламе результатов интеллектуальной деятельности (музыка, видеокурсы, книги, программные продукты и&nbsp;т.п.) предоставление гарантийного письма является обязательным.'
                                ),
                                info(iget2(
                                    'b-moderation-reject',
                                    's-pomoshchyu-garantiynogo-pisma',
                                    'С помощью гарантийного письма вы подтверждаете, что соблюдаете права третьих лиц в&nbsp;отношении интеллектуальной собственности. При этом вы не обязаны быть автором или правообладателем объекта рекламирования.'
                                )),
                                guaranteeButton(bid),
                                iget2(
                                    'b-moderation-reject',
                                    'pered-zapolneniem-garantiynogo-pisma',
                                    'Перед заполнением гарантийного письма, пожалуйста, ознакомьтесь с&nbsp;{rules}.',
                                    {
                                        rules: function(text) {
                                            return link(
                                                u.getCommonHelpUrl('/direct/moderation/letter-of-guarantee.html'),
                                                text
                                            );
                                        }(iget2('b-moderation-reject', 'pered-zapolneniem-garantiynogo-pisma-param-rules', 'правилами его оформления')),

                                        context: 'rules - правилами его оформления'
                                    }
                                )
                            ],
                            links: [sendDocs, reasonsQuestion]
                        },
                        spam: {
                            title: iget2(
                                'b-moderation-reject',
                                'reklamnye-materialy-ne-mogut',
                                'Рекламные материалы не могут быть размещены в&nbsp;Яндекс.Директе'
                            ),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'v-hode-proverki-byli',
                                'В ходе проверки были выявлены несоответствия рекламной политике Яндекса.</br>Более подробные комментарии не могут быть предоставлены. В&nbsp;соответствии с&nbsp;пунктом 15 {rules} Яндекс оставляет за&nbsp;собой право отклонить любой рекламный материал без объяснения причин.</br></br>Если на&nbsp;счёте вашей рекламной кампании остались средства, они могут быть возвращены — для этого необходимо предоставить оригинал заявления от&nbsp;плательщика.',
                                {
                                    rules: function(text) {
                                        return link(u.getLegalUrl('/general_adv_rules/'), text);
                                    }(iget2('b-moderation-reject', 'v-hode-proverki-byli-param-rules', 'Требований к&nbsp;рекламным материалам')),

                                    context: 'rules - Требований к&nbsp;рекламным материалам'
                                }
                            ),
                            links: [fundReturn, serviceQuestion]
                        },
                        'general-for-empty-reason': {
                            title: '',
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'common-moderation-error',
                                    'Убедитесь, что реклама ваших товаров и услуг допускается действующим законодательством и рекламной политикой Яндекса. Проверьте содержание объявления и рекламируемого сайта, внесите изменения и отправьте объявление на повторную проверку. {rules}',
                                    {
                                        rules: function(text) {
                                            return link(u.getCommonHelpUrl('/direct/moderation/adv-rules.html'), text)
                                        }(iget2('b-moderation-reject', 'pravila-i-trebovania', 'Правила и требования к рекламе'))
                                    }
                                )
                            ],
                            links: [reasonsQuestion]
                        },
                        'russian-law': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-reklama-vashih-128',
                                'Убедитесь, что реклама ваших товаров и&nbsp;услуг допускается действующим законодательством'
                            ),
                            recommendation: [
                                russianLawRec,
                                notice(getText('russian-law', type))
                            ],
                            links: [reasonsQuestion]
                        },
                        'russian-law-callout': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-reklama-vashih-128',
                                'Убедитесь, что реклама ваших товаров и&nbsp;услуг допускается действующим законодательством'
                            ),
                            recommendation: russianLawRec,
                            links: [reasonsQuestion]
                        },
                        'yandex-policy': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-reklamnye-materialy-129',
                                'Убедитесь, что рекламные материалы соответствуют рекламной политике Яндекса'
                            ),
                            recommendation: [
                                list([
                                    [
                                        getText('yandex-policy-title', type),
                                        '&nbsp;',
                                        iget2('b-moderation-reject', 'naprimer', 'Например') + ':',
                                        list({ markers: 'none' }, [
                                            iget2(
                                                'b-moderation-reject',
                                                'kak-zarabotat-v-nbsp',
                                                '«Как заработать в&nbsp;интернете? Курс на&nbsp;DVD» — понятно, что именно рекламируется.'
                                            ),
                                            iget2(
                                                'b-moderation-reject',
                                                'kak-zarabotat-v-nbsp-130',
                                                '«Как заработать в&nbsp;интернете? Спешите узнать» — объект рекламирования не ясен.'
                                            )
                                        ])
                                    ],
                                    getText('yandex-policy-site', type),
                                    iget2(
                                        'b-moderation-reject',
                                        'na-stranice-perehoda-razmeshcheny',
                                        'На странице перехода размещены контактные данные рекламодателя.'
                                    ),
                                    getText('yandex-policy-registration', type),
                                    iget2('b-moderation-reject', 'na-sayte-otsutstvuyut-virusy', 'На сайте отсутствуют вирусы.'),
                                    iget2(
                                        'b-moderation-reject',
                                        'na-stranice-perehoda-ukazana',
                                        'На странице перехода указана информация о&nbsp;стоимости и форме оплаты предлагаемых услуг.'
                                    )
                                ]),
                                generalAdvRulesLink
                            ],
                            links: [reasonsQuestion]
                        },
                        'site-requirements': {
                            title: iget2('b-moderation-reject', 'ubedites-chto-sayt-sootvetstvuet', 'Убедитесь, что сайт соответствует требованиям Яндекса'),
                            recommendation: [
                                list([
                                    getText('site-requirements-info', type),
                                    iget2(
                                        'b-moderation-reject',
                                        'stranica-perehoda-otkryvaetsya-korrektno',
                                        'Страница перехода открывается корректно во всех браузерах.'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'reklamnye-bloki-na-nbsp',
                                        'Рекламные блоки на&nbsp;странице перехода занимают менее половины экрана.'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'na-stranice-perehoda-otsutstvuyut',
                                        'На странице перехода отсутствуют всплывающие окна (pop-up и&nbsp;pop-under).'
                                    ),
                                    getText('site-requirements-language', type)
                                ]),
                                generalAdvRulesLink
                            ],
                            links: [checkAgain]
                        },
                        'news-requirements': {
                            title: getText('news-requirements-title', type),
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'vy-mozhete-ispolzovat-utochnyayushchie',
                                    'Вы можете использовать уточняющие фразы: «подробнее на», «читайте на», «обзор» и&nbsp;т.п.'
                                ),
                                info(getText('news-requirements', type))
                            ],
                            links: [reasonsQuestion]
                        },
                        'false-info': {
                            title: getText('false-info-title', type),
                            recommendation: [
                                p(getText('false-info', type)),
                                p(iget2(
                                    'b-moderation-reject',
                                    'esli-u-vas-net',
                                    'Если у вас нет таких заключений, замените слова в&nbsp;превосходной степени — например, «лучший», «новейший» — на&nbsp;фразы «один из&nbsp;лучших», «современный» и&nbsp;т.п.'
                                ))
                            ],
                            links: [falseInfoProofs]
                        },
                        errors: {
                            title: getText('errors-title', type),
                            recommendation: [
                                getText('errors-subtitle', type) + ':',
                                list(
                                    u._.compact([
                                        type === 'video' ?
                                            iget2(
                                                'b-moderation-reject',
                                                'pisat-video-imena-sobstvennye-so',
                                                'писать имена собственные со строчной буквы;'
                                            ) :
                                            iget2(
                                                'b-moderation-reject',
                                                'pisat-imena-sobstvennye-so',
                                                'писать имена собственные со строчной буквы (в&nbsp;том числе в&nbsp;ключевых фразах, если вы используете шаблон);'
                                            ),
                                        type === 'video' ?
                                            '' :
                                            iget2('b-moderation-reject', 'ispolzovat-r-nbsp-a', 'использовать р&nbsp;а&nbsp;з&nbsp;р&nbsp;я&nbsp;д&nbsp;к&nbsp;у;'),
                                        type === 'video' ?
                                            '' :
                                            iget2(
                                                'b-moderation-reject',
                                                'pisat-slova-zaglavnymi-bukvami',
                                                'писать слова заглавными буквами (за&nbsp;исключением аббревиатур);'
                                            ),
                                        iget2('b-moderation-reject', 'ispolzovat-zhargon-i-nbsp', 'использовать жаргон и&nbsp;просторечные выражения;'),
                                        iget2('b-moderation-reject', 'ispolzovat-nekorrektnye-sokrashcheniya', 'использовать некорректные сокращения;'),
                                        iget2(
                                            'b-moderation-reject',
                                            'ispolzovat-mnogokratnye-povtory-odnih',
                                            'использовать многократные повторы одних и&nbsp;тех же слов или фраз.'
                                        )
                                    ])
                                ),
                                link(u.getLegalUrl('/adv_rules/'), iget2('b-moderation-reject', 'polnyy-spisok-trebovaniy', 'Полный список требований')),
                                type === 'common' ?
                                    info(iget2(
                                        'b-moderation-reject',
                                        'proverte-chtoby-pri-podstanovke',
                                        'Проверьте, чтобы при подстановке ключевых слов в&nbsp;шаблон все слова в&nbsp;тексте были согласованы между собой.'
                                    )) :
                                    ''
                            ],
                            links: [reasonsQuestion]
                        },
                        tm: {
                            title: iget2('b-moderation-reject', 'predostavte-original-garantiynogo-pisma', 'Предоставьте оригинал гарантийного письма'),
                            recommendation: [tmRec],
                            links: [sendDocs]
                        },
                        'invalid-url': {
                            title: iget2('b-moderation-reject', 'proverte-dostupnost-sayta', 'Проверьте доступность сайта'),
                            recommendation: getText('invalid-url', type),
                            links: [checkAgain]
                        },
                        politics: {
                            title: iget2('b-moderation-reject', 'v-yandeks-direkte-ne', 'В Яндекс.Директе не&nbsp;может быть размещена политическая реклама'),
                            recommendation: [
                                p(iget2(
                                    'b-moderation-reject',
                                    'eto-kasaetsya-ne-nbsp',
                                    'Это касается не&nbsp;только непосредственно предвыборной агитации, но и&nbsp;рекламы сайтов депутатов, политических деятелей, их приёмных.'
                                )),
                                p(iget2(
                                    'b-moderation-reject',
                                    'my-takzhe-ne-nbsp',
                                    'Мы также не&nbsp;размещаем рекламу опросов общественного мнения относительно выборов и&nbsp;референдумов и&nbsp;рекламу прогнозов результатов выборов.'
                                ))
                            ],
                            links: [reasonsQuestion]
                        },
                        'contact-info': {
                            title: iget2('b-moderation-reject', 'isklyuchite-kontaktnye-dannye-iz', 'Исключите контактные данные из&nbsp;текста объявления'),
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'v-obyavlenii-ne-nbsp',
                                    'В объявлении не&nbsp;должны быть указаны номера телефонов, электронные или почтовые адреса. Вы можете указать эти данные в&nbsp;{vcard}.',
                                    {
                                        vcard: function(text) {
                                            return link(u.getCommonHelpUrl('/direct/features/vcards.xml'), text);
                                        }(iget2('b-moderation-reject', 'v-obyavlenii-ne-nbsp-param-vcard', 'виртуальной визитке')),

                                        context: 'vcard - виртуальной визитке'
                                    }
                                )
                            ],
                            links: [reasonsQuestion]
                        },
                        'license-required': {
                            title: iget2(
                                'b-moderation-reject',
                                'predostavte-podtverzhdayushchie-dokumenty-na',
                                'Предоставьте подтверждающие документы на рекламируемый товар / услугу'
                            ),
                            recommendation: [
                                licenseHelpText,
                                guaranteeLetter,
                                documentsTimeHelp
                            ],
                            links: [sendDocs, reasonsQuestion]
                        },
                        'irrelevant-keywords': {
                            title: iget2('b-moderation-reject', 'dobavte-ili-skorrektiruyte-klyuchevye', 'Добавьте или скорректируйте ключевые фразы'),
                            recommendation: [
                                list([
                                    getText('irrelevant-keywords', type),
                                    iget2(
                                        'b-moderation-reject',
                                        'klyuchevye-frazy-ne-nbsp',
                                        'Ключевые фразы не&nbsp;могут быть общими по&nbsp;значению («купить», «лето», «скидка»).'
                                    )
                                ]),
                                info(link(
                                    u.getCommonHelpUrl('/direct/quick-start/choose-keywords.xml'),
                                    iget2('b-moderation-reject', 'kak-naibolee-effektivno-podobrat', 'Как наиболее эффективно подобрать ключевые фразы')))
                            ],
                            links: [reasonsQuestion]
                        },
                        'russian-law-vcard': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-podrobnoe-opisanie',
                                'Убедитесь, что подробное описание товара (услуги) в&nbsp;виртуальной визитке соответствует действующему законодательству'
                            ),
                            recommendation: russianLawRec,
                            links: [reasonsQuestion]
                        },
                        'license-required-vcard': {
                            title: iget2(
                                'b-moderation-reject',
                                'predostavte-podtverzhdayushchie-dokumenty-na-131',
                                'Предоставьте подтверждающие документы на&nbsp;рекламируемый товар или услугу'
                            ),
                            recommendation: [
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'podrobnoe-opisanie-obekta-reklamirovaniya',
                                    'Подробное описание объекта рекламирования (например, товара или услуги) в&nbsp;вашей виртуальной визитке требует предоставления документов.'
                                )),
                                licenseHelpText,
                                guaranteeLetter,
                                documentsTimeHelp
                            ],
                            links: [sendDocs, reasonsQuestion]
                        },
                        'false-info-vcard': {
                            title: iget2(
                                'b-moderation-reject',
                                'isklyuchite-iz-nbsp-virtualnoy',
                                'Исключите из&nbsp;виртуальной визитки слова в&nbsp;превосходной или&nbsp;сравнительной степени'
                            ),
                            recommendation: [
                                p(iget2(
                                    'b-moderation-reject',
                                    'vy-mozhete-ispolzovat-slova',
                                    'Вы можете использовать слова в&nbsp;превосходной или сравнительной степени, только если на&nbsp;странице перехода размещено заключение третьих лиц о&nbsp;преимуществе вашего товара или услуги перед другими товарами.'
                                )),
                                p(iget2(
                                    'b-moderation-reject',
                                    'esli-u-vas-net-132',
                                    'Если у вас нет таких заключений, вы можете заменить слова в&nbsp;превосходной степени — например, «лучший», «новейший» — на&nbsp;фразы «один из&nbsp;лучших», «современный» и&nbsp;т.п.'
                                ))
                            ],
                            links: [falseInfoProofs]
                        },
                        'phone-doesn\'t-answer': {
                            title: iget2(
                                'b-moderation-reject',
                                'proverte-dostupnost-telefona-ukazannogo',
                                'Проверьте доступность телефона, указанного в&nbsp;виртуальной визитке'
                            ),
                            recommendation: [
                                p(iget2(
                                    'b-moderation-reject',
                                    'my-ne-nbsp-smogli',
                                    'Мы не&nbsp;смогли дозвониться по&nbsp;номеру из&nbsp;виртуальной визитки в&nbsp;обозначенное время работы.'
                                )),
                                p(iget2(
                                    'b-moderation-reject',
                                    'proverte-pravilno-nbsp-li',
                                    'Проверьте, правильно&nbsp;ли указаны рабочее время и&nbsp;номер телефона. При&nbsp;необходимости внесите изменения и&nbsp;отправьте объявление на&nbsp;повторную проверку.'
                                ))
                            ],
                            links: [
                                link(u.getCommonHelpUrl('/direct/troubleshooting/moderation.html#dissent'), iget2(
                                    'b-moderation-reject',
                                    'oshibki-ustraneny-nuzhna-povtornaya-100',
                                    'Ошибки устранены, нужна повторная проверка объявления'
                                ))
                            ]
                        },
                        'invalid-contacts': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-kontaktnye-dannye',
                                'Убедитесь, что контактные данные в&nbsp;виртуальной визитке указаны корректно'
                            ),
                            recommendation: [
                                list([
                                    [
                                        iget2(
                                            'b-moderation-reject',
                                            'telefon-dolzhen-byt-propisan',
                                            'Телефон должен быть прописан в&nbsp;формате: (код страны) (код города или мобильного оператора) (номер телефона).'
                                        ),
                                        '&nbsp;',
                                        iget2('b-moderation-reject', 'primer-7-916-123', 'Пример: +7 (916) 123-45-67.')
                                    ],
                                    iget2(
                                        'b-moderation-reject',
                                        'adres-nazvanie-organizacii-i',
                                        'Адрес, название организации и&nbsp;имя контактного лица должны быть написаны с&nbsp;заглавной буквы.'
                                    )
                                ])
                            ],
                            links: [reasonsQuestion]
                        },
                        'yandex-policy-auto': {
                            title: iget2('b-moderation-reject', 'reklama-tovara-uslugi-zapreshchena', 'Реклама товара/услуги запрещена'),
                            recommendation: [
                                list([
                                    [
                                        getText('yandex-policy-auto', type),
                                        iget2(
                                            'b-moderation-reject',
                                            'c-pravilami-razmeshcheniya-reklamy',
                                            'C правилами размещения рекламы на Яндексе вы можете ознакомиться {link}. ',
                                            {
                                                link: function(text) {
                                                    return link(iget2('b-moderation-reject', 'https-legal-yandex-ru', 'https://legal.yandex.ru/adv_rules/'), text);
                                                }(iget2('b-moderation-reject', 'c-pravilami-razmeshcheniya-reklamy-param-link', 'здесь')),

                                                context: 'link - здесь'
                                            }
                                        )
                                    ]
                                ]),
                                iget2('b-moderation-reject', 'link-obekta-reklamirovaniya-net', '{link}.', {
                                    link: function(text) {
                                        return link(u.getCommonHelpUrl('/direct/troubleshooting/error-ban.xml'), text);
                                    }(
                                        iget2('b-moderation-reject', 'link-obekta-reklamirovaniya-net-param-link', 'Объекта рекламирования нет в списке запрещённых')
                                    ),

                                    context: 'link - Объекта рекламирования нет в списке запрещённых'
                                })
                            ],
                            links: [reasonsQuestion]
                        },
                        'yandex-policy-vcard': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-podrobnoe-opisanie-133',
                                'Убедитесь, что подробное описание объекта рекламирования (товара или услуги) в&nbsp;виртуальной визитке соответствует рекламной политике Яндекса'
                            ),
                            recommendation: [
                                list([
                                    [
                                        iget2('b-moderation-reject', 'v-tekste-obyavleniya-chyotko', 'В тексте объявления чётко описан объект рекламирования.'),
                                        '&nbsp;',
                                        iget2('b-moderation-reject', 'naprimer', 'Например') + ':',
                                        list({ markers: 'none' }, [
                                            iget2(
                                                'b-moderation-reject',
                                                'kak-zarabotat-v-nbsp',
                                                '«Как заработать в&nbsp;интернете? Курс на&nbsp;DVD» — понятно, что именно рекламируется.'
                                            ),
                                            iget2(
                                                'b-moderation-reject',
                                                'kak-zarabotat-v-nbsp-134',
                                                '«Как заработать в&nbsp;интернете? Спешите узнать» — объект рекламирования не&nbsp;ясен.'
                                            )
                                        ])
                                    ],
                                    iget2(
                                        'b-moderation-reject',
                                        'esli-v-nbsp-obyavlenii',
                                        'Если в&nbsp;объявлении используется и&nbsp;виртуальная визитка, и&nbsp;ссылка на&nbsp;сайт, то информация о&nbsp;скидке, акции или о&nbsp;конкретном товаре (услуге) в&nbsp;них должна совпадать.'
                                    )
                                ]),
                                generalAdvRulesLink
                            ],
                            links: [reasonsQuestion]

                        },
                        blacklist: {
                            title: iget2(
                                'b-moderation-reject',
                                'proverte-korrektnost-telefona-ukazannogo',
                                'Проверьте корректность телефона, указанного в&nbsp;виртуальной визитке'
                            ),
                            recommendation: [
                                p(iget2(
                                    'b-moderation-reject',
                                    'my-ne-nbsp-smogli-135',
                                    'Мы не&nbsp;смогли дозвониться до вас по&nbsp;номеру из&nbsp;виртуальной визитки.  Вероятно, контактный телефон указан с ошибкой.'
                                )),
                                p(iget2(
                                    'b-moderation-reject',
                                    'vnesite-neobhodimye-izmeneniya-i',
                                    'Внесите необходимые изменения и&nbsp;отправьте объявление на&nbsp;повторную проверку.'
                                ))
                            ],
                            links: [reasonsQuestion]

                        },
                        'errors-vcard': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-v-nbsp-136',
                                'Убедитесь, что в&nbsp;подробном описании объекта рекламирования (товара/услуги) в&nbsp;виртуальной визитке нет опечаток'
                            ),
                            recommendation: [
                                iget2('b-moderation-reject', 'ne-dopuskaetsya', 'Не допускается') + ':',
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'pisat-imena-sobstvennye-so',
                                        'писать имена собственные со строчной буквы (в&nbsp;том числе в&nbsp;ключевых фразах, если вы используете шаблон);'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'ispolzovat-r-nbsp-a-137',
                                        'использовать р&nbsp;а&nbsp;з&nbsp;р&nbsp;я&nbsp;д&nbsp;к&nbsp;у в&nbsp;тексте объявления;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'pisat-slova-zaglavnymi-bukvami',
                                        'писать слова заглавными буквами (за&nbsp;исключением аббревиатур);'
                                    ),
                                    iget2('b-moderation-reject', 'ispolzovat-nekorrektnye-sokrashcheniya', 'использовать некорректные сокращения;'),
                                    iget2(
                                        'b-moderation-reject',
                                        'ispolzovat-mnogokratnye-povtory-odnih',
                                        'использовать многократные повторы одних и&nbsp;тех же слов или фраз.'
                                    )
                                ]),
                                generalAdvRulesLink
                            ],
                            links: [reasonsQuestion]
                        },
                        'irrelevant-image': {
                            title: getText('irrelevant-image-title', type),
                            recommendation: [
                                p(getText('irrelevant-image')),
                                link(u.getLegalUrl('/direct_adv_rules/'), iget2('b-moderation-reject', 'polnyy-spisok-trebovaniy-k', 'Полный список требований к&nbsp;изображениям'))
                            ],
                            links: [reasonsQuestion]
                        },
                        'yandex-policy-image': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-izobrazhenie-sootvetstvuet-138',
                                'Убедитесь, что изображение соответствует рекламной политике Яндекса'
                            ),
                            recommendation: [
                                iget2('b-moderation-reject', 'izobrazhenie-ne-nbsp-dolzhno', 'Изображение не&nbsp;должно содержать') + ':',
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'kontaktnuyu-informaciyu-naprimer-nomer',
                                        'контактную информацию (например, номер телефона, электронный адрес);'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'odnotonnuyu-zalivku-tekst-logotipy',
                                        'однотонную заливку, текст, логотипы или водяные знаки, которые занимают более 20% от&nbsp;всего изображения;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'elementy-interfeysa-kotorye-mogut',
                                        'элементы интерфейса, которые могут ввести пользователя в заблуждение;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'slishkom-otkrovennye-illyustracii-odezhdy',
                                        'слишком откровенные иллюстрации одежды и нижнего белья, фитнеса, салонов красоты, туристических и других услуг;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'izobrazheniya-kotorye-ustoychivo-associiruyutsya',
                                        'изображения, которые устойчиво ассоциируются с лженовостями и другим "жёлтым" контентом. Например, фото до/после, «проблемные» обнажённые части тела, откровенные позы и т.д.;'
                                    ),
                                    iget2('b-moderation-reject', 'intimnye-foto-foto-intimnyh', 'интимные фото/фото интимных товаров;')
                                ]),
                                notice(iget2(
                                    'b-moderation-reject',
                                    'etot-spisok-ne-nbsp',
                                    'Этот список не&nbsp;является полным. {rules} о требованиях к изображениям.',
                                    {
                                        rules: function(text) {
                                            return link(u.getLegalUrl('/direct_adv_rules/'), text);
                                        }(iget2('b-moderation-reject', 'etot-spisok-ne-nbsp-param-rules', 'Подробнее')),

                                        context: 'rules - Подробнее'
                                    }
                                ))
                            ],
                            links: [reasonsQuestion]
                        },
                        'poor-image': {
                            title: iget2('b-moderation-reject', 'zagruzite-v-nbsp-interfeys', 'Загрузите в&nbsp;интерфейс изображение с&nbsp;большим разрешением'),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'kachestvo-zagruzhennoy-kartinki-ochen',
                                'Качество загруженной картинки очень низкое. Такое изображение будет неинформативным для пользователей.'
                            ),
                            links: [reasonsQuestion]
                        },
                        'russian-law-image': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-izobrazhenie-ne',
                                'Убедитесь, что изображение не&nbsp;противоречит действующему законодательству'
                            ),
                            recommendation: [
                                iget2('b-moderation-reject', 'nelzya-reklamirovat', 'Нельзя рекламировать') + ':',
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'azartnye-igry-sportivnye-prognozy-139',
                                        'азартные игры, спортивные прогнозы, букмекерские конторы;'
                                    ),
                                    iget2('b-moderation-reject', 'alkogolnuyu-produkciyu-140', 'алкогольную продукцию;'),
                                    iget2('b-moderation-reject', 'tabachnuyu-produkciyu-i-nbsp-141', 'табачную продукцию и&nbsp;курительные принадлежности;'),
                                    iget2('b-moderation-reject', 'oruzhie-142', 'оружие;'),
                                    iget2(
                                        'b-moderation-reject',
                                        'drugie-rules-tovary-i',
                                        'другие {rules}, реклама которых запрещена или ограничена в&nbsp;интернете.',
                                        {
                                            rules: function(text) {
                                                return link(u.getLegalUrl('/adv_rules/'), text);
                                            }(iget2('b-moderation-reject', 'drugie-rules-tovary-i-param-rules', 'товары и&nbsp;услуги')),

                                            context: 'rules - товары и&nbsp;услуги'
                                        }
                                    )
                                ])
                            ],
                            links: [reasonsQuestion]
                        },
                        'russian-law-ql': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-nazvaniya-bystryh',
                                'Убедитесь, что названия быстрых ссылок не&nbsp;противоречат действующему законодательству'
                            ),
                            recommendation: russianLawRec,
                            links: [reasonsQuestion]
                        },
                        'ql-requirements': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-stranicy-na',
                                'Убедитесь, что страницы, на&nbsp;которые ведут быстрые ссылки, соответствуют требованиям Яндекса'
                            ),
                            recommendation: [
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'na-stranice-perehoda-prisutstvuet-143',
                                        'На странице перехода присутствует информация о&nbsp;скидке, акции или конкретном товаре (услуге), если эта информация есть в&nbsp;названии быстрой ссылки.'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'stranica-perehoda-otkryvaetsya-korrektno',
                                        'Страница перехода открывается корректно во всех браузерах.'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'reklamnye-bloki-na-nbsp',
                                        'Рекламные блоки на&nbsp;странице перехода занимают менее половины экрана.'
                                    ),
                                    iget2('b-moderation-reject', 'bystrye-ssylki-ne-nbsp', 'Быстрые ссылки не&nbsp;дублируют основную ссылку объявления.'),
                                    iget2(
                                        'b-moderation-reject',
                                        'na-stranice-perehoda-otsutstvuyut',
                                        'На странице перехода отсутствуют всплывающие окна (pop-up и&nbsp;pop-under).'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'sayt-predstavlen-na-nbsp',
                                        'Сайт представлен на&nbsp;языке рекламного объявления или в&nbsp;тексте объявления указан язык сайта.'
                                    )
                                ]),
                                generalAdvRulesLink
                            ],
                            links: [checkAgain]
                        },
                        'errors-ql': {
                            title: iget2('b-moderation-reject', 'ubedites-chto-v-nbsp-144', 'Убедитесь, что в&nbsp;названиях быстрых ссылок нет опечаток'),
                            recommendation: [
                                iget2('b-moderation-reject', 'ne-dopuskaetsya', 'Не допускается') + ':',
                                list([
                                    iget2('b-moderation-reject', 'pisat-imena-sobstvennye-so-145', 'писать имена собственные со строчной буквы;'),
                                    iget2('b-moderation-reject', 'ispolzovat-r-nbsp-a', 'использовать р&nbsp;а&nbsp;з&nbsp;р&nbsp;я&nbsp;д&nbsp;к&nbsp;у;'),
                                    iget2(
                                        'b-moderation-reject',
                                        'pisat-slova-zaglavnymi-bukvami',
                                        'писать слова заглавными буквами (за&nbsp;исключением аббревиатур);'
                                    ),
                                    iget2('b-moderation-reject', 'ispolzovat-nekorrektnye-sokrashcheniya', 'использовать некорректные сокращения;'),
                                    iget2(
                                        'b-moderation-reject',
                                        'ispolzovat-mnogokratnye-povtory-odnih',
                                        'использовать многократные повторы одних и&nbsp;тех же слов или фраз.'
                                    )
                                ]),
                                link(u.getCommonHelpUrl('/direct-tooltips/quick-links.xml'), iget2('b-moderation-reject', 'polnyy-spisok-trebovaniy', 'Полный список требований'))
                            ],
                            links: [reasonsQuestion]
                        },
                        'invalid-url-ql': {
                            title: iget2(
                                'b-moderation-reject',
                                'proverte-dostupny-li-stranicy',
                                'Проверьте, доступны ли страницы, на&nbsp;которые ведут быстрые ссылки'
                            ),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'pri-proverke-odna-ili',
                                'При проверке одна или все страницы сайта, на&nbsp;которые ведут быстрые ссылки, были недоступны.'
                            ),
                            links: [checkAgain]
                        },
                        'url-to-other-domain': {
                            title: iget2('b-moderation-reject', 'proverte-adresa-bystryh-ssylok', 'Проверьте адреса быстрых ссылок'),
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'ubedites-chto-bystrye-ssylki',
                                    'Убедитесь, что быстрые ссылки ведут на&nbsp;тот же сайт или раздел сайта, что и&nbsp;основная ссылка из&nbsp;рекламного объявления.'
                                ),
                                info(iget2(
                                    'b-moderation-reject',
                                    'isklyucheniem-yavlyayutsya-ssylki-na',
                                    'Исключением являются ссылки на&nbsp;страницу объекта рекламирования в&nbsp;социальных сетях.'
                                ))
                            ],
                            links: [reasonsQuestion]
                        },
                        'license-required-ql': {
                            title: iget2(
                                'b-moderation-reject',
                                'predostavte-podtverzhdayushchie-dokumenty-na-146',
                                'Предоставьте подтверждающие документы на&nbsp;рекламируемый товар (работу/услугу)'
                            ),
                            recommendation: [
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'bystrye-ssylki-vedut-na',
                                    'Быстрые ссылки ведут на&nbsp;страницы с&nbsp;товарами (работами/услугами), реклама которых может быть размещена только после того, как вы предоставите необходимый комплект подтверждающих документов (лицензия, сертификат, декларация, регистрационное удостоверение и&nbsp;т.п.).'
                                )),
                                licenseHelpText,
                                guaranteeLetter,
                                documentsTimeHelp

                            ],
                            links: [sendDocs, reasonsQuestion]
                        },
                        'irrelevant-mobile_content': {
                            title: iget2(
                                'b-moderation-reject',
                                'opisanie-prilozheniya-v-tekste',
                                'Описание приложения в тексте или теме объявления не соответствует описанию приложения в магазине.'
                            ),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-ssylka-s',
                                'Убедитесь, что ссылка с объявления ведет на соответствующее приложение в магазине.'
                            )
                        },
                        'irrelevant-site_link-description': {
                            title: iget2('b-moderation-reject', 'opisanie-bystroy-ssylki-ne', 'Описание быстрой ссылки не соответствует тексту объявления.'),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'opisanie-bystroy-ssylki-dolzhno',
                                'Описание быстрой ссылки должно соответствовать теме и тексту объявления, рекламируемому товару / услуге.'
                            )
                        },
                        'yandex-policy-callout': {
                            title: iget2('b-moderation-reject', 'po-politike-yandeksa', 'По политике Яндекса'),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'utochnenie-ne-sootvetstvuet-reklamnoy',
                                'Уточнение не соответствует рекламной политике Яндекса. Требования к рекламным материалам вы можете прочитать {link}.',
                                {
                                    link: function(text) {
                                        return link(u.getLegalUrl('/general_adv_rules/'), text);
                                    }(iget2('b-moderation-reject', 'utochnenie-ne-sootvetstvuet-reklamnoy-param-link', 'здесь')),

                                    context: 'link - здесь'
                                }
                            )
                        },
                        'errors-callout': {
                            title: iget2('b-moderation-reject', 'opechatki-v-tekste', 'Опечатки в тексте'),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'v-tekste-utochneniya-est',
                                'В тексте уточнения есть ошибки. Подробнее о причине отклонения вы можете прочитать {link}.',
                                {
                                    link: function(text) {
                                        return link(u.getCommonHelpUrl('direct-tooltips/moderation/errors-text.xml'), text);
                                    }(iget2('b-moderation-reject', 'v-tekste-utochneniya-est-param-link', 'здесь')),

                                    context: 'link - здесь'
                                }
                            )
                        },
                        'contact-info-callout': {
                            title: iget2('b-moderation-reject', 'kontakty-v-tekste', 'Контакты в тексте'),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'v-tekste-utochneniya-ukazany',
                                'В тексте уточнения указаны контактные данные. Подробнее о причине отклонения вы можете прочитать {link}.',
                                {
                                    link: function(text) {
                                        return link(
                                            u.getCommonHelpUrl('direct-tooltips/moderation/contacts-in-text.xml'),
                                            text
                                        );
                                    }(iget2('b-moderation-reject', 'v-tekste-utochneniya-ukazany-param-link', 'здесь')),

                                    context: 'link - здесь'
                                }
                            )
                        },
                        'false-info-callout': {
                            title: iget2('b-moderation-reject', 'lozhnaya-informaciya-v-tekste', 'Ложная информация в тексте'),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'v-tekste-utochneniya-est-147',
                                'В тексте уточнения есть слова в превосходной или сравнительной степени. Подробнее о причине отклонения вы можете прочитать {link}.',
                                {
                                    link: function(text) {
                                        return link(
                                            u.getCommonHelpUrl(
                                                'direct-tooltips/moderation/superlative-comparative-text.xml'),
                                            text
                                        );
                                    }(iget2('b-moderation-reject', 'v-tekste-utochneniya-est-147-param-link', 'здесь')),

                                    context: 'link - здесь'
                                }
                            )
                        },
                        'dhref-requirements': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-otobrazhaemaya-ssylka',
                                'Убедитесь, что отображаемая ссылка отвечает требованиям Яндекса'
                            ),
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'otobrazhaemaya-ssylka-dolzhna-sootvetstvovat',
                                    'Отображаемая ссылка должна соответствовать тексту объявления.'
                                ),
                                '&nbsp;',
                                iget2('b-moderation-reject', 'v-tekste-otobrazhaemoy-ssylki', 'В тексте отображаемой ссылки не допускаются') + ':',
                                list([
                                    iget2('b-moderation-reject', 'zhargonnye-grubye-prostorechnye-vyrazheniya', 'жаргонные, грубые, просторечные выражения;'),
                                    iget2('b-moderation-reject', 'slova-nabrannye-translitom-ili', 'слова, набранные транслитом или заглавными буквами;'),
                                    iget2(
                                        'b-moderation-reject',
                                        'opechatki-grammaticheskie-i-orfograficheskie',
                                        'опечатки, грамматические и орфографические ошибки;'
                                    ),
                                    iget2('b-moderation-reject', 'slova-v-sravnitelnoy-ili', 'слова в сравнительной или превосходной степени;'),
                                    iget2('b-moderation-reject', 'kontaktnye-dannye', 'контактные данные.')
                                ]),
                                generalAdvRulesLink
                            ],
                            links: [reasonsQuestion]
                        },
                        'license-required-dhref': {
                            title: iget2(
                                'b-moderation-reject',
                                'predostavte-podtverzhdayushchie-dokumenty-na',
                                'Предоставьте подтверждающие документы на рекламируемый товар / услугу'
                            ),
                            recommendation: [
                                licenseHelpText,
                                guaranteeLetter,
                                documentsTimeHelp
                            ],
                            links: [sendDocs, reasonsQuestion]
                        },
                        'tm-dhref': {
                            title: iget2('b-moderation-reject', 'predostavte-original-garantiynogo-pisma', 'Предоставьте оригинал гарантийного письма'),
                            recommendation: [tmRec],
                            links: [sendDocs]
                        },
                        'russian-law-dhref': {
                            title: iget2(
                                'b-moderation-reject',
                                'reklama-ne-sootvetstvuet-deystvuyushchemu',
                                'Реклама не соответствует действующему законодательству'
                            ),
                            recommendation: russianLawRec,
                            links: [reasonsQuestion]
                        },
                        'ad_image-bad-text': {
                            title: getText('ad_image-bad-text', type),
                            recommendation: [
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'tekst-dolzhen-svobodno-vosprinimatsya',
                                        'Текст должен свободно восприниматься пользователем без дополнительных действий или специальных средств.'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'vremya-pereklyucheniya-kadrov-na',
                                        'Время переключения кадров на анимированном изображении должно быть достаточным для прочтения всего показанного текста при средней скорости чтения 150 знаков в минуту.'
                                    )
                                ])
                            ]
                        },
                        'ad_image-no-border': {
                            title: getText('ad_image-no-border-title', type),
                            recommendation: [
                                p({ space: 'yes' }, getText('ad_image-no-border', type)),
                                p(generalAdvRulesLink)
                            ]
                        },
                        'ad_image-bad-quality': {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-zagruzhennoe-izobrazhenie',
                                'Убедитесь, что загруженное изображение качественное и чёткое'
                            ),
                            recommendation: [
                                list([
                                    getText('ad_image-bad-quality', type),
                                    iget2(
                                        'b-moderation-reject',
                                        'v-animacii-izobrazheniya-nelzya',
                                        'В анимации изображения нельзя использовать резко перемещающиеся и мелькающие графические элементы, мигающий фон.'
                                    )
                                ]),
                                p(generalAdvRulesLink)
                            ]
                        },
                        'ad_image-no-source': {
                            title: getText('ad_image-no-source-title', type),
                            recommendation: [
                                p({ space: 'yes' }, getText('ad_image-no-source', type)),
                                p(generalAdvRulesLink)
                            ]
                        },
                        'ad_image-nasty': {
                            title: getText('ad_image-nasty-title', type),
                            recommendation: [
                                iget2('b-moderation-reject', 'nelzya-ispolzovat-izobrazheniya-kotorye', 'Нельзя использовать изображения, которые'),
                                list({ listing: 'yes' }, [
                                    iget2(
                                        'b-moderation-reject',
                                        'mogut-byt-vosprinyaty-neodnoznachno',
                                        'могут быть восприняты неоднозначно, шокировать или вызывать негативные эмоции'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'soderzhat-naturalisticheskie-izobrazheniya-bolezney',
                                        'содержат натуралистические изображения болезней, медицинских процедур и операций'
                                    )
                                ])
                            ]
                        },
                        'ad_image-interface-element': {
                            title: getText('ad_image-interface-element-title', type),
                            recommendation: [
                                getText('ad_image-interface-element', type),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'elementy-interfeysa-ili-dizayna',
                                        'Элементы интерфейса или дизайна, которые могут ввести пользователя в заблуждение и не приводят к желаемому результату (например, кнопка купить, которая ведёт на главную страницу сайта, а не на страницу заказа конкретного товара).'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'elementy-dizayna-stranic-yandeksa',
                                        'Элементы дизайна страниц Яндекса (например, изображение поисковой строки, процесса ввода поискового запроса, кнопок, имитирующих интерфейс Яндекса).'
                                    )
                                ]),
                                p(generalAdvRulesLink)
                            ]
                        },
                        'ad_image-finance': {
                            title: getText('ad_image-finance-title', type),
                            recommendation: [
                                p({ title: 'yes' }, iget2('b-moderation-reject', 'cennye-bumagi', 'Ценные бумаги')),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'reklama-cennyh-bumag-dolzhna',
                                        'Реклама ценных бумаг должна содержать наименование лиц, обязавшихся по ценным бумагам и  источник информации, подлежащей раскрытию в соответствии с законодательством Российской Федерации о ценных бумагах.'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'reklama-emissionnyh-cennyh-bumag',
                                        'Реклама эмиссионных ценных бумаг должна содержать наименование эмитента и  источник информации, подлежащей раскрытию в соответствии с законодательством Российской Федерации о ценных бумагах.'
                                    )
                                ]),
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'informaciya-o-predostavitele-uslug',
                                    'Информация о предоставителе услуг должна быть визуально выделена (например, размером, шрифтом или подчёркиванием) и свободно восприниматься потребителем одновременно с рекламой без совершения дополнительных действий.'
                                )),
                                p(advRulesLink),

                                p({ title: 'yes' }, iget2('b-moderation-reject', 'bankovskie-strahovye-uslugi', 'Банковские, страховые услуги')),
                                p(iget2(
                                    'b-moderation-reject',
                                    'reklama-bankovskih-strahovyh-i',
                                    'Реклама банковских, страховых и иных финансовых услуг должна содержать наименование юридического лица (или ФИО индивидуального предпринимателя) и источник информации, подлежащей раскрытию в соответствии с законодательством Российской Федерации.'
                                )),
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'informaciya-o-predostavitele-uslug',
                                    'Информация о предоставителе услуг должна быть визуально выделена (например, размером, шрифтом или подчёркиванием) и свободно восприниматься потребителем одновременно с рекламой без совершения дополнительных действий.'
                                )),
                                p(advRulesLink),

                                p({ title: 'yes' }, iget2('b-moderation-reject', 'upravlenie-akciyami', 'Управление акциями')),
                                p(iget2(
                                    'b-moderation-reject',
                                    'reklama-po-upravleniyu-akciyami',
                                    'Реклама по управлению акциями, фондами, накоплениями и т.п. должна содержать:'
                                )),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'naimenovanie-yuridicheskogo-lica-ili',
                                        'Наименование юридического лица (или ФИО индивидуального предпринимателя).'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'istochnik-informacii-podlezhashchey-raskrytiyu',
                                        'Источник информации, подлежащей раскрытию в соответствии с законодательством Российской Федерации о ценных бумагах.'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'svedeniya-o-nomere-telefona',
                                        'Сведения о номере телефона, месте или адресе, где до заключения договора можно ознакомиться с условиями управления активами и получить иную информацию, предусмотренную законодательством.'
                                    )
                                ]),
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'informaciya-o-predostavitele-uslug',
                                    'Информация о предоставителе услуг должна быть визуально выделена (например, размером, шрифтом или подчёркиванием) и свободно восприниматься потребителем одновременно с рекламой без совершения дополнительных действий.'
                                )),
                                p(advRulesLink)
                            ]
                        },
                        'ad_image-lottery': {
                            title: getText('ad_image-lottery-title', type),
                            recommendation: [
                                getText('ad_image-lottery', type),
                                list([
                                    iget2('b-moderation-reject', 'sroki-provedeniya-meropriyatiya', 'Сроки проведения мероприятия.'),
                                    iget2(
                                        'b-moderation-reject',
                                        'istochnik-gde-mozhno-oznakomitsya',
                                        'Источник, где можно ознакомиться с информацией об организаторе мероприятия, правилах проведения, количестве призов или выигрышей, сроках, месте и порядке их получения.'
                                    )
                                ]),
                                link(u.getLegalUrl('/adv_rules/'), iget2('b-moderation-reject', 'polnyy-spisok-trebovaniy', 'Полный список требований'))
                            ]
                        },
                        'ad_image-share-building': {
                            title: iget2('b-moderation-reject', 'dolevoe-stroitelstvo', 'Долевое строительство'),
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'reklama-dolevogo-stroitelstva-dolzhna',
                                    'Реклама долевого строительства должна содержать наименование юридического лица (или ФИО индивидуального предпринимателя), а также сведения о месте и способах получения проектной декларации. Данная информация должна быть визуально выделена (например, размером, шрифтом или подчёркиванием) и свободно восприниматься потребителем одновременно с рекламой без совершения дополнительных действий.'
                                )
                            ]
                        },
                        'ad_image-distance': {
                            title: getText('ad_image-distance-title', type),
                            recommendation: [
                                p({ space: 'yes' }, getText('ad_image-distance', type)),
                                link(u.getLegalUrl('/adv_rules/'), iget2('b-moderation-reject', 'polnyy-spisok-trebovaniy', 'Полный список требований'))
                            ]
                        },
                        'ad_image-medicine': {
                            title: getText('ad_image-medicine-title', type),
                            recommendation: [
                                p({ space: 'yes' }, getText('ad_image-medicine-warning', type)),
                                p({ space: 'yes' }, getText('ad_image-medicine-bad', type)),
                                p(displayRulesLink)
                            ]
                        },
                        'ad_image-smi': {
                            title: getText('ad_image-smi-title', type),
                            recommendation: [
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'v-sootvetstvii-s-chastyu',
                                    'В соответствии с частью 10.1 статьи 5 Федерального закона «О рекламе» не допускается размещение рекламы информационной продукции без указания её возрастной категории.'
                                )),
                                p(iget2('b-moderation-reject', 'k-takoy-informacionnoy-produkcii', 'К такой информационной продукции относятся:')),
                                list([
                                    iget2('b-moderation-reject', 'bazy-dannyh-i-programmnoe', 'Базы данных и программное обеспечение.'),
                                    iget2('b-moderation-reject', 'filmy-igry-muzyka', 'Фильмы, игры, музыка.'),
                                    iget2(
                                        'b-moderation-reject',
                                        'koncerty-vystavki-festivali-spektakli',
                                        'Концерты, выставки, фестивали, спектакли, шоу, спортивные матчи и другие зрелищные мероприятия (но не сами театры, концертные залы и другие места, если в их рекламе не упоминается конкретное мероприятие).'
                                    ),
                                    iget2('b-moderation-reject', 'pechatnye-i-internet-izdaniya', 'Печатные и интернет-издания — газеты, журналы, книги.'),
                                    iget2(
                                        'b-moderation-reject',
                                        'konkretnye-programmy-i-peredachi',
                                        'Конкретные программы и передачи на ТВ и радио (но не сами телеканалы и радиостанции, если в их рекламе не упоминается конкретная передача).'
                                    )
                                ]),
                                p(displayRulesLink)
                            ]
                        },
                        'ad_image-babyfood': {
                            title: getText('ad_image-babyfood-title', type),
                            recommendation: [
                                p({ space: 'yes' }, getText('ad_image-babyfood', type)),
                                p(displayRulesLink)
                            ]
                        },
                        'ad_image-house-cooperative': {
                            title: getText('ad_image-house-cooperative-title', type),
                            recommendation: [
                                getText('ad_image-house-cooperative', type),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'naimenovanie-yuridicheskogo-lica-fio',
                                        'Наименование юридического лица/ФИО индивидуального предпринимателя.'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'informaciyu-o-poryadke-pokrytiya',
                                        'Информацию о порядке покрытия членами ЖНК понесённых ими убытков'
                                    ),
                                    iget2('b-moderation-reject', 'svedeniya-o-vklyuchenii-zhnk', 'Сведения о включении ЖНК в реестр ЖНК.'),
                                    iget2('b-moderation-reject', 'adres-sayta-na-kotorom', 'Адрес сайта, на котором осуществляется раскрытие информации.')
                                ])
                            ]
                        },
                        video_smi: {
                            title: iget2('b-moderation-reject', 'ubedites-chto-v-videorolike', 'Убедитесь, что в видеоролике указана возрастная категория'),
                            recommendation: [
                                p({ space: 'yes' },
                                    iget2(
                                        'b-moderation-reject',
                                        'v-obyavleniyah-s-targetingom',
                                        'В объявлениях с таргетингом на Россию, которые содержат рекламу, попадающую под действие закона ФЗ от 29 декабря 2010 года N 436-ФЗ "О защите детей от информации, причиняющей вред их здоровью и развитию", необходимо указывать возрастную категорию.'
                                    )
                                ),
                                iget2('b-moderation-reject', 'k-takoy-informacionnoy-produkcii', 'К такой информационной продукции относятся:'),
                                list([
                                    iget2('b-moderation-reject', 'bazy-dannyh-i-programmnoe-148', 'базы данных и программное обеспечение;'),
                                    iget2('b-moderation-reject', 'filmy-igry-muzyka-149', 'фильмы, игры, музыка;'),
                                    iget2(
                                        'b-moderation-reject',
                                        'koncerty-vystavki-festivali-spektakli-150',
                                        'концерты, выставки, фестивали, спектакли, шоу, спортивные матчи и другие зрелищные мероприятия (но не сами театры, концертные залы и другие места, если в их рекламе не упоминается конкретное мероприятие);'
                                    ),
                                    iget2('b-moderation-reject', 'pechatnye-i-internet-izdaniya-151', 'печатные и интернет-издания (газеты, журналы, книги);'),
                                    iget2(
                                        'b-moderation-reject',
                                        'konkretnye-programmy-i-peredachi-152',
                                        'конкретные программы и передачи на ТВ и радио (но не сами телеканалы и радиостанции, если в их рекламе не упоминается конкретная передача).'
                                    )
                                ]),
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'vremya-pokaza-vozrastnoy-kategorii',
                                    'Время показа возрастной категории должно быть достаточным для прочтения всего текста при средней скорости чтения 150 знаков в минуту.'
                                )),
                                p(advRulesLink)
                            ]
                        },
                        video_medicine: {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-videorolik-soderzhit',
                                'Убедитесь, что видеоролик содержит необходимое предупреждение'
                            ),
                            recommendation: [
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'videorolik-o-lekarstvennyh-sredstvah',
                                    'Видеоролик о лекарственных средствах, медицинской технике, медицинских услугах и методах лечения должен содержать предупреждение о наличии противопоказаний, необходимости ознакомления с инструкцией по применению или получения консультации специалиста. Предупреждение должно показываться на протяжении всего времени и занимать не менее 5% площади видеоролика.'
                                )),
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'videorolik-s-reklamoy-bad',
                                    'Видеоролик с рекламой БАД должен содержать предупреждение о том, что товар не является лекарством. Предупреждение должно показываться на протяжении всего времени и занимать не менее 10% площади видеоролика.'
                                )),
                                p(advRulesLink)
                            ]
                        },
                        video_house_cooperative: {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-videorolik-soderzhit-153',
                                'Убедитесь, что видеоролик содержит информацию о жилищно-накопительном кооперативе'
                            ),
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'videorolik-s-reklamoy-zhilishchno',
                                    'Видеоролик с рекламой жилищно-накопительного кооператива (ЖНК) должен содержать:'
                                ),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'naimenovanie-yuridicheskogo-lica-fio-154',
                                        'наименование юридического лица/ФИО индивидуального предпринимателя;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'informaciyu-o-poryadke-pokrytiya-155',
                                        'информацию о порядке покрытия членами ЖНК понесённых им убытков;'
                                    ),
                                    iget2('b-moderation-reject', 'svedeniya-o-vklyuchenii-zhnk-156', 'сведения о включении ЖНК в реестр ЖНК;'),
                                    iget2('b-moderation-reject', 'adres-sayta-na-kotorom-157', 'адрес сайта, на котором осуществляется раскрытие информации.')
                                ]),
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'vremya-pokaza-dannoy-informacii',
                                    'Время показа данной информации должно быть достаточным для прочтения всего текста при средней скорости чтения 150 знаков в минуту.'
                                )),
                                p(generalAdvRulesLink)
                            ]
                        },
                        video_lottery: {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-videorolik-soderzhit-158',
                                'Убедитесь, что видеоролик содержит необходимую информацию о мероприятии'
                            ),
                            recommendation: [
                                iget2(
                                    'b-moderation-reject',
                                    'videorolik-s-reklamoy-stimuliruyushchego',
                                    'Видеоролик с рекламой  стимулирующего мероприятия (например, лотереи или конкурса) должен содержать:  '
                                ),
                                list([
                                    iget2('b-moderation-reject', 'sroki-provedeniya-meropriyatiya-159', 'сроки проведения мероприятия;'),
                                    iget2(
                                        'b-moderation-reject',
                                        'istochnik-gde-mozhno-oznakomitsya-160',
                                        'источник, где можно ознакомиться с информацией об организаторе мероприятия, правилах проведения, количестве призов или выигрышей, сроках, месте и порядке их получения.'
                                    )
                                ]),
                                p(advRulesLink)
                            ]
                        },
                        video_share_building: {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-videorolik-soderzhit-161',
                                'Убедитесь, что видеоролик содержит информацию о долевом строительстве'
                            ),
                            recommendation: [
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'reklama-dolevogo-stroitelstva-dolzhna',
                                    'Реклама долевого строительства должна содержать наименование юридического лица (или ФИО индивидуального предпринимателя), а также сведения о месте и способах получения проектной декларации. Данная информация должна быть визуально выделена (например, размером, шрифтом или подчёркиванием) и свободно восприниматься потребителем одновременно с рекламой без совершения дополнительных действий.'
                                )),
                                p(advRulesLink)
                            ]
                        },
                        video_finance: {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-videorolik-soderzhit-162',
                                'Убедитесь, что видеоролик содержит достаточную информацию о предоставителе услуг'
                            ),
                            recommendation: [
                                list([
                                    {
                                        block: 'b-cut',
                                        content: [
                                            {
                                                block: 'link',
                                                mods: {
                                                    pseudo: 'yes'
                                                },
                                                mix: [{
                                                    block: 'b-cut',
                                                    elem: 'link'
                                                }],
                                                content: iget2('b-moderation-reject', 'cennye-bumagi', 'Ценные бумаги')
                                            },
                                            {
                                                block: 'b-cut',
                                                elem: 'content',
                                                content: [
                                                    list([
                                                        iget2(
                                                            'b-moderation-reject',
                                                            'reklama-cennyh-bumag-dolzhna-163',
                                                            'реклама ценных бумаг должна содержать наименование лиц, обязавшихся по ценным бумагам и источник информации, подлежащей раскрытию в соответствии с законодательством Российской Федерации о ценных бумагах;'
                                                        ),
                                                        iget2(
                                                            'b-moderation-reject',
                                                            'reklama-emissionnyh-cennyh-bumag-164',
                                                            'реклама эмиссионных ценных бумаг должна содержать наименование эмитента и источник информации, подлежащей раскрытию в соответствии с законодательством Российской Федерации о ценных бумагах.'
                                                        )
                                                    ]),
                                                    p({ space: 'yes' }, iget2(
                                                        'b-moderation-reject',
                                                        'informaciya-o-predostavitele-uslug',
                                                        'Информация о предоставителе услуг должна быть визуально выделена (например, размером, шрифтом или подчёркиванием) и свободно восприниматься потребителем одновременно с рекламой без совершения дополнительных действий.'
                                                    )),
                                                    p({ space: 'yes' }, advRulesLink)
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        block: 'b-cut',
                                        content: [
                                            {
                                                block: 'link',
                                                mods: {
                                                    pseudo: 'yes'
                                                },
                                                mix: [{
                                                    block: 'b-cut',
                                                    elem: 'link'
                                                }],
                                                content: iget2('b-moderation-reject', 'bankovskie-strahovye-uslugi', 'Банковские, страховые услуги')
                                            },
                                            {
                                                block: 'b-cut',
                                                elem: 'content',
                                                content: [
                                                    p({ space: 'yes' }, iget2(
                                                        'b-moderation-reject',
                                                        'reklama-bankovskih-strahovyh-i',
                                                        'Реклама банковских, страховых и иных финансовых услуг должна содержать наименование юридического лица (или ФИО индивидуального предпринимателя) и источник информации, подлежащей раскрытию в соответствии с законодательством Российской Федерации.'
                                                    )),
                                                    p({ space: 'yes' }, iget2(
                                                        'b-moderation-reject',
                                                        'informaciya-o-predostavitele-uslug',
                                                        'Информация о предоставителе услуг должна быть визуально выделена (например, размером, шрифтом или подчёркиванием) и свободно восприниматься потребителем одновременно с рекламой без совершения дополнительных действий.'
                                                    )),
                                                    p({ space: 'yes' }, advRulesLink)
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        block: 'b-cut',
                                        content: [
                                            {
                                                block: 'link',
                                                mods: {
                                                    pseudo: 'yes'
                                                },
                                                mix: {
                                                    block: 'b-cut',
                                                    elem: 'link'
                                                },
                                                content: iget2('b-moderation-reject', 'upravlenie-akciyami', 'Управление акциями')
                                            },
                                            {
                                                block: 'b-cut',
                                                elem: 'content',
                                                content: [
                                                    p({ space: 'yes' }, iget2(
                                                        'b-moderation-reject',
                                                        'reklama-po-upravleniyu-akciyami',
                                                        'Реклама по управлению акциями, фондами, накоплениями и т.п. должна содержать:'
                                                    )),
                                                    list([
                                                        iget2(
                                                            'b-moderation-reject',
                                                            'naimenovanie-yuridicheskogo-lica-ili-165',
                                                            'наименование юридического лица (или ФИО индивидуального предпринимателя);'
                                                        ),
                                                        iget2(
                                                            'b-moderation-reject',
                                                            'istochnik-informacii-podlezhashchey-raskrytiyu-166',
                                                            'источник информации, подлежащей раскрытию в соответствии с законодательством Российской Федерации о ценных бумагах;'
                                                        ),
                                                        iget2(
                                                            'b-moderation-reject',
                                                            'svedeniya-o-nomere-telefona-167',
                                                            'сведения о номере телефона, месте или адресе, где до заключения договора можно ознакомиться с условиями управления активами и получить иную информацию, предусмотренную законодательством.'
                                                        )
                                                    ]),
                                                    p({ space: 'yes' }, iget2(
                                                        'b-moderation-reject',
                                                        'informaciya-o-predostavitele-uslug',
                                                        'Информация о предоставителе услуг должна быть визуально выделена (например, размером, шрифтом или подчёркиванием) и свободно восприниматься потребителем одновременно с рекламой без совершения дополнительных действий.'
                                                    )),
                                                    p({ space: 'yes' }, advRulesLink)
                                                ]
                                            }
                                        ]
                                    }
                                ])
                            ]
                        },
                        video_distance_sales: {
                            title: iget2('b-moderation-reject', 'ubedites-chto-videorolik-soderzhit-168', 'Убедитесь, что видеоролик содержит данные о продавце'),
                            recommendation: [
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'videorolik-s-reklamoy-distancionnoy',
                                    'Видеоролик с рекламой дистанционной продажи товаров должен содержать наименование юридического лица/ФИО индивидуального предпринимателя, его адрес и ОГРН. Время показа данной информации должно быть достаточным для прочтения всего текста при средней скорости чтения 150 знаков в минуту.'
                                )),
                                p(advRulesLink)
                            ]
                        },
                        video_babyfood: {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-videorolik-sootvetstvuet',
                                'Убедитесь, что видеоролик соответствует требованиям к рекламе детского питания'
                            ),
                            recommendation: [
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'videorolik-s-reklamoy-produktov',
                                    'Видеоролик с рекламой продуктов, включенных в рацион ребенка в течение его первого года жизни, или заменителей женского молока должен содержать возрастное ограничение применения и предупреждение о необходимости консультации специалиста. Эта информация должна показываться на протяжении всего времени.'
                                )),
                                p(advRulesLink)
                            ]
                        },
                        video_requirements: {
                            title: iget2(
                                'b-moderation-reject',
                                'ubedites-chto-videorolik-sootvetstvuet-169',
                                'Убедитесь, что видеоролик соответствует требованиям'
                            ),
                            recommendation: [
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'videorolik-dolzhen-byt-kachestvennym',
                                    'Видеоролик должен быть качественным и чётким, без артефактов сжатия. В анимации видеоролика нельзя использовать резко перемещающиеся и мелькающие графические элементы, мигающий фон, изображения с гипнотическим эффектом.'
                                )),
                                p(iget2('b-moderation-reject', 'videorolik-ne-dolzhen-soderzhat', 'Видеоролик не должен содержать:')),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'elementy-interfeysa-ili-dizayna-170',
                                        'элементы интерфейса или дизайна, которые могут ввести пользователя в заблуждение и не приводят к желаемому результату;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'obrazy-kotorye-ne-sootvetstvuyut',
                                        'образы, которые не соответствуют заголовку, тексту объявления и содержанию страницы перехода;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'izobrazheniya-kotorye-mogut-byt',
                                        'изображения, которые могут быть восприняты неоднозначно, шокировать или вызвать негативные эмоции;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'naturalisticheskie-izobrazheniya-bolezney-medicinskih',
                                        'натуралистические изображения болезней, медицинских процедур и операций, военных действий.'
                                    )
                                ]),
                                p(generalAdvRulesLink)
                            ]
                        },
                        audio_requirements: {
                            title: iget2('b-moderation-reject', 'ubedites-chto-audiorolik-sootvetstvuet', 'Убедитесь, что аудиоролик соответствует требованиям'),
                            recommendation: [
                                p(iget2('b-moderation-reject', 'proverte-chto-audiorolik', 'Проверьте, что аудиоролик:')),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'gramotnyy-kachestvennyy-i-komfortnyy',
                                        'грамотный, качественный и комфортный для восприятия, без помех, резких или слишком громких звуков;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'sootvetstvuet-zagolovku-tekstu-obyavleniya',
                                        'соответствует заголовку, тексту объявления и содержанию страницы перехода;'
                                    ),
                                    iget2('b-moderation-reject', 'sinhronnyy-s-videorolikom', 'синхронный с видеороликом.')
                                ]),
                                p({ space: 'yes' }, iget2(
                                    'b-moderation-reject',
                                    'ne-ispolzuyte-v-audiorolike',
                                    'Не используйте в аудиоролике звуки, которые имитируют дорожное движение и могут навредить участникам дорожного движения (звуки сирен, клаксонов, визг тормозов и т.п.).'
                                )),
                                p(generalAdvRulesLink)
                            ]
                        },
                        'turbolanding-requirements': {
                            title: iget2('b-moderation-reject', 'turbolanding-requirements-title', 'Требования к турбо-странице'),
                            recommendation: [
                                p(iget2('b-moderation-reject', 'turbolanding-requirements-1', 'Убедитесь, что турбо-страница соответствует требованиям Яндекса:')),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'turbolanding-requirements-2',
                                        'содержание страницы соответствует рекламному тексту (например, если в объявлении есть информация о конкретном товаре, акции, скидке и т.п., то эта же информация должна присутствовать на странице перехода);'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'turbolanding-requirements-3',
                                        'не предлагаются {special-categories-link};',
                                        {
                                            'special-categories-link': function(text) {
                                                return link(u.getCommonHelpUrl('/direct/moderation/restricted-categories.html'), text);
                                            }(iget2('b-moderation-reject', 'special-categories-link-turbo', 'запрещенные товары или услуги'))
                                        }
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'turbolanding-requirements-4',
                                        'рекламные блоки занимают менее половины экрана;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'turbolanding-requirements-5',
                                        'страница составлена на языке рекламного объявления или в объявлении указан язык сайта.'
                                    )
                                ])
                            ],
                            links: [generalAdvRulesLink, checkAgain]
                        },
                        'turbolanding-requirements-slink': {
                            title: iget2('b-moderation-reject', 'turbolanding-requirements-slink-title', 'Требования к турбо-странице'),
                            recommendation: [
                                p(iget2('b-moderation-reject', 'turbolanding-requirements-slink-1', 'Убедитесь, что турбо-страница соответствует требованиям Яндекса:')),
                                list([
                                    iget2(
                                        'b-moderation-reject',
                                        'turbolanding-requirements-slink-2',
                                        'содержание страниц соответствует рекламному тексту (например, если в быстрой ссылке есть информация о конкретном товаре, акции, скидке и т.п., то эта же информация должна присутствовать на странице перехода);'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'turbolanding-requirements-slink-3',
                                        'не предлагаются {special-categories-link};',
                                        {
                                            'special-categories-link': function(text) {
                                                return link(u.getCommonHelpUrl('/direct/moderation/restricted-categories.html'), text);
                                            }(iget2('b-moderation-reject', 'special-categories-link-turbo', 'запрещенные товары или услуги'))
                                        }
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'turbolanding-requirements-slink-4',
                                        'рекламные блоки занимают менее половины экрана;'
                                    ),
                                    iget2(
                                        'b-moderation-reject',
                                        'turbolanding-requirements-slink-5',
                                        'страницы составлены на языке рекламного объявления или в названии быстрых ссылок указан язык сайта;'
                                    )
                                ])
                            ],
                            links: [generalAdvRulesLink, checkAgain]
                        },
                        other: { title: iget2('b-moderation-reject', 'drugie-prichiny', 'Другие причины') }, //В этой причине текст приходит из модерации,
                        'html5-nasty': {
                            title: iget2('b-moderation-reject', 'html5-nasty-title', 'Неприятное изображение'),
                            recommendation: iget2('b-moderation-reject', 'html5-nasty-recommendation', 'Использовано изображение, которое может быть воспринято неоднозначно, шокировать или вызывать негативные эмоции')
                        },
                        'html5-babyfood': {
                            title: iget2('b-moderation-reject', 'html5-babyfood-title', 'Детское питание'),
                            recommendation: iget2('b-moderation-reject', 'html5-babyfood-recommendation', 'Специальные требования к рекламе детского питания')
                        },
                        'html5-finance': {
                            title: iget2('b-moderation-reject', 'html5-finance-title', 'Финансы'),
                            recommendation: iget2('b-moderation-reject', 'html5-finance-recommendation', 'Специальные требования к рекламе финансовых услуг')
                        },
                        'html5-funds': {
                            title: iget2('b-moderation-reject', 'html5-funds-title', 'Фонды'),
                            recommendation: iget2('b-moderation-reject', 'html5-funds-recommendation', 'Специальные требования к рекламе управления акциями, фондами, накоплениями')
                        },
                        'html5-bad-quality': {
                            title: iget2('b-moderation-reject', 'html5-bad-quality-title', 'Плохое качество'),
                            recommendation: iget2('b-moderation-reject', 'html5-bad-quality-recommendation', 'Качество изображения должно быть высоким, не допускаются яркие цвета, мелькающие кадры')
                        },
                        'html5-russian-law': {
                            title: iget2('b-moderation-reject', 'html5-russian-law-title', 'По закону России'),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'html5-russian-law-recommendation',
                                'Реклама не соответствует действующему законодательству. Подробнее о причине отклонения вы можете прочитать {link}.',
                                {
                                    link: function(text) {
                                        return link(
                                            u.getCommonHelpUrl(
                                                'direct-tooltips/moderation/law.html'),
                                            text
                                        );
                                    }(iget2('b-moderation-reject', 'v-tekste-utochneniya-est-147-param-link', 'здесь')),

                                    context: 'link - здесь'
                                }
                            )
                        },
                        'html5-illness': {
                            title: iget2('b-moderation-reject', 'html5-illness-title', 'Болезнь'),
                            recommendation: iget2('b-moderation-reject', 'html5-illness-recommendation', 'Не допускается натуралистическое изображение болезни.')
                        },
                        'html5-some-hrefs-not-declared': {
                            title: iget2('b-moderation-reject', 'html5-some-hrefs-not-declared-title', 'Отсутствуют некоторые ссылки'),
                            recommendation: iget2('b-moderation-reject', 'html5-some-hrefs-not-declared-recommendation', '')
                        },
                        'html5-spam': {
                            title: iget2('b-moderation-reject', 'html5-spam-title', 'NDA. Пункт 15'),
                            recommendation: iget2(
                                'b-moderation-reject',
                                'html5-spam-recommendation',
                                'Рекламные материалы не могут быть размещены в Директе. Подробнее о причине отклонения вы можете прочитать {link}.',
                                {
                                    link: function(text) {
                                        return link(
                                            u.getCommonHelpUrl(
                                                'direct-tooltips/moderation/no-comments.html'),
                                            text
                                        );
                                    }(iget2('b-moderation-reject', 'v-tekste-utochneniya-est-147-param-link', 'здесь')),

                                    context: 'link - здесь'
                                }
                            )
                        },
                        'html5-house-cooperative': {
                            title: iget2('b-moderation-reject', 'html5-house-cooperative-title', 'Кооператив'),
                            recommendation: iget2('b-moderation-reject', 'html5-house-cooperative-recommendation', 'Специальные требования к рекламе жилищного кооператива')
                        },
                        'html5-interface-element': {
                            title: iget2('b-moderation-reject', 'html5-interface-element-title', 'Элементы интерфейса'),
                            recommendation: iget2('b-moderation-reject', 'html5-interface-element-recommendation', 'В баннере не допускаются нефункциональные элементы интерфейса и элементы дизайна страниц Яндекса.')
                        },
                        'html5-no-border': {
                            title: iget2('b-moderation-reject', 'html5-no-border-title', 'Нет границ'),
                            recommendation: iget2('b-moderation-reject', 'html5-no-border-recommendation', 'Рекламный баннер должен иметь видимые границы')
                        },
                        'html5-hrefs-errors': {
                            title: iget2('b-moderation-reject', 'html5-hrefs-errors-title', 'Убедитесь, что ссылка с баннера задана верно'),
                            recommendation: [
                                iget2('b-moderation-reject', 'html5-hrefs-errors-recommendation-list-title', 'В соответствии с требованиями') + ':',
                                list([
                                    iget2('b-moderation-reject', 'html5-hrefs-errors-recommendation-list-item1', 'Открытие рекламируемого сайта должно происходить в новом окне, только по клику пользователя левой кнопки мыши на баннер.'),
                                    [
                                        iget2('b-moderation-reject', 'html5-hrefs-errors-recommendation-list-item2', 'Переход на страницу, указанную в поле «Ссылка на сайт» на странице создания или редактирования группы объявлений, должен генерироваться в баннере вызовом метода yandexHTML5BannerApi.getClickNum(1).'),
                                        p({ space: 'yes' }, ' '),
                                        p({ space: 'yes' }, {
                                            block: 'b-moderation-reject',
                                            elem: 'code',
                                            text: '<a id="click_area" href="#" target="_blank">...</a>\n' +
                                            '// after the "<a href...></a>" element\n' +
                                            '// assign them the href attribute values as follows:\n' +
                                            '<script>\n' +
                                            '   document.getElementById("click_area").href = yandexHTML5BannerApi.getClickURLNum(1);\n' +
                                            '</script>'
                                        })
                                    ]
                                ]),
                                p(html5RequirementsLink)
                            ]
                        },
                        'html5-no-source': {
                            title: iget2('b-moderation-reject', 'html5-no-source-title', 'Нет источника'),
                            recommendation: iget2('b-moderation-reject', 'html5-no-source-recommendation', 'Нет четкого представления, кто является источником информации')
                        },
                        'html5-smi': {
                            title: iget2('b-moderation-reject', 'html5-smi-title', 'Защита детей'),
                            recommendation: iget2('b-moderation-reject', 'html5-smi-recommendation', 'Специальные требования к рекламе информационной продукции')
                        },
                        'html5-securities': {
                            title: iget2('b-moderation-reject', 'html5-securities-title', 'Ценные бумаги'),
                            recommendation: iget2('b-moderation-reject', 'html5-securities-recommendation', 'Специальные требования к рекламе ценных бумаг')
                        },
                        'html5-lottery': {
                            title: iget2('b-moderation-reject', 'html5-lottery-title', 'Стим.лотерея'),
                            recommendation: iget2('b-moderation-reject', 'html5-lottery-recommendation', 'Специальные требования к рекламе стимулирующих мероприятий')
                        },
                        'html5-share-building': {
                            title: iget2('b-moderation-reject', 'html5-share-building-title', 'Долевое строительство'),
                            recommendation: iget2('b-moderation-reject', 'html5-share-building-recommendation', 'Специальные требования к рекламе долевого строительства')
                        },
                        'html5-bad-text': {
                            title: iget2('b-moderation-reject', 'html5-bad-text-title', 'Нечитаемый текст'),
                            recommendation: iget2('b-moderation-reject', 'html5-bad-text-recommendation', 'Текст должен свободно восприниматься пользователем')
                        },
                        'html5-medicine': {
                            title: iget2('b-moderation-reject', 'html5-medicine-title', 'Медицина'),
                            recommendation: iget2('b-moderation-reject', 'html5-medicine-recommendation', 'Специальные требования к рекламе лекарственных средств, медицинских услуг, в том числе методов лечения, медицинской техники')
                        }
                    };

                return {
                    verdicts: verdicts,
                    commonLinks: [sendDocs, reasonsQuestion]
                };
            }

        }

    });

})();
