BEM.DOM.decl({ block: 'b-online-set-phrases-prices' }, {
    onSetMod: {
        js: function() {
            this._subscriptionManager = BEM.create('i-subscription-manager');

            //модель кампании связанная с конструктором ставок
            this._campModel = BEM.MODEL.getOrCreate(this.params.modelParams);

            u.graspSelf.call(this, {
                _pricesConstructor: 'b-prices-constructor inside',
                _saveBtn: 'button on save',
                _spin: 'spin on spin'
            });

            // @anyakey для DIRECT-61961. Хинты на кнопках с предупреждением, что для некоторых групп
            // ставки выставлены не будут
            this._initHints();
            this._initEvents();

            BEM.blocks['b-metrika2'].params({
                params: {
                    'inline-prices-constructor': {
                        show: true
                    }
                }
            });
        },

        progress: function(modName, modVal) {
            this._saveBtn.setMod('disabled', modVal ? 'yes' : '');
            this._spin.setMod('progress', modVal ? 'yes' : '');
        }
    },

    /**
    * Удаляет блок и подписки на события
    */
    destruct: function() {
        this._subscriptionManager.dispose();
        this._subscriptionManager.destruct();

        return this.__base.apply(this, arguments);
    },

    /**
     * Инициализация хинта "мало показов"
     * @private
     */
    _initHints: function() {
        var hintable = this.findBlocksInside('b-hintable'),
            platform = this._campModel.get('platform'),
            mediaType = this._campModel.get('mediaType'),
            url = {
                dynamic: 'rarely-served-dynamic-text-ads',
                performance: 'rarely-served-smart-banners'
            }[mediaType] || 'rarely-served-text-ads';

        hintable && hintable.forEach(function(block) {
            block.setHintContent(
                iget2(
                    'b-online-set-phrases-prices',
                    'v-nekotoryh-gruppah-na',
                    'В некоторых группах на странице мало показов. Ставки для них установлены не будут. {link}',
                    {
                        link: function(text) {
                            return BEMHTML.apply({
                                block: 'b-help-link',
                                mods: { type: 'modal' },
                                stretched: true,
                                url: u.getHelpUrl(url),
                                text: text
                            })
                        }(iget2('b-online-set-phrases-prices', 'v-nekotoryh-gruppah-na-param-link', 'Подробнее')),

                        context: 'link - Подробнее'
                    }
                )
            );
            block.setMod('disabled', platform == 'context' ? 'yes' : 'no');
        });
    },

    /**
     * Инициализация событий блока
     * @private
     */
    _initEvents: function() {
        this._subscriptionManager.on(this._campModel, 'platform', 'change', this._onPlatformChange, this);
        this._subscriptionManager.on(this._saveBtn, 'click', this._onSave, this);
    },

    /**
     * Обработчик события нажатия на кнопку 'Назначить'
     */
    _onSave: function() {
        var _this = this,
            campModel = this._campModel,
            platform = campModel.get('platform'),
            value = this._getValue(
                campModel.get('strategy'),
                platform,
                this.params.showBothPlatforms
            ),
            visitParams = { use: true };

        if (!this._validate()) {
            return;
        }

        // состояние загрузки устанавливаем, так как при большом количестве фраз обновление разметки
        // занимает заметное количество времени
        this.setMod('progress', 'yes');

        // setTimeout, чтобы успело установиться состояние загрузки до начала сложных операций вычислений
        setTimeout(function() {
            BEM.channel('direct-feeds').trigger('initFeeds');
            _this._initGroupControls();
            _this._updatePrices(value);
            _this.delMod('progress');
        }, 500);

        // DIRECT-79202 логируем изменение ставок для трафаретных торгов
        if (campModel.get('mediaType') === 'text' && platform === 'search') {
            visitParams['traffic-volume'] = {};
            visitParams['traffic-volume'][value.position_ctr_correction] = value;
        }

        BEM.blocks['b-metrika2'].params({
            params: {
                'inline-prices-constructor': visitParams
            }
        });
    },

    /**
     * Смена плафтормы
     * @param {Event} e - событие
     * @param {Object} data - данные события
     * @private
     */
    _onPlatformChange: function(e, data) {
        var hintable = this.findBlocksInside('b-hintable');

        this._disablePlace(data.value, false);
        this._disablePlace(data.value == 'context' ? 'search' : 'context', true);
        this.setMod('platform', data.value);

        hintable && hintable.forEach(function(block) {
            block.setMod('disabled', data.value == 'context' ? 'yes' : 'no');
        })
    },

    /**
     * Делает конструктор ставок недоступным на поиске/контексте
     * @param {'search'|'context'} place
     * @param {Boolean} isDisabled
     * @returns {BEM}
     */
    _disablePlace: function(place, isDisabled) {
        this._pricesConstructor.disablePlace(place, isDisabled);

        return this;
    },

    /**
     * Возвращает текущие значения контролов конструктора ставок
     * @param {Object} strategy - стратегия показа
     * @param {'search'|'context'} platform - платформа показа
     * @param {Boolean} showBothPlatforms - флаг, что надо показвать обе платформы
     * @returns {Object}
     */
    _getValue: function(strategy, platform, showBothPlatforms) {
        return this._filterData(this._pricesConstructor.getData(), strategy, platform, showBothPlatforms);
    },

    /**
     * Инициализирует блоки фильтров и их контролы
     * @returns {BEM.DOM}
     */
    _initGroupControls: function() {
        return BEM.blocks['b-groups-list'].initPhrasesLists();
    },

    /**
     * Изменяет ставки на кампании в соответствии с установленными в конструкторе ставок параметрами
     * @param {Object} value - параметры конструктора ставок
     * @returns {BEM}
     */
    _updatePrices: function(value) {
        BEM.blocks['i-utils'].campOptions.set('price_editor', JSON.stringify(value), function() {});

        var currency = this._campModel.get('currency'),
            strategy = this._campModel.get('strategy_name'),
            groupDMName = this._campModel.getChildGroupModelName();

        if (strategy == 'different_places') {
            if (this._isSearchPlace()) {
                this.__self._updateSearchPrices(
                    this._campModel.get('mediaType') === 'mcbanner' ? 'firstPage' : 'interpolation',
                    { name: 'm-phrase-bidable', parentName: groupDMName, parentId: '*' },
                    value,
                    currency,
                    this.params.hasRarelyUsedGroups
                );
            }
            if (this._isContextPlace()) {
                this.__self._updateContextPrices(
                    { name: 'm-phrase-bidable', parentName: groupDMName, parentId: '*' },
                    value,
                    strategy,
                    currency);
            }
        } else {
            this.__self._updateSearchPrices(
                this._campModel.get('mediaType') === 'mcbanner' ? 'firstPage' : 'interpolation',
                { name: 'm-phrase-bidable', parentName: groupDMName, parentId: '*' },
                value,
                currency,
                this.params.hasRarelyUsedGroups
            );
        }

        return this;
    },

    /**
     * только для стратегии different_places
     * выбрано размещение "на поиске"
     * @returns {Boolean}
     */
    _isSearchPlace: function() {
        return this._pricesConstructor.isSearchPlace();
    },

    /**
     * только для стратегии different_places
     * выбрано размещение "на контекстных площадках"
     * @returns {Boolean}
     */
    _isContextPlace: function() {
        return this._pricesConstructor.isContextPlace();
    },

    /**
     * Валидирует заданные параметры конструктора ставок
     * @returns {Boolean}
     */
    _validate: function() {
        var validateRes = this._pricesConstructor.validate();

        if (validateRes.errors) {
            BEM.blocks['b-user-dialog'].alert({
                message: validateRes.errors.map(function(error) {
                    return {
                        elem: 'error-popup-item',
                        content: error.text
                    }
                })
            });

            return false;
        }

        return true;
    },

    /**
     * Фильтрует данные, уходящие на сохранение
     * @param {Object} data - данные для фильтрации
     * @param {Object} strategy - имя стратегии
     * @param {'search'|'context'} platform - платформа
     * @param {Boolean} showBoth - показывать обе платформы
     * @returns {Object}
     */
    _filterData: function(data, strategy, platform, showBoth) {
        var res = {},
            whiteList = this.__self.getCurrentWhiteList(strategy, platform, showBoth);

        whiteList.forEach(function(field) {
            res[field] = data[field];
        });

        return res;
    }

}, {
    /**
     * Возвращает значения границ для вычисления ставки интерполяцией
     * @param {Number} ctrCorrection объем трафика
     * @param {Object} trafficVolume статистика соответствия ставки объему трафика
     * @returns {{leftBorder: Array, rightBorder: Array}}
     */
    _getInterpolationBorders: function(ctrCorrection, trafficVolume) {
        var trafficVolumeSortedValues = u['traffic-volume'].getTrafficVolumeKeys(trafficVolume),
            leftBorderValueIndex = u._.findIndex(trafficVolumeSortedValues, function(value) {
                return Number(value) <= ctrCorrection;
            }),
            rightBorder = trafficVolumeSortedValues[leftBorderValueIndex + 1],
            leftBorder = trafficVolumeSortedValues[leftBorderValueIndex];

        return {
            leftBorder: [Number(leftBorder), trafficVolume[leftBorder].bid_price / 1e6],
            rightBorder: [Number(rightBorder), trafficVolume[rightBorder].bid_price / 1e6]
        }
    },

    /**
     * Рассчитывает и устанавливает новые значения ставок для фраз и автотаргетинга на поиске
     * @param {'interpolation' | 'firstPage'} calcPriceMethod
     * @param {Object} modelsParams данные моделей которые надо проапдейтить
     * @param {String} value цена в конструкторе ставок
     * @param {String} currency валюта
     * @param {Boolean} hasRarelyUsedGroups есть ли на странице редко загруженные группы
     */
    _updateSearchPrices: function(calcPriceMethod, modelsParams, value, currency, hasRarelyUsedGroups) {
        var ctrCorrection = value.position_ctr_correction,
            increasePercentValue = value.proc_search,
            maxPrice = value.price_search,
            processGroupsIds = {},
            pricesList = [],
            _this = this;

        BEM.MODEL.forEachModel(function() {
            var model = this,
                autoPrice,
                autoPriceString,
                groupModel = model.getParentModel(),
                groupId = groupModel.get('adgroup_id');

            if (model.get('deleted')) {
                return true;
            }

            // не выставляем ставку для групп в которых только ГО или/и Видео
            if (u.imageAd.groupHasOnlyCpcVideoAndImageAds(groupModel.get('group_banners_types'))) {
                return true;
            }

            // не выставляем ставку для групп с мало показов
            if (hasRarelyUsedGroups) {
                if (processGroupsIds[groupId] == undefined) {
                    processGroupsIds[groupId] = groupModel.get('banners').some(
                        function(banner) {
                            return +banner.get('is_bs_rarely_loaded');
                        });
                }

                if (processGroupsIds[groupId]) {
                    return true;
                }
            }

            // не выставляем ставку если её нельзя редактировать
            if (!model.isEditablePriceField('price')) {
                return true;
            }

            // рассчитываем цену на поиске
            switch (calcPriceMethod) {
                case 'interpolation':
                    autoPrice = _this._calcPriceByInterpolation(ctrCorrection, model.toJSON().traffic_volume);
                    break;

                case 'firstPage':
                    autoPrice = _this._calcPriceForFirstPage({
                        guarantee: model.get('guarantee'),
                        premium: model.get('premium')
                    });
                    break;

                default:
                    throw new Error('Unsupported calculation price method');
            }

            autoPrice = _this._calcPriceByAdditionalParams(autoPrice, increasePercentValue, maxPrice);

            if (autoPrice) {
                autoPriceString = u.numberFormatter.format(autoPrice);
                pricesList.push(autoPriceString);
                model.update({ price: autoPriceString }, { source: _this });
            }
        }, modelsParams);

        // обновляем ставки для автотаргетинга (расчитываются по-другому)
        this._updateRelevanceMatchPricesSearch(modelsParams.parentName, currency, false);
    },

    /**
     * Обновляет ставку на автотаргетинг для поиска
     * @param {String} parentName - имя модели группы
     * @param {String} currency - валюта
     */
    _updateRelevanceMatchPricesSearch: function(parentName, currency) {
        var _this = this;

        BEM.MODEL.forEachModel(function() {
            var model = this,
                phrases = BEM.MODEL.get({ name: 'm-phrase-bidable', parentName: parentName, parentId: model.id }),
                pricesList = phrases.map(function(phrase) {
                    return phrase.get('price');
                });

            pricesList.length &&
                this.update({ price: u['relevance-match'].calcPercentile(pricesList, currency) }, { source: _this });
        }, { name: 'm-relevance-match', parentName: parentName, parentId: '*' })
    },

    /**
     * Обновляет ставку на автотаргетинг для сети
     * @param {String} parentName - имя модели группы
     * @param {String} currency - валюта
     */
    _updateRelevanceMatchPricesContext: function(parentName, currency, isContext) {
        var _this = this;

        BEM.MODEL.forEachModel(function() {
            var model = this,
                phrases = BEM.MODEL.get({ name: 'm-phrase-bidable', parentName: parentName, parentId: model.id }),
                pricesAndClicksList = phrases.map(function(phrase) {
                    return {
                        price: phrase.get('price_context'),
                        clicks: phrase.get('ctx_clicks')
                    };
                });

            pricesAndClicksList.length && this.update(
                { price_context: u['relevance-match'].calcAverage(pricesAndClicksList, currency) },
                { source: _this }
            );
        }, { name: 'm-relevance-match', parentName: parentName, parentId: '*' })
    },

    /**
     * Расчет ставки в соответсвии с дополнитеьными параметрами конструктора
     * @param {Number} price вычисленная конструктором ставка
     * @param {Number} increasePercentValue процент, на который нужно увеличить вычисленную конструктором ставку
     * @param {Number} maxPrice максимальная ставка
     */
    _calcPriceByAdditionalParams: function(price, increasePercentValue, maxPrice) {
        var result = price;

        // Прибавляем проценты
        if (increasePercentValue) {
            result += result * increasePercentValue / 100;
        }

        // Сравниваем с максимальной ценой
        if (result > maxPrice) {
            result = maxPrice;
        }

        return result;
    },

    /**
     * Расчет ставки для конкретной фразы
     * @param {String} ctrCorrection объем трафика
     * @param {Object} trafficVolume статистика соответствия ставки объему трафика
     * @returns {Number}
     */
    _calcPriceByInterpolation: function(ctrCorrection, trafficVolume) {
        var trafficVolumeSortedValues = u['traffic-volume'].getTrafficVolumeKeys(trafficVolume);

        if (ctrCorrection !== 'max') {
            ctrCorrection = Number(ctrCorrection);
        }

        // если объем трафика больше максимального в торгах, то берем значение максимально доступного в торгах
        if (Number(trafficVolumeSortedValues[0]) <= ctrCorrection || ctrCorrection === 'max') {
            return trafficVolume[trafficVolumeSortedValues[0]].bid_price / 1e6;
        }

        // если объем трафика меньше минимального в торгах, то берем значение минимально доступного в торгах
        if (Number(trafficVolumeSortedValues[trafficVolumeSortedValues.length - 1]) >= ctrCorrection) {
            return trafficVolume[trafficVolumeSortedValues[trafficVolumeSortedValues.length - 1]].bid_price / 1e6;
        }

        // Определяем ставку с помощью линейной интерполяции
        var interpolationBorders = this._getInterpolationBorders(ctrCorrection, trafficVolume),
            a = interpolationBorders.leftBorder,
            b = interpolationBorders.rightBorder;

        return a[1] + (ctrCorrection - a[0]) * (b[1] - a[1]) / (b[0] - a[0]);
    },

    /**
     * Расчет ставки для конкретной фразы для попадания на 1 страницу
     * @param {Object} auctionData
     * @param {Array} auctionData.premium массив со ставками в спецразмещении
     * @param {Array} auctionData.guarantee массив со ставками в гарантии
     * @returns {Number}
     */
    _calcPriceForFirstPage: function(auctionData) {
        return u.auction.getDataByPlace(auctionData, 'max').bid_price / 1e6;
    },

    /**
     * Рассчитываем и обновляем ставки для фраз на контексте
     * @param {Object} modelsParams данные моделей, которые надо обновить
     * @param {String} value цена
     * @param {String} strategy стратегия
     * @param {String} currency валюта
     */
    _updateContextPrices: function(modelsParams, value, strategy, currency) {
        var scope = value.context_scope,
            price = value.price_context,
            proc = value.proc_context || 0,
            _this = this;

        value.phrases_context = 1;

        BEM.MODEL.forEachModel(function() {
            var model = this;

            if (model.get('deleted')) return true;

            var pval = model.get('probs'),
                minPrice = u.currencies.getConst(currency, 'DEFAULT_PRICE'),
                probs = [],
                prices = [];

            if (strategy != 'different_places' && model.get('state') == 'active') return true;

            pval.split(';').forEach(function(part) {
                var tmp = part.split(',');

                prices.push(+tmp[0]);
                probs.push(+tmp[1]);
            });

            var calc_price = model.get('is_retargeting') ?
                minPrice :
                BEM.blocks['i-autobroker']
                    .calcContextPriceByCoverage(scope, model.get('pokazometer_data')) / 1e6 || price;

            if (calc_price) {
                if (proc) {
                    // Прибавляем проценты
                    calc_price *= 1 + proc / 100;
                }
                //округляем до шага торгов
                calc_price = u.currencies.roundPriceToStep(calc_price, currency, 'up');
                calc_price = Math.min(price, calc_price);
            } else {
                calc_price = minPrice;
            }

            if (calc_price) {
                model.update({ price_context: calc_price }, { source: _this });
            }
        }, modelsParams);

        // обновляем ставки для автотаргетинга (расчитываются по-другому)
        this._updateRelevanceMatchPricesContext(modelsParams.parentName, currency);
    },

    /**
     * Возвращает список разрешенных конструкторов по типу
     * @param {String} type тип конструкторов
     * @returns {Array}
     * @private
     */
    _getWhiteList: function(type) {
        var searchFields = ['price_search', 'position_ctr_correction', 'proc_search'],
            contextFields = ['price_context', 'context_scope', 'proc_context'];

        switch (type) {
            // на всех площадках
            case 'common-wizard':
                return [].concat(searchFields, ['is_simple']);

            // отдельное размещение, одновременно показывается поиск и контекст
            case 'both-wizard':
                return [].concat(searchFields, contextFields, ['is_simple', 'context_toggle', 'search_toggle']);

            // отдельное размещение, поиск
            case 'search-wizard':
                return [].concat(searchFields, contextFields, ['is_simple', 'platform']);

            // отдельное размещение, контекст
            case 'context-wizard':
                return [].concat(searchFields, contextFields, ['is_simple', 'platform']);
        }
    },

    /**
     * Возвращает текущий тип конструктора ставок
     * @param {Object} strategy
     * @param {'search'|'context'} platform
     * @returns {string}
     * @private
     */
    _getConstructorType: function(strategy, platform, showBoth) {
        if (strategy.name != 'different_places') {
            return 'common-wizard';
        } else {
            if (showBoth) {
                return 'both-wizard';
            } else {
                return platform + '-' + 'wizard';
            }
        }
    },

    /**
     *
     * @param {Object} strategy
     * @param {'search'|'context'} platform
     * @param {Boolean} showBoth
     * @returns {String}
     */
    getCurrentWhiteList: function(strategy, platform, showBoth) {
        return this._getWhiteList(this._getConstructorType(strategy, platform, showBoth));
    }

});
