/**
 * Группа фраз/категорий/условий ретаргетинга для списка фраз
 */
BEM.DOM.decl({ block: 'b-phrases-list-group', implements: 'i-sortable-interface' }, {

    onSetMod: {

        js: function() {
            this.initModels();
            this._initGroupControls();
        }

    },

    /**
     * Инициализирует групповые контролы для данной группы (используется b-phrases-list)
     * @returns {BEM}
     */
    _initGroupControls: function() {
        if (this.inited) return this;

        this.phrases = this.getPhrases();

        this.size = this.phrases.length;

        this.mainPrice = this.findBlockOn(this.elem('price-control', 'obj-type', 'common'), 'b-edit-phrase-price');
        this.mainPrice && this.mainPrice.on('change', function(e, data) {
            this.trigger('change-common-price', data);
        }, this);

        this.findBlocksInside('input-block', 'b-edit-phrase-price').forEach(function(control) {
            var objType = control.getMod('obj-type');

            control.on('change', function(e, data) {
                this.setPriceForAll(data.value, objType);
            }, this);
        }, this);

        this.findBlocksInside('b-group-phrase').forEach(function(row) {
            row.on('destruct', this.onPhraseDestruct, this);
        }, this);

        this.inited = true;

        return this;
    },

    onPhraseDestruct: function(e) {
        var type = e.block.getMod('type'),
            elemType = {
                phrases: 'phrases',
                retargeting: 'retargetings',
                interest: 'interests',
                'relevance-match': 'relevance-match'
            }[type],
            rows = this.findBlocksInside('b-group-phrase')
                .filter(function(block) {
                    return !block.hasMod('deleted', 'yes') &&
                        !block.hasMod('hidden', 'yes');
                }),
            curTypeRows = rows.filter(function(block) {
                return block.hasMod('type', type);
            });

        if (!rows.length && !this.hasMod('state', 'active')) {
            // если в группе не осталось ни одной строки, удаляем ее
            this.destruct();
        } else {
            if (!curTypeRows.length) {
                // если не осталось ни одной строки текущего типа, то удаляем заголовок для текущего типа
                this.elem('input-block', 'type', elemType).remove();
                this.elem('group-marker', 'type', elemType).remove();
                // заголовок с поиском может быть только при БТ и он всегда 1
                type == 'relevance-match' && this.elem('platform-title', 'type', 'search').remove()
            }
            if (rows.length > 0 && rows[0].hasMod('type', 'relevance-match')) {
                this.setMod(this.elem('relevance-match'), 'is-first-type', 'yes');
            }
        }
    },

    /**
     * Устанавливает общую ставку для всех фраз
     * @param {Number} value - значение ставки
     */
    setPriceForAll: function(value, objType) {
        this.findBlocksInside('b-edit-phrase-price').forEach(function(control) {
            var modelName;

            if (control.hasMod('multiedit', 'yes')) {
                control.setControlValue('');
            } else {
                modelName = {
                    phrases: 'm-phrase-bidable',
                    retargetings: 'm-retargeting-bidable',
                    interests: 'm-interest-bidable'
                }[objType];

                (!modelName || control.model && modelName === control.model.name) &&
                value && !control.hasMod('disabled') &&
                    control.setModelValue(value);
            }
        }, this);
    },

    /**
     * Возвращает родительский список фраз
     * @returns {BEM}
     */
    getParent: function() {
        return this.parent || (this.parent = this.findBlockOutside('b-phrases-list'));
    },

    /**
     * Возвращает BEM-Блоки для фраз, содержащихся в данной группе
     * @returns {Object}
     */
    getPhrases: function() {
        return this.phrases ||
            (this.phrases = this.findBlocksInside('b-group-phrase').filter(function(phrase) {
                return phrase.getMod('deleted') != 'yes';
            }));
    },

    /**
     * Инициализируем модели для данной группы
     * @returns {BEM}
     */
    initModels: function() {
        if (this.models) return this;

        this.models = Object.keys(this.getPhrases()).map(function(i) {
            return this.phrases[i].getModel()
        }, this);

        // Костыль из-за срочности по задаче DIRECT-49094
        // Во время ревью коммитов завести задачу на ответственного
        // За эти правки в r101743
        if (!this.models.length) return this;

        this.models.forEach(function(model) {
            var parent = model.getParentModel();

            model && BEM.MODEL.on({
                name: model.name,
                id: model.get('modelId'),
                parentName: parent.name,
                parentId: parent.id
            }, 'destruct', this.onModelDestruct, this);
        }, this);

        return this;
    },

    /**
     * Какая-то из моделей группы была удалена
     * @returns {BEM}
     */
    onModelDestruct: function(e, data) {
        // у групп есть три поля phrasesIds, retargetingsIds, interestsIds,
        // в которые сохраняются id моделей m-phrase-bidable, m-retargeting-bidable, m-interest-bidable,
        // связанных с группой
        // при удалении моделей условий показа нужно синхронизировать список id в группе
        var size = this.size - 1,
            model = data.model,
            parent = model.getParentModel(),
            fieldName = {
                'm-phrase-bidable': 'phrasesIds',
                'm-retargeting-bidable': 'retargetingsIds',
                'm-interest-bidable': 'interestsIds'
            }[model.name],
            ids = model.name == 'm-relevance-match' ?
                '' :
                parent.get(fieldName).filter(function(id) {
                    return id != model.id;
                });

        // у бесфразного таргетинга нет списка id,
        // модель 1 на группу и modelId совпадает с modelId группы
        model.name == 'm-relevance-match' || parent.set(fieldName, ids);

        // DIRECT-81651 как минимум для ретаргетинга на момент onModelDestruct уже наступил destruct самого блока
        this.domElem && this.setMod('empty', size == 0 ? 'yes' : 'no');

        return this;
    },

    /**
     * Сортирует набор блоков elements по полю fieldName
     * @param {Array} elements
     * @param {String} fieldName
     * @param {Boolean} reverse - обратный/прямой вид сортировки
     * @returns {Object}
     */
    getSortedFragment: function(elements, fieldName, reverse) {
        reverse = reverse ? -1 : 1;

        var domElem,
            _this = this,
            platform = this.params.platform,
            fragment = document.createDocumentFragment(),
            sorted = elements.sort(function(phraseA, phraseB) {
                var sortTr,
                    modelA = phraseA.getModel(),
                    modelB = phraseB.getModel(),
                    valA = modelA.get(fieldName),
                    valB = modelB.get(fieldName);

                if (fieldName == 'phrase') {
                    valA = valA.toLowerCase();
                    valB = valB.toLowerCase();
                }

                //для турции своя лексикографическая сортировка
                if (u.consts('locale') == 'tr' && fieldName == 'phrase') {
                    sortTr = u.sortTr(valA, valB);
                    return reverse * sortTr;
                } else {
                    if (valA < valB) {
                        return -reverse;
                    } else if (valA > valB) {
                        return reverse;
                    }
                }

                return 0;
            });

        sorted.map(function(block, i) {
            if (block.domElem.length > 1) {
                domElem = block.domElem.filter(function(i, blockPart) {
                    return _this.getMod($(blockPart), 'platform') == platform;
                });
            } else {
                domElem = block.domElem;
            }
            block.setMod(domElem, 'first', i == 0 ? 'yes' : '');

            fragment.appendChild(domElem.get(0));
        }, this);

        return fragment;
    },

    /**
     * Возвращает массив моделей соответствующих фразам данной группы
     * @returns {BEM.MODEL[]}
     */
    getModels: function() {
        this.initModels();

        return this.models;
    },

    /**
     * Проверяет, проинициализированы ли фразы и контролы для данной группы
     * @returns {Boolean}
     */
    isControlsInited: function() {
        return this._controlsInited;
    },

    /**
     * Устанавливает флаг _controlsInited в положение true
     * @returns {BEM}
     */
    setControlsInited: function() {
        this._controlsInited = true;

        return this;
    },

    /**
     * Сортирует элементы внутри данной группы
     * @param {String} sortField
     * @param {Boolean} toMin
     * @returns {BEM}
     */
    sort: function(sortField, toMin) {
        var markerTypes = {
            phrases: 'phrases',
            retargeting: 'retargetings',
            interest: 'interests'
        };

        this.initModels();

        ['phrases', 'retargeting', 'interest'].forEach(function(el) {
            this[el + 'Blocks'] = this[el + 'Blocks'] ||
                this.findBlocksInside({ block: 'b-group-phrase', modName: 'type', modVal: el });
            var fragment = this.getSortedFragment(this[el + 'Blocks'], sortField, toMin);
            this.elem(el).remove();
            this.elem('group-marker', 'type', markerTypes[el]).after(fragment);
        }, this);

        return this
    }

}, {

    live: function() {
        this
            .liveInitOnBlockInsideEvent('init', 'b-edit-phrase-price')
            .liveBindTo('inactive-toggle-button', 'click', function(e) {
                var button = this.findBlockOn(e.data.domElem, 'button');

                if (button) {
                    button.toggleMod('arrow', 'down', 'up');

                    button.isDisabled() || this.toggle();
                }
            });
    }
});
