BEM.DOM.decl('b-private-deals-inspector', {

    onSetMod: {
        js: function() {
            this._delModProgress = function() { return this.delMod('progress') }.bind(this);
            this._setModProgress = function() { return this.setMod('progress', 'yes') }.bind(this);

            this
                .bindToDoc('keydown', this._onKeyDown)
                .bindToDoc('click', this._onDocClick);
        },

        progress: function(modName, modVal) {
            this._ignoreCloseEvents = !!modVal;
        },

        visibility: {
            '*': function(modName, modVal) {
                this._ignoreCloseEvents = !modVal;
            },

            '': function() {
                this.dropBlockInsideCache('unit', 'b-private-deals-inspector-unit');

                BEM.DOM.update(this.findElem('unit'), '');
            }
        }
    },

    /**
     * Проверка на наличие изменений во внутреннем блоке
     * @returns {Boolean}
     */
    isChanged: function() {
        var unit = this.blockInside('unit', 'b-private-deals-inspector-unit');

        return unit && unit.isChanged();
    },

    /**
     * Запрашиваем и показываем данные по номеру сделки
     * @param {Number} id номер сделки
     * @param {Boolean} [overwrite] флаг о перезаписи данных
     */
    show: function(id, overwrite) {
        var unit = this.blockInside('unit', 'b-private-deals-inspector-unit');

        if (!overwrite && unit && unit.isChanged()) {
            this._confirm(iget2(
                'b-private-deals-inspector',
                'are-you-sure-to-close',
                'Изменения не будут сохранены. Продолжить?'
            ))
                .then(this.show.bind(this, id, true))
                .catch(function() {});
        } else {
            this._dealId = id;

            this.trigger('deal-show', { id: this._dealId });

            this
                .setMod('visibility', 'visible')
                .setMod('progress', 'yes');

            BEM.blocks['i-web-api-request'].privateDeals.getDealsDetails(u.consts('ulogin'), [id])
                .then(function(resp) {
                    return u._.get(resp, 'result[0]');
                })
                .then(function(data) {
                    this._updateObjectLoginByCid(data.campaigns);

                    this._updateUnit({ full: data });
                }.bind(this))
                .catch(this._loadDealFailed.bind(this))
                .then(this._delModProgress);
        }
    },

    /**
     * Закрытие инспектора
     * @param {Boolean} [isForce] флаг о принудительном закрытии
     */
    close: function(isForce) {
        var unit = this.blockInside('unit', 'b-private-deals-inspector-unit');

        if (!isForce && unit && unit.isChanged()) {
            this._confirm(iget2(
                'b-private-deals-inspector',
                'are-you-sure-to-close',
                'Изменения не будут сохранены. Продолжить?'
            ))
                .then(this.close.bind(this, true))
                .catch(function() {})
        } else {
            this.delMod('visibility');
            this.trigger('deal-hide', { id: this._dealId });
        }
    },

    /**
     * Обработчик ошибки загрузки данных сделки
     * @private
     */
    _loadDealFailed: function() {
        BEM.DOM.update(this.findElem('unit'), BEMHTML.apply({
            block: 'b-private-deals-inspector',
            dealId: this._dealId,
            elem: 'load-error'
        }));
    },

    /**
     * Номер сделки
     */
    _dealId: null,

    /**
     * Обработчик клика внутри document
     * @param {Event} e
     * @private
     */
    _onDocClick: function(e) {
        if (!this._ignoreCloseEvents) {
            var inspectorLeftOffset = this.domElem.position().left,
                eventLeftOffset = e.pageX,
                isOpened = this.getMod('visibility') === 'visible';

            if ((eventLeftOffset < inspectorLeftOffset) && isOpened) {
                this.close();
            }
        }
    },

    /**
     * Обработчик нажатия на кнопки
     * @param {Event} e
     * @private
     */
    _onKeyDown: function(e) {
        if (BEM.blocks.keycodes.is(e.keyCode, 'ESC')) {
            if (!this._ignoreCloseEvents &&
                !this.blockInside('unit', 'b-private-deals-inspector-unit').haveReactionToEsc()) {

                this.close();
            }
        }
    },

    /**
     * Отрисовка внутреннего блока с контролами для редактирования
     * @param {Object} data
     * @returns {Promise}
     * @private
     */
    _renderUnit: function(data) {
        this._data = data;

        return new Promise(
            function(resolve, reject) {
                BEM.DOM.update(
                    this.findElem('unit'),
                    BEMHTML.apply({
                        block: 'b-private-deals-inspector-unit',
                        mods: { 'read-only': this.params.readOnly && 'yes' },
                        data: data
                    }),
                    resolve
                );
            }.bind(this)
        )
            .then(function(ctx) {
                this.dropBlockInsideCache('unit', 'b-private-deals-inspector-unit');

                return this.blockInside(ctx, 'b-private-deals-inspector-unit')
            }.bind(this))
            .catch(this._loadDealFailed.bind(this))
    },

    /**
     * Получение умолчальных значений состояния
     * @param {Object} data
     * @returns {{isTitleEditing: boolean, editedTitle, deals: boolean, clients: boolean}}
     * @private
     */
    _getDefaultState: function(data) {
        return {
            isTitleEditing: false,
            editedTitle: u._.get(data, 'deal.name'),
            deals: u._.get(data, 'deal.status') === 'ACTIVE' ?
                true :
                u._.set({}, '' + u._.get(data, 'deal.id'), true),
            clients: true
        };
    },

    /**
     * Обновление данных внутреннего блока с контролами
     * @param {Object} params
     * @returns {Promise<T>}
     * @private
     */
    _updateUnit: function(params) {
        var data = this._data,
            savedKeysFromOldState = params.savedStateKeys || [],
            state;

        if (params.full) {
            data = params.full;
            state = this._getDefaultState(data);
        } else if (params.chunk) {
            var unit = this.blockInside('unit', 'b-private-deals-inspector-unit');

            data = u._.reduce(params.chunk, function(resultData, value, path) {
                return u._.set(resultData, path, value)
            }, this._data || {});

            state = u._.assign(
                this._getDefaultState(data),
                u._.pick(unit.getState(), savedKeysFromOldState)
            );
        }

        return this
            ._renderUnit(data)
            .then(function(unit) {
                return unit.setState(state);
            });
    },

    /**
     * Обработчик клика по крестику
     * @private
     */
    _onCloseClick: function() {
        this.close();
    },

    /**
     * Принятие/отклонение сделки
     * @param {Boolean} isAccept флаг принятия сделки
     * @returns {*|Promise<T>}
     * @private
     */
    _acceptRejectDeal: function(isAccept) {
        BEM.blocks['b-metrika2'].reachGoal(isAccept ? 'APPROVE_DEAL' : 'REJECT_DEAL');

        return BEM.blocks['i-web-api-request'].privateDeals[isAccept ? 'activateDeals' : 'completeDeals'](
            u.consts('ulogin'),
            [this._dealId]
        )
            .then(function(response) {
                if (!response.success) {
                    throw response;
                }

                return this._updateUnit({
                    chunk: { 'deal.status': isAccept ? 'ACTIVE' : 'ARCHIVED' }
                });
            }.bind(this))
            .catch(function(response) {
                var errors = u._.get(
                        response,
                        'validation_result.errors',
                        [{ path: '' }]
                    ),
                    mapper = u['b-private-deals-inspector'].getStatusErrorsMapper(),
                    unit = this.blockInside('unit', 'b-private-deals-inspector-unit');

                if (unit) {
                    unit.showErrors(
                        errors.map(function(error) {
                            return {
                                path: error.path,
                                description: mapper({}, error)
                            };
                        })
                    );
                }
            }.bind(this));
    },

    /**
     * Обработчик клика по кнопке сохранения изменений
     * @param {Event} e
     * @param {Object} data
     * @private
     */
    _onUnitClickSave: function(e, data) {
        var id = this._dealId;

        this._setModProgress()
            ._saveDeal('save', data.value)
            .then(function() {
                this.trigger('deal-change', { id: id });
            }.bind(this))
            .then(this._delModProgress)
            .catch(function() {
                if (u._.get(data, 'value.toAdd', []).length || u._.get(data, 'value.toRemove', []).length) {
                    this.trigger('deal-change', { id: id });
                }

                this._delModProgress();
            }.bind(this));
    },

    /**
     * Сохраняет изменения в инспекторе
     * @param {'save'|'accept'|'reject'} action Действие которое вызвало сохранение
     * @param {object} value
     * @param {string[]} [value.toAdd]
     * @param {string[]} [value.toRemove]
     * @param {object} [value.addedCampaignsInfo]
     * @param {string} [value.name]
     * @return {Promise}
     * @private
     */
    _saveDeal: function(action, value) {
        var campaignIdsToAdd = value.toAdd || [],
            campaignIdsToRemove = value.toRemove || [],
            addedCampaignsInfo = value.addedCampaignsInfo || {},
            isNameChanged = !u._.isUndefined(value.name),
            currentName = u._.get(this._data, 'deal.name'),
            dealId = Number(this._dealId),
            // При отклонении сделки мы игнорируем привязку/отвязку кампаний
            notReject = action !== 'reject',
            mapper = function(cid) {
                return { campaign_id: cid, deal_id: dealId };
            },
            requestParams,
            isLinkedCampsChanged = !!(campaignIdsToAdd.length || campaignIdsToRemove.length);

        this._updateObjectLoginByCid(addedCampaignsInfo);

        if (!isLinkedCampsChanged && !isNameChanged) {
            return this._updateUnit({ full: this._data });
        }

        if (isLinkedCampsChanged) {
            BEM.blocks['b-metrika2'].reachGoal('BIND_CAMPAIGNS_TO_DEAL');
        }

        if (isNameChanged) {
            BEM.blocks['b-metrika2'].reachGoal('CHANGE_DEAL_NAME');
        }

        return BEM.blocks['i-web-api-request'].privateDeals.updateDeals(
            u.consts('ulogin'),
            requestParams = {
                deals: isNameChanged ? [{ deal_id: dealId, name: value.name }] : [],
                add_links: notReject ? campaignIdsToAdd.map(mapper) : [],
                remove_links: notReject ? campaignIdsToRemove.map(mapper) : []
            }
        )
            .then(function(response) {
                // DIRECT-77860: warnings в нашем случае должны восприниматься как ошибка, детали в задаче
                var warnings = u._.get(response, 'validation_result.warnings') || [];

                if (warnings.length) {
                    u._.set(
                        response,
                        'validation_result.errors',
                        (u._.get(response, 'validation_result.errors') || []).concat(warnings));
                }

                return response;
            })
            .catch(function(response) {
                u._.set(response, 'validation_result.errors[0].path', '');

                return response;
            })
            .then(function(response) {
                var currentCampaigns = u._.get(this._data, 'campaigns', []),
                    addedIds = u._.map(u._.get(response, 'result.add_links'), 'campaign_id'),
                    removedIds = u._.map(u._.get(response, 'result.remove_links'), 'campaign_id'),
                    summaryCampaigns = response.success ?
                        currentCampaigns
                            .concat(addedIds.map(function(cid) { return addedCampaignsInfo[cid] }))
                            .filter(function(camp) {
                                return !u._.includes(removedIds, camp.campaignId);
                            }) :
                        currentCampaigns,
                    updatedName = response.success ?
                        u._.get(response, 'result.deals.0.name', currentName) :
                        currentName,
                    errors = u._.get(response, 'validation_result.errors'),
                    savedStateKeys = (action === 'save' ? ['deals', 'clients'] : []).concat(
                        u._.some(errors, { path: 'deals[0].name' }) ? ['isTitleEditing', 'editedTitle'] : []
                    );

                return Promise.all([
                    this._updateUnit({
                        chunk: {
                            campaigns: summaryCampaigns,
                            'deal.name': updatedName
                        },
                        savedStateKeys: savedStateKeys
                    }),
                    response
                ]);
            }.bind(this))
            .then(function(results) {
                var unit = results[0],
                    response = results[1],
                    errors = u._.get(response, 'validation_result.errors', []),
                    nameWasChangedWithoutErrors = !u._.some(errors, { path: 'deals[0].name' });

                if (errors.length) {
                    var mapper = u['b-private-deals-inspector'].getErrorsMapper();

                    if (unit) {
                        unit.showErrors(
                            this._extendErrorsParams(errors, requestParams).map(function(error) {
                                return {
                                    path: error.path,
                                    description: mapper(requestParams, error)
                                };
                            })
                        );
                    }
                }

                return Promise[response.success && nameWasChangedWithoutErrors ? 'resolve' : 'reject']();
            }.bind(this));
    },

    /**
     * Обработчик события изменения статуса сделки
     * @param {jQuery.Event} e
     * @param {object} data
     * @param {'accept'|'reject'} data.status Новый статус
     * @param {boolean} data.isChanged Флан наличия изменений
     * @param {object} data.value Значение юнита
     * @private
     */
    _onUnitUpdateStatus: function(e, data) {
        var id = this._dealId,
            isAccept = data.status === 'accept',
            isChanged = data.isChanged,
            isNameChanged = !!data.value.name,
            value = data.value,
            rejectMessage = [
                (isChanged && isNameChanged) ?
                        iget2(
                            'b-private-deals-inspector',
                            'are-you-sure-to-reject-deal-and-save-title',
                            'Вы уверены, что хотите сохранить название и отклонить сделку?'
                        ) :
                        iget2(
                            'b-private-deals-inspector',
                            'are-you-sure-to-reject',
                            'Вы уверены, что хотите отклонить сделку?'
                        ),
                '</br>',
                iget2(
                    'b-private-deals-inspector',
                    'deistvie-ne-mojet-bit-otmeneno',
                    'Действие не может быть отменено'
                )
            ];

        (isAccept ? Promise.resolve() : this._confirm(rejectMessage))
            .then(this._setModProgress)
            .then(function() {
                // Сохраняем изменения, если принимаем сделку (переводим в статус "Активная")
                // или если отклоняем сделку (переводим в статус "Отклонена") и изменено название сделки
                return isChanged && (isAccept || isNameChanged) ?
                    this._saveDeal(data.status, value) :
                    Promise.resolve();
            }.bind(this))
            .then(this._acceptRejectDeal.bind(this, isAccept))
            .then(function() {
                this.trigger('deal-change', { id: id });
            }.bind(this))
            .then(this._delModProgress)
            .catch(this._delModProgress);
    },

    /**
     * Предупреждение с заданным сообщением
     * @param {String|Array} message
     * @private
     * @return {Promise}
     */
    _confirm: function(message) {
        this._ignoreCloseEvents = true;

        return new Promise(function(resolve, reject) {
            BEM.blocks['b-user-dialog'].confirm({
                message: message,
                onConfirm: function() {
                    this._ignoreCloseEvents = false;

                    resolve()
                },
                onCancel: function() {
                    this._ignoreCloseEvents = false;

                    reject();
                },
                callbackCtx: this
            });
        }.bind(this));
    },

    /**
     * Собирает объект ключом которого cid, значением - login
     * @param {object|array} campaigns
     * @private
     */
    _updateObjectLoginByCid: function(campaigns) {
        this._loginByCid = u._.assign(
            this._loginByCid || {},
            u._.reduce(campaigns, function(result, campaign) {
                return u._.set(result, campaign.campaignId, campaign.userName)
            }, {})
        );
    },

    /**
     * Добавляет в параметры ошибок привязки/отвязки `cid` и `ulogin`
     * @param {{description: String, path: String, params: object}[]} errors
     * @param {object} requestParams
     * @returns {{description: String, path: String, params: object}[]}
     * @private
     */
    _extendErrorsParams: function(errors, requestParams) {
        var loginByCid = this._loginByCid || {};

        return errors.map(function(error) {
            var pathArray = u['b-errors-presenter2'].toPath(error.path);

            if (u._.includes(['add_links', 'remove_links'], pathArray[0])) {
                var cid = u._.get(
                    requestParams,
                    u['b-errors-presenter2'].joinPath([pathArray[0], pathArray[1], 'campaign_id'])
                );

                error.params = u._.assign(
                    error.params || {},
                    {
                        cid: cid,
                        ulogin: loginByCid[cid]
                    }
                );
            }

            return error;
        });
    }

}, {
    live: function() {
        this
            .liveInitOnBlockInsideEvent('save', 'b-private-deals-inspector-unit', function(e, data) {
                this._onUnitClickSave(e, data);
            })
            .liveInitOnBlockInsideEvent('update-status', 'b-private-deals-inspector-unit', function(e, data) {
                this._onUnitUpdateStatus(e, data);
            })
            .liveInitOnBlockInsideEvent('click', 'button2', function(e) {
                if (e.block.getMod('use') === 'data-reload') {
                    this.show(e.block.params.dealId, true);
                }
            })
            .liveBindTo('close', 'click', function() {
                this._onCloseClick();
            });
    }
});
