BEM.DOM.decl('b-private-deals-table', {

    onSetMod: {

        js: function() {
            this._initSearch();

            this._syncColumnsWidth();
            this._syncBodyScroll();

            this.bindTo('body-wrap', 'scroll', this._onBodyScroll);

            this._resizeWatcher = BEM.blocks['resize-watcher'].getInstance({
                owner: this,
                ignoreHeight: true
            });

            this._resizeWatcher.on('change', function() {
                this._syncColumnsWidth();
                this._syncBodyScroll();
            }, this);
        }

    },

    /**
     * Обновление статуса сделки
     * @param {Number} dealId id сделки
     * @param {Object} deal данные сделки
     */
    updateDeal: function(dealId, deal) {
        var dealRow = this._getRowById(dealId),
            index = this.getMod(dealRow, 'index'),
            searchIndex = '',
            newDealRow = BEM.DOM.replace(dealRow, BEMHTML.apply({
                block: 'b-private-deals-table',
                elem: 'row',
                elemMods: {
                    highlight: this.getMod(dealRow, 'highlight')
                },
                columnsList: this.params.columnsList,
                deal: deal,
                index: index,
                addToSearch: function(text) {
                    searchIndex += (text + '').toLowerCase() + '\n';

                    return {
                        block: this.block,
                        elem: 'hlt',
                        content: text
                    };
                }
            }));

        this._iSearchable.updateItems(
            u._.set(
                this._searchItems,
                index,
                {
                    searchIndex: searchIndex,
                    domElem: newDealRow
                }
            )
        );

        this._dropHltBlocksCache(index);

        this.reSearch();
    },

    /**
     * Обновление списка сделок
     * @param {Array} deals Данные сделок
     */
    updateDeals: function(deals) {
        var searchIndex = [],
            highlightedRow = this.findElem('row', 'highlight', 'yes'),
            body = $(BEMHTML.apply({
                block: 'b-private-deals-table',
                elem: 'body',
                deals: deals,
                columnsList: this.params.columnsList,
                addToSearch: function(rowIndex, text) {
                    searchIndex[rowIndex] = (searchIndex[rowIndex] || '') + (text + '').toLowerCase() + '\n';

                    return {
                        block: this.block,
                        elem: 'hlt',
                        content: text
                    };
                }
            })),
            rows = this.findElem(body, 'row');

        if (highlightedRow.length) {
            this._setDealHighlight(
                this.getMod(highlightedRow, 'id'),
                body
            );
        }

        this._searchItems = searchIndex.map(function(searchIndexItem, i) {
            return {
                searchIndex: searchIndexItem,
                domElem: rows.eq(i)
            };
        });

        this._iSearchable.updateItems(this._searchItems);
        this._iSearchable.onFirst('search.finish', function() {
            BEM.DOM.replace(this.findElem('body'), body);
        }, this);

        this._dropHltBlocksCache();
        this.reSearch();
    },

    /**
     * Блокируем элемент body наложением паранжи со спиннером
     */
    lockBody: function() {
        this.setMod(this.findElem('body'), 'progress', 'yes');
    },

    /**
     * Установка подсветки строки сделки в таблице
     * @param {string} id
     */
    setDealHighlight: function(id) {
        this._setDealHighlight(id, this.findElem('body'));
    },

    /**
     * Выключение подсветки строки сделки в таблице
     */
    dropDealHighlight: function() {
        this.delMod(
            this.findElem('row', 'highlight', 'yes'),
            'highlight'
        );
    },

    /**
     * Убираем "стрелочку" сортировки
     */
    dropSortStates: function() {
        this.delMod(this.elem('head-sort'), 'sort');
    },

    /**
     * Установка состояния сортировки
     * @param {String} field столбец по которому сортируем
     * @param {String} reverse порядок сортировки
     */
    setSortStates: function(field, reverse) {
        var sortItem = this.elem('head-sort', 'sort-field', u.beminize(field)),
            icon = this.findBlockInside(sortItem, 'icon');

        this.dropSortStates();
        if (sortItem.length) {
            this.setMod(sortItem, 'sort', 'yes');
            icon.setMod('direction', reverse === 'desc' ? 'bottom' : 'top');
        }
    },

    /**
     * Метод поиска
     * @param {String} text Искомая строка
     */
    search: function(text) {
        var query = text.toLowerCase().trim();

        if (!u._.isEqual(query, this._lastQuery)) {
            this._iSearchable.search({ text: this._lastQuery = query });
        }
    },

    /**
     * Выполняет предыдущий поиск
     */
    reSearch: function() {
        this._iSearchable.search({ text: this._lastQuery || '' });
    },

    /**
     * Подсветка совпавшей подстроки
     * @param {Object} data
     * @param {Object} data.item найденный элемент
     * @param {Object} data.iterator итератор поиска
     */
    highlight: function(data) {
        var iterator = data.iterator,
            item = data.item,
            index = iterator.iteration,
            domElem = item.domElem,
            originalValues = this.getOriginalValues(domElem, index);

        this.getHltBlocks(domElem, index).each(function(index, hlt) {
            var html = originalValues[index].replace(iterator.regexp, function(str) {
                return BEMHTML.apply({
                    block: 'b-private-deals-table',
                    elem: 'hlted',
                    content: str
                });
            });

            if (hlt.innerHTML !== html) {
                hlt.innerHTML = html;
            }
        });
    },

    /**
     * Получаем массив строк среди которых происходит поиск
     * @param {jQuery} domElem
     * @param {Number} index
     * @returns {Array}
     */
    getOriginalValues: function(domElem, index) {
        var hltBlocksText = domElem.data('highlight.original');

        if (!hltBlocksText) {
            hltBlocksText = this.getHltBlocks(domElem, index).map(function(i, htl) {
                return htl.innerHTML;
            });

            domElem.data('highlight.original', hltBlocksText);
        }

        return hltBlocksText;
    },

    /**
     * Получаем искомые элементы в указанной DOM-ноде
     * @param {jQuery} domElem
     * @param {Number} index
     * @returns {jQuery}
     */
    getHltBlocks: function(domElem, index) {
        this._chacheHltBlocks || (this._chacheHltBlocks = []);

        return this._chacheHltBlocks[index] || (this._chacheHltBlocks[index] = this.findElem(domElem, 'hlt'));
    },

    /**
     * Установка подсветки строки сделки в таблице
     * @param {Number} id
     * @param {jQuery} body
     * @private
     */
    _setDealHighlight: function(id, body) {
        this.dropDealHighlight();

        this.setMod(
            this._getRowById(id, body),
            'highlight',
            'yes'
        );
    },

    /**
     * Обработчик скролла внутри элемента body
     * @private
     */
    _onBodyScroll: function() {
        var bodyScrollLeft = this.elem('body-wrap').scrollLeft();

        if (bodyScrollLeft !== this._lastBodyScrollLeft) {
            this._lastBodyScrollLeft = bodyScrollLeft;
            this._syncBodyScroll();
        }
    },

    /**
     * Обработчик заполнения инпута для фильтрации
     * @param {Event} e
     * @param {Object} data
     * @private
     */
    _onFilterInputChange: function(e, data) {
        this.search(data.text);
    },

    /**
     * Синхронизация ширины колонок для правильного отображения шапки
     * @private
     */
    _syncColumnsWidth: function() {
        var visibleRow = this.findElem('row', 'hidden', 'no').eq(0),
            visibleDataCells = this.findElem(visibleRow, 'data-cell'),
            fixedCellsContainer = this.elem('head-cell-content');

        visibleDataCells.each(function(i, originalCell) {
            fixedCellsContainer.eq(i).width($(originalCell).width());
        });

        // Для "особенных" браузеров мы еще и подгоняем ширину шапки под ширину строки таблицы
        this.elem('head').width(visibleRow.width());
    },

    /**
     * Синхронизация расположения шапки относительно скролла внутри body
     * @private
     */
    _syncBodyScroll: function() {
        this.elem('head').css('left', -this.elem('body-wrap').scrollLeft());
    },

    /**
     * Получение строки сделки по её id
     * @param {Number} id сделки
     * @param {jQuery} [body]
     * @returns {jQuery}
     * @private
     */
    _getRowById: function(id, body) {
        return this.findElem.apply(this, u._.compact([body, 'row', 'id', id]));
    },

    /**
     * Обработчик события выбора сделки в таблице
     * @param {Event} e
     * @private
     */
    _onRowClick: function(e) {
        e.stopImmediatePropagation();

        var dealId = this.elemParams(this.findElem($(e.target).parents(this.buildSelector('row')), 'data-cell-name')).id,
            selection = window.getSelection(),
            selectedText = selection && selection.type === 'Range' &&
                $(selection.focusNode).parents(this.buildSelector('row')).length;

        if (selectedText) {
            e.preventDefault();
        } else {
            this.trigger('deal-selected', { id: dealId });
        }
    },

    /**
     * Обработчик события выбора сделки в таблице
     * @param {Event} e
     * @private
     */
    _onHeadCellClick: function(e) {
        var field = this.elemParams($(e.target)).field;

        this.trigger('deals-sort', { field: field });
    },

    /**
     * Инициализация поиска
     */
    _initSearch: function() {
        var rows = this.elem('row'),
            searchIndex = this.findBlockOutside('b-private-deals-table-search-index').params.searchIndex,
            items = searchIndex.map(function(searchIndexItem, i) {
                return {
                    searchIndex: searchIndexItem,
                    domElem: rows.eq(i)
                };
            });

        this._searchItems = items;

        this._iSearchable = BEM.create(
            { block: 'i-searchable', mods: { type: 'deals' } },
            { items: items }
        )
            .on('search.found', this._onSearchFound, this)
            .on('search.missed', this._onSearchMissed, this)
            .on('search.start', this._onSearchStart, this)
            .on('search.finish', this._onSearchFinish, this);
    },

    /**
     * Обработчик события совпадения
     * @private
     * @param {jQuery.Event} [e]
     * @param {Object} [data]
     *  @param {Object} data.item найденный элемент
     *  @param {Object} data.iterator итератор поиска
     */
    _onSearchFound: function(e, data) {
        var iterator = data.iterator,
            domElem = data.item.domElem,
            isVisible = iterator.empty || iterator.visible.length;

        this.setMod(domElem, 'hidden', isVisible ? 'no' : 'yes');

        if (isVisible) {
            this.highlight(data);
        }
    },

    /**
     * Обработчик события не совпадения
     * @private
     * @param {jQuery.Event} [e]
     * @param {Object} [data]
     * @param {Object} data.item не найденный элемент
     * @param {Object} data.iterator итератор поиска
     */
    _onSearchMissed: function(e, data) {
        this.setMod(data.item.domElem, 'hidden', 'yes');
    },

    /**
     * Обработчик события начала поиска
     * @private
     * @param {jQuery.Event} [e]
     * @param {Object} [data]
     * @param {Object} data.iterator итератор поиска
     */
    _onSearchStart: function(e, data) {},

    /**
     * Синхронизация выделения четных/нечетных строк
     * @param {jQuery} body
     * @private
     */
    _syncEvenOddHighlight: function(body) {
        this
            .delMod(this.findElem(body, 'row', 'pos', 'even'), 'pos')
            .setMod(this.findElem(body, 'row', 'hidden', 'no').filter(':even'), 'pos', 'even');
    },

    /**
     * Обработчик события окончания поиска
     * @private
     * @param {jQuery.Event} [e]
     * @param {Object} [data]
     * @param {Object} data.iterator итератор поиска
     */
    _onSearchFinish: function(e, data) {
        var isRowsFounded = !!data.iterator.found;

        this.setMod('filter-result', isRowsFounded ? '' : 'empty');

        if (isRowsFounded) {
            var body = $(u._.get(data, 'items[0].domElem')).parents(this.buildSelector('body'));

            this._syncColumnsWidth();
            this._syncEvenOddHighlight(body);
        }
    },

    /**
     * Сбрасывает кэш блоков для подсветки найденнго
     * @param {Number} [index]
     * @private
     */
    _dropHltBlocksCache: function(index) {
        if (u._.isUndefined(index)) {
            this._chacheHltBlocks = [];
        } else if (this._chacheHltBlocks) {
            this._chacheHltBlocks[index] = null;
        }
    },

    destruct: function() {
        this._iSearchable.destruct();

        this.__base.apply(this, arguments);
    }

},{
    live: function() {
        this
            .liveBindTo('row', 'click', function(e) {
                this._onRowClick(e);
            })
            .liveBindTo('head-sort', 'click', function(e) {
                this._onHeadCellClick(e);
            })
            .liveInitOnBlockInsideEvent('change', 'textinput', function(e, data) {
                if (this.elem('filter-input').is(e.block.domElem)) {
                    this._onFilterInputChange(e, data);
                }
            });

        return false;
    }
});
