/**
* @fires b-regions-selector#open-popup событие открытия попапа редактирования регионов показа
* @fires b-regions-selector#save событие сохранения настроек регионов в попапе
*/
BEM.DOM.decl('b-regions-selector', {

    onSetMod: {
        js: function() {
            this._subscribe = BEM.create('i-subscription-manager');
            this._regions = this.params.regions;

            // i-model не успевает создать модель не в ЛИ, используется явное создание через i-glue
            this._model = this.findBlockOn('i-glue').model;

            this.setDisabledRegions(this.params.disableRegions);

            this._subscribe.on(this._model, 'geo', 'change', function() {
                this.toggleMod(this.elem('geo-warning'), 'active', 'yes', '', !this._model.get('geo'));
                this.toggleMod(this.elem('selected-regions'), 'empty', 'yes', '', !this._model.get('geo'));
            }, this);

            this._switcher = this.findBlockOn('switcher', 'button');
            this._switcher && this._subscribe.on(this._switcher, 'click', this._onSwitcherClick, this);
        },

        disabled: function(modName, newValue) {
            this._switcher.setMod('disabled', newValue);

            this._popup && this._popup.hide({ force: true });
        }
    },

    getHintText: function() {
        return { isEmpty: this._model.get('geo') === '', text: this._model.get('geoText') };
    },

    /**
     * Возращает дата модель региона
     */
    getGeoModel: function() {
        return this._model;
    },

    /**
     * Устанавливает выбранные регионы
     * @param {String[]|undefined} regionIds список id выбранных регионов (остальные сбрасываются)
     */
    setSelectedRegions: function(regionIds) {
        var changesInfo = regionIds.split(', ').reduce(function(result, id) {
                id && (result[id] = true);

                return result;
            }, {}),
            geoChangesJson,
            data;

        data = {
            geo: regionIds,
            geoText: regionIds.length ? u.getGeoNames(regionIds) : iget2('b-regions-selector', 'empty', 'Регионы показа не заданы.'),
            changesInfo: changesInfo
        };

        if (regionIds.length) {
            geoChangesJson = this._getGeoChangesFromSavingData(data);
        } else {
            geoChangesJson = {};
        }

        this._model.update({
            geo: data.geo,
            geoText: data.geoText,
            geoChangesJson: JSON.stringify(geoChangesJson),
            wasReset: true
        });
    },

    /**
     * Устанавливает недоступные к выбору регионы
     * @param {String[]|undefined} regionIds список id недоступных регионов
     */
    setDisabledRegions: function(regionIds) {
        var currentGeo = this._model.get('geo'),
            geoChanges = this._model.get('geoChangesJson') ? JSON.parse(this._model.get('geoChangesJson')) : {},
            newGeo,
            deleteMinusRegions = false;

        this._disableRegions = regionIds;

        if (!(regionIds || []).length || !currentGeo) {
            return;
        }

        // если недоступные регионы оказались среди выбранных ранее - надо их удалить
        newGeo = currentGeo.split(',').reduce(function(res, item) {
            if (regionIds.indexOf(Math.abs(item) + '') === -1 && (item > 0 || !deleteMinusRegions)) {
                res.push(item);
                deleteMinusRegions = false;
            } else {
                deleteMinusRegions = true;
            }

            return res;
        }, []).join(',');

        regionIds.forEach(function(disabledRegionId) {
            if (geoChanges[disabledRegionId]) {
                delete geoChanges[disabledRegionId];
            }
        });

        this._model.update({
            geo: newGeo,
            geoText: u.getGeoNames(newGeo),
            geoChangesJson: JSON.stringify(geoChanges)
        });
    },

    /**
     * Открывает попап редактирования настроек регионов
     * @returns {BEM} блок попапа выбора региона
     * @private
     */
    openRegionsPopup: function() {
        var popup = BEM.DOM.blocks['b-modal-popup-decorator'].create2(undefined, { bodyScroll: false }, $),
            editRegionsPopupBlock;

        this._popup = popup;

        popup
            .on('close-blocked', function() {
                BEM.blocks['b-confirm'].open({
                    message: iget2('b-regions-selector', 'izmeneniya-ne-budut-sohraneny', 'Изменения не будут сохранены. Продолжить?'),
                    onYes: function() {
                        popup.hide({ force: true });
                    }
                });
            }, this)
            .on('close', function() {
                // нужно прервать запросы с валидацией, чтобы не вызвать неожиданных проблем
                this._abortRegionsPopupRequests();
                popup.destruct();
                this._popup = null;
            }, this);

        editRegionsPopupBlock = popup.setPopupContent({
            block: 'b-edit-regions-popup',
            hasBidsCorrections: this.params.hasBidsCorrections,
            isBidsCorrectionsModeOn: !!this._model.get('isBidsCorrectionsModeOn'),
            isCommon: this.params.isCommon,
            disableEditBids: this.params.disableEditBids,
            disableEditBidsHint: this.params.disableEditBidsHint,
            regions: this._regions,
            selectedRegions: (this._model.get('geo') || '') + '',
            disableRegions: this._disableRegions,
            independentRegions: this.params.independentRegions,
            quickSelectRegions: this.params.quickSelectRegions,
            quickSelectUserRegion: this.params.quickSelectUserRegion,
            hasIndependentRegionsWarning: this._model.get('hasIndependentRegionsWarning'),
            hintForDifferentRegions: this.params.hintForDifferentRegions
        });

        editRegionsPopupBlock.initialize({
            parentPopup: popup.getPopup()
        });

        editRegionsPopupBlock.on('save', this._onRegionsPopupSave, this);

        editRegionsPopupBlock.on('cancel', function() {
            popup.hide({ force: true });
        }, this);

        popup.show();

        this.trigger('open-popup');

        return editRegionsPopupBlock;
    },

    _onSwitcherClick: function() {
        this._editRegionsBlock = this.openRegionsPopup();
    },

    /**
     * Обработчик события save блока b-edit-regions-popup
     * @param {Event} event
     * @param {Object} data
     * @private
     */
    _onRegionsPopupSave: function(event, data) {
        var newGeoData = {
                geo: data.geo,
                geoText: data.geoText,
                geoChangesJson: JSON.stringify(this._getGeoChangesFromSavingData(data)),
                hasIndependentRegionsWarning: data.hasIndependentRegionsWarning
            },
            bidsToReset;

        this._editRegionsBlock.setMod('progress', 'yes');

        if (this.params.hasBidsCorrections && !this.params.disableEditBids) {
            // для сброса значения ставки на сервер отправляем значение 0
            bidsToReset = Object.keys(this.params.bids).reduce(function(result, initialBidItemId) {
                (newGeoData.bids || {})[initialBidItemId] === undefined && (result[initialBidItemId] = 0);

                return result;
            }, {});

            u._.extend(newGeoData, {
                bids: data.bids,
                // значение ставки на сервере хранится со смещением
                bidsJson: JSON.stringify(Object.keys(data.bids).reduce(function(result, regionId) {
                    result[regionId] = data.bids[regionId] + u.consts('bidOffset');

                    return result;
                }, bidsToReset)),
                isBidsCorrectionsModeOn: +data.isBidsCorrectionsModeOn
            });
        }

        this._editRegionsBlock.hideError();

        this._validateRegions(newGeoData)
            .then(function() {
                this._model.update(newGeoData);

                this._updateRegionsInfo({
                    bids: data.bids,
                    resolvedRegions: data.resolvedRegions && Object.keys(data.resolvedRegions).length ?
                        data.resolvedRegions :
                        undefined
                });

                this.trigger('save');

                this.setMod(
                    this.elem('bids-hint'),
                    'hidden',
                    data.isBidsCorrectionsModeOn && Object.keys(data.bids).length ? '' : 'yes'
                );

                this._popup.hide({ force: true });
            }, function(errorText) {
                this._editRegionsBlock.showError(errorText);
            }).always(function() {
                this._editRegionsBlock.delMod('progress');
            }.bind(this));
    },

    /**
     * Валидирует выбранные регионы
     * @param {Object} data
     * @return {Promise<String[]>}
     * @private
     */
    _validateRegions: function(data) {
        var deferred = $.Deferred();

        if (data.geo === 0) {
            deferred.rejectWith(this, [iget2('b-regions-selector', 'neobhodimo-zadat-regiony-pokaza', 'Необходимо задать регионы показа')]);
        } else {
            deferred.resolveWith(this, []);
        }

        return deferred.promise();
    },

    /**
     * Удаляет блок, модель и подписки i-subscription-manager
     * @override
     */
    destruct: function() {
        this._subscribe.dispose();
        this._popup && this._popup.hide({ force: true });

        return this.__base.apply(this, arguments);
    },

    /**
     * Вычисляет объект изменений регионов для отправки серверу
     * @param {Object} data данные сохраняемого дерева регионов
     * @param {Object|0} data.geo id выбранных регионов, разделенных запятой
     * @param {Object} data.changesInfo информация о том, в каких галках регионов были изменения
     * и были приведены к единому виду
     * @param {Boolean} data.wasReset флаг, идентифицирующий факт сброса всех регионов
     * @returns {Object}
     */
    _getGeoChangesFromSavingData: function(data) {
        var newRegions,
            result = {},
            currentGeoChanges = this._model.get('geoChangesJson') ? JSON.parse(this._model.get('geoChangesJson')) : {};

        // если пользователь сбрасывал регионы,
        // то серверу, надо сформировать данные отмененных регионов и полем merge_geo: 0
        if (data.wasReset || currentGeoChanges.merge_geo !== undefined) {
            newRegions = data.geo === 0 ?
                {} :
                data.geo.split(',').reduce(function(res, geoItem) {
                    res[Math.abs(geoItem)] = {
                        is_negative: geoItem > 0 ? 0 : 1
                    };

                    return res;
                }, {});

            u._.assign(result, newRegions, { merge_geo: 0 });
        } else {
            newRegions = Object.keys(data.changesInfo).reduce(function(res, regionId) {
                res[regionId] = {
                    is_negative: data.changesInfo[regionId] ? 0 : 1
                };

                return res;
            }, {});

            // в задаче https://st.yandex-team.ru/DIRECT-65840 нашли, что в сценарии
            // добавить новый регион + удалить новый регион новый регион добавляется
            // поэтому меняю result, currentGeoChanges на currentGeoChanges, result
            u._.assign(result, currentGeoChanges, newRegions);
        }

        return result;
    },

    /**
     * Обновляет значения ставок регионов
     * @param {Object} newInfo
     * @param {Object | undefined} newInfo.bids данные ставок для регионов
     * @param {Object | undefined} newInfo.resolvedRegions данные о регионах,
     * для которых регионы в группах кампании различались и были приведены к единому виду
     */
    _updateRegionsInfo: function(newInfo) {
        var updateInfoForRegion = function(region, parentRegion) {
            var bidValue;

            if (newInfo.bids) {
                bidValue = newInfo.bids[region.id];

                if (parentRegion) {
                    region.defaultBid = parentRegion.bid != undefined ? parentRegion.bid : parentRegion.defaultBid;
                }

                region.bid = bidValue;
            }

            if (newInfo.resolvedRegions && newInfo.resolvedRegions[region.id]) {
                region.contrastValueGroups = undefined;
            }

            region.inner && region.inner.forEach(function(innerRegion) {
                updateInfoForRegion(innerRegion, region);
            });
        };

        if (!newInfo.bids && !newInfo.resolvedRegions) {
            return;
        }

        this._regions.forEach(function(region) {
            updateInfoForRegion(region);
        });
    },

    /**
     * Остановить все запросы при закрытии попапа с регионами
     * @protected
     */
    _abortRegionsPopupRequests: function() {},

    /**
     * Включает видимость кнопки выключения/выключения outboard controls
     * @param {Boolean} [visible] по умолчанию выставлена как true
     */
    _toggleSwitchButtonVisibility: function(visible) {
        arguments.length || (visible = true);

        this.toggleMod(this.elem('switcher'), 'invisible', 'yes', !visible);
    }
});
