/**
 * Редактирование условия подбора аудитории
 * popup обертка над блоком из bricks
 *
 * @param {Boolean} params.isExist существует ли запись или создаем новую
 *
 * @fires:
 * Событие после успешного сохранения
 * created ( condition: {RetargetingConditionDTO} )     // создали новое условие
 * changed ( condition: {RetargetingConditionDTO} )     // изменили данные условия
 *
 * @namespace b-retargeting-condition-edit-popup
 */
BEM.DOM.decl({
    block: 'b-retargeting-condition-edit-popup',
    implements: 'i-modal-popup-inner-block-interface'
}, /** @lends b-retargeting-condition-edit-popup */{

    /**
     * блок редактирования из bricks - обеспечивает всю функциональность
     * @type {b-retargeting-condition-edit}
     */
    _editBlock: null,

    /**
     * @type {button} Блок кнопки сохранения
     */
    _saveButton: null,

    /**
     * @type {button} Блок кнопки отмены редактирования (закрывает popup даже если есть не сохраненные изменения)
     */
    _cancelButton: null,

    /**
     * @type {button} Блок кнопки сохранить как новую - показывается в случае редактирования существующей записи
     */
    _saveAsNewButton: null,

    /**
     * @type {b-modal-popup-decorator} объект popup, в котором находится блок.
     * Записывается в это свойство при создании в статическом методе "createInstance"
     */
    _popup: null,

    onSetMod: {
        /**
         * @lends b-retargeting-condition-edit-popup
         */
        js: function() {
            u.graspSelf.call(this, {
                _editBlock: 'b-retargeting-condition-wizard inside body',
                _saveButton: 'button inside footer-but-save',
                _cancelButton: 'button inside footer-but-cancel',
                _saveAsNewButton: '? button inside footer-but-save-as-new'
            });

            this._saveButton.on('click', this._onSave, this);
            this._cancelButton.on('click', this._onCancel, this);
            this._saveAsNewButton && this._saveAsNewButton.on('click', this._onSaveAsNew, this);

            this._editBlock
                .on('change', function(e, params) { this._onChange(params) }, this)
                .on('warning', function(e, params) { this._onWarning(params) } , this)
                .on('ready', function() { this.trigger('repaint') }, this);
        }
    },

    /**
     * Возвращает - было ли изменение, требующих сохранение
     * @returns {$.Deferred<Boolean>}
     */
    isChanged: function() {
        var deferred = $.Deferred();
        deferred.resolve(this._editBlock.isChanged());
        return deferred;
    },

    /**
     * Обрабатывает предупреждения дочерних блоков
     * @param {{ msg:String }} reason
     * @private
     */
    _onWarning: function(reason) {
        BEM.blocks['b-confirm']
            .alert(reason.msg, this._popup && this._popup.getPopup());
    },

    /**
     * Отменяет редактирование. Закрывает popup
     * @private
     */
    _onCancel: function() {
        this._popup && this._popup.hide({ force: true });
    },

    /**
     * Сохраняет изменение
     * @returns {Promise<{ success: Boolean, [errorMsg]: String, condition: RetargetingConditionDTO }>}
     * @private
     */
    _onSave: function() {
        this._disableBtnSave(true);
        this._editBlock.save()
            .then(function(response) {
                if (response.success) {
                    switch (response.operation) {
                        case 'CREATE': this.trigger('created', response.condition); break;
                        case 'CHANGE': this.trigger('changed', response.condition); break;
                    }
                    this._onCancel();
                } else {
                    this._disableBtnSave(false);
                }
            }.bind(this));
    },

    /**
     * Сохраняет изменение в существующей записи как новой
     * @returns {Promise<{ success: Boolean, [errorMsg]: String, condition: RetargetingConditionDTO }>}
     * @private
     */
    _onSaveAsNew: function() {
        this._disableBtnSaveAsNew(true);
        this._editBlock.save('SAVE_AS_NEW')
            .then(function(response) {
                if (response.success) {
                    this.trigger('created', response.condition);
                    this._onCancel();
                } else {
                    this._disableBtnSaveAsNew(false);
                }
            }.bind(this));
    },

    /**
     * Обработчик изменений в данных в дочерних блоках
     * @param {{ isChanged, isValid, saveAsNew, mode }} params
     * @private
     */
    _onChange: function(params) {
        this._disableBtnSave(!(params.isChanged && params.isValid));
        if (this._saveAsNewButton && params.mode === 'edit') {
            this._disableBtnSaveAsNew(!params.saveAsNew);
        }
    },

    /**
     * Устанавливает состояние кнопки сохранения
     * @param {Boolean} disabled
     * @private
     */
    _disableBtnSave: function(disabled) {
        disabled ? this._saveButton.setMod('disabled', 'yes') : this._saveButton.delMod('disabled');
    },

    /**
     * Устанавливает состояния кнопки "сохранить как новую"
     * @param {Boolean} disabled
     * @private
     */
    _disableBtnSaveAsNew: function(disabled) {
        disabled ? this._saveAsNewButton.setMod('disabled', 'yes') : this._saveAsNewButton.delMod('disabled');
    }
}, {

    /**
     * Создает блок редактирования условия аудитории в popup
     * Блок вызывается в нескольких местах - что бы не возникало дублирование (создание popup, создание контента для
     * него)
     * @param {{ [parentPopup]: popup, conditionId: Number|*, showBtnSaveAsNew: Boolean }} params
     *
     * @returns {b-retargeting-condition-edit-popup}
     * @public
     */
    createInstance: function(params) {
        var modalPopupDecorator = BEM.DOM.blocks['b-modal-popup-decorator'].create2(null, { bodyScroll: false }, $),
            popup = modalPopupDecorator.getPopup(),
            instance;

        params.parentPopup && popup.setParent(params.parentPopup);
        instance = modalPopupDecorator.on('close-blocked', function() {
            BEM.blocks['b-confirm'].open({
                message: iget2('b-retargeting-condition-edit-popup', 'izmeneniya-ne-budut-sohraneny', 'Изменения не будут сохранены. Продолжить?'),
                onYes: function() {
                    modalPopupDecorator.hide({ force: true });
                },
                fromPopup: modalPopupDecorator.getPopup()
            });
        }, this)
            .setPopupContent({
                block: 'b-retargeting-condition-edit-popup',
                conditionId: +params.conditionId || 0,
                showBtnSaveAsNew: params.showBtnSaveAsNew !== false
            })
            .on('cancel', function() {
                this.hide({ force: true });
            }, modalPopupDecorator);

        popup.on('hide', function() {
            modalPopupDecorator.destruct();
        });

        instance._popup = modalPopupDecorator;
        modalPopupDecorator.show();

        return instance;
    }
});
