BEM.DOM.decl({
    block: 'b-sitelinks-selector',
    implements: 'i-outboard-controls',
    baseBlock: 'i-glue'
}, {

    onSetMod: {
        js: function() {
            this.model = BEM.MODEL.getOne(this.params.modelParams) ||
                BEM.MODEL.create(this.params.modelParams, { modelId: this.params.modelParams.id });

            this._titles = {};
            this._descriptions = {};
            this._hrefs = {};
            this._urlProtocols = {};
            this._hrefsValidations = [];
            this._turbolandings = {};

            this._hasErrors = false;

            this._forEachSitelink(function(i) {
                i = i.toString();
                this._hrefsValidations[i] = 1;

                this.model
                    .on('href' + i, 'change', { index: i, name: 'href' }, this._onModelFieldChange, this)
                    .on('title' + i, 'change', { index: i, name: 'title' }, this._onModelFieldChange, this)
                    .on('url_protocol' + i, 'change',
                        { index: i, name: 'url_protocol' },
                        this._onModelFieldChange, this)
                    .on('description' + i,
                        'change',
                        { index: i, name: 'description' },
                        this._onModelFieldChange, this);

                this._titles[i] = this.findBlockOn(this.elem('title', 'index', i), 'input')
                    .on('change', { index: i, name: 'title' }, this._onInputChange, this)
                    .on('blur', { index: i, name: 'title' }, this._onInputBlur, this);

                this._hrefs[i] = this.findBlockOn(this.elem('href', 'index', i), 'input')
                    .on('change', { index: i, name: 'href' }, this._onInputChange, this)
                    .on('blur', { index: i, name: 'href' }, this._onInputBlur, this);

                this._urlProtocols[i] = this.findBlockOn(this.elem('url-protocol', 'index', i), 'select')
                    .on('change', { index: i, name: 'url_protocol' }, this._onInputChange, this);

                this._descriptions[i] = this.findBlockOn(this.elem('description', 'index', i), 'input')
                    .on('change', { index: i, name: 'description' }, this._onInputChange, this)
                    .on('blur', { index: i, name: 'description' }, this._onInputBlur, this);
            });

            this
                .bindTo('copy', 'click', this._copy)
                .bindTo('clear', 'click', this._clear)
                .bindTo('toggler-link', 'click', function() {
                    this._toggleDescriptions();
                });
        }
    },

    /**
     * Возвращает массив полей
     * @return {Array<String>}
     * @private
     */
    _getFields: function() {
        return ['title', 'href', 'url_protocol', 'description']
    },

    /**
     * Возвращает контрол для ввода урла/имени сайтлинка с данным номером
     * @param {'title'|'href'} name - имя/урл сайтлинка
     * @param {Number} index - номер сайтлинка
     * @returns {BEM}
     */
    _getControl: function(name, index) {
        var cached = {
            title: '_titles',
            description: '_descriptions',
            href: '_hrefs',
            url_protocol: '_urlProtocols',
            turbolanding: '_turbolandings'
        };

        return this[cached[name]][index];
    },

    /**
     * Инпут для ввода урла/названия сайтлинка изменился
     * @param {Object} e
     * @param {Object} e.data
     * @param {'title'|'href'} e.data.name - имя/урл сайтлинка
     * @param {Number} e.data.index - номер сайтлинка
     * @returns {BEM}
     * @private
     */
    _onInputChange: function(e) {
        var name = e.data.name,
            index = e.data.index,
            block = e.block,
            value = $.trim(this._getControl(name, index).val());

        this.model.set(name + index, value);

        if (name == 'href' && block.getMod('focused') == 'yes') {
            var protocol = u.getUrlProtocol(value);

            protocol && this.model.set('url_protocol' + index, protocol);
        }

        return this;
    },

    /**
     * Инпут для ввода урла/названия сайтлинка потерял фокус
     * @param {Object} e
     * @param {Object} e.data
     * @param {'title'|'href'} e.data.name  - имя/урл сайтлинка
     * @param {Number} e.data.index - номер сайтлинка
     * @returns {BEM}
     * @private
     */
    _onInputBlur: function(e) {
        var name = e.data.name,
            index = e.data.index;

        this._getControl(name, index).val(this.model.get(name + index, 'format'));

        return this;
    },

    /**
     * Выполнить callback для каждого сайтлинка
     * @param {Function} callback
     * @returns {BEM}
     * @private
     */
    _forEachSitelink: function(callback) {
        for (var i = 0; i < CONSTS.SITELINKS_NUMBER; i++) {
            callback.call(this, i);
        }

        return this;
    },

    /**
     * Срабатывает на изменении поля модели сайтлинков
     * @param {Object} e
     * @param {Object} e.data
     * @param {'title'|'href'} e.data.name - имя/урл сайтлинка
     * @param {String} e.data.index - номер сайтлинка
     * @param {Object} data
     * @private
     */
    _onModelFieldChange: function(e, data) {
        var name = e.data.name,
            index = e.data.index,
            // хак для DIRECT-37725, который пришлось сделать, т. к. в основной блок почему-то просочились знания про модификатор
            isDifferentLink = e.data.isDifferentLink,
            control = this._getControl(name, index),
            isInput = name !== 'url_protocol' && name !== 'turbolanding';
        // сейчас селекта только два, возможно потом проверку надо будет переделать

        if (name === 'turbolanding') {
            // этот костыль нужен потому, что в момент обновления id
            // в поле `'turbolanding' + index` модель, у которой обновлены не все поля
            setTimeout((function() {
                control.setValue(this.model.get(name + index).toJSON());
            }).bind(this), 0);
        } else {
            control.getMod('focused') !== 'yes' && control.val(data.value);
        }

        if (isInput) {
            control.setMod(
                control.elem('hint-manual'),
                'visibility',
                (!control.val() && isDifferentLink) ? 'visible' : ''
            );

            if (name === 'href') {
                this._hideHrefWarning(index);
            }

            this._updateCounters();
        }

        this._checkHref(index);
    },

    /**
     * Возвращает длину заголовка сайтлинка с заданным номером
     * @param {Number} i номер быстрой ссылки
     * @returns {Number}
     * @private
     */
    _getTitleLength: function(i) {
        return this.model.get('title' + i).length;
    },

    /**
     * Возвращает длину урла сайтлинка с заданным номером
     * @param {Number} i i номер быстрой ссылки
     * @returns {Number}
     * @private
     */
    _getHrefLength: function(i) {
        return this.model.get('href' + i).length;
    },

    /**
     * Обновить состояние счетчиков
     * @returns {BEM}
     * @private
     */
    _updateCounters: function() {
        var totalTitle = this.model.get('title'),
            totalTitleLength = totalTitle && totalTitle.length || 0,
            titleLength,
            hasTitleError,
            hasTotalTitleError,
            hasDescriptionLenghtError,
            hrefMaxLength = this.params.hrefMaxLength,
            descriptionMaxLength = u.consts('ONE_SITELINK_DESC_MAX_LENGTH'),
            siteLinksMaxLength = u.consts('SITELINKS_MAX_LENGTH'),
            oneSitelinkMaxLength = u.consts('ONE_SITELINK_MAX_LENGTH');

        this._forEachSitelink(function(i) {
            titleLength = this._getTitleLength(i);

            // title counter
            this.elem('title-counter', 'index', '' + i).text(titleLength);

            this.elem('title-total-counter', 'index', '' + i).text(siteLinksMaxLength - totalTitleLength);

            this.setMod(
                this.elem('title-counter', 'index', '' + i),
                'overflow',
                titleLength > oneSitelinkMaxLength ? 'yes' : 'no');

            hasTitleError = hasTitleError || (titleLength > oneSitelinkMaxLength);

            this.setMod(
                this.elem('title-total-counter', 'index', '' + i),
                'overflow',
                totalTitleLength > siteLinksMaxLength ? 'yes' : 'no');

            hasTotalTitleError = hasTotalTitleError || (totalTitleLength > siteLinksMaxLength);

            // href counter
            var hrefLength = this._getHrefLength(i);

            this.elem('href-counter', 'index', '' + i)
                .text(hrefMaxLength - hrefLength);

            this.setMod(
                this.elem('href-counter', 'index', '' + i),
                'overflow',
                hrefLength > hrefMaxLength ? 'yes' : 'no');

            // href counter
            var descriptionLength = this.model.get('description' + i).length;

            this.elem('description-counter', 'index', '' + i)
                .text(descriptionMaxLength - descriptionLength);

            this.setMod(
                this.elem('description-counter', 'index', '' + i),
                'overflow',
                descriptionLength > descriptionMaxLength ? 'yes' : 'no');

            hasDescriptionLenghtError = hasDescriptionLenghtError || descriptionLength > descriptionMaxLength;
        });

        this
            .toggleMod(this.elem('error', 'type', 'length'), 'show', 'yes', hasTitleError)
            .toggleMod(this.elem('error', 'type', 'total-length'), 'show', 'yes', hasTotalTitleError)
            .toggleMod(this.elem('error', 'type', 'description-length'), 'show', 'yes', hasDescriptionLenghtError);

        return this;
    },

    /**
     * Проверить все введенные ссылки
     * @returns {BEM}
     * @private
     */
    _checkAllHrefs: function() {
        if (!this.model.isEmpty()) {
            this._forEachSitelink(function(i) {
                this._checkHref(i);
            });
        }

        return this;
    },

    /**
     * Проверка, что в ссылке хотя бы что-то заполнено
     * @private
     */
    _isEmpty: function(index) {
        var href = this.model.get('href' + index),
            title = this.model.get('title' + index);

        return !title.match(/\S/) && !href.match(/\S/);
    },

    /**
     * Проверить ссылку на корректность
     * @param {Number} index
     * @returns {BEM}
     * @private
     */
    _checkHref: function(index) {
        var href = this.model.get('href' + index),
            title = this.model.get('title' + index),
            protocol = this.model.get('url_protocol' + index),
            turbolanding = this.model.get('turbolanding' + index),
            isEqualTurbolanding = false,
            isEqualProtocol = true,
            mBanner = BEM.MODEL.get({ name: 'm-banner' }),
            _this = this,
            hrefWarnings = [],
            hasHref = this.bannerModel && this.bannerModel.get('has_href');

        this._hasErrors = false;

        // если быстрая ссылка не пустая и не идет полное обновление модели
        if (!this._disableValidation && !this._isEmpty(index)) {

            // 1. должен быть заполнен заголовок
            if (!title.match(/\S/)) {
                hrefWarnings.push(iget2('b-sitelinks-selector', 'neobohdimo-zapolnit-tekst-bystroj-ssylki', 'Необходимо заполнить текст для быстрой ссылки.'));
                this._hasErrors = true;
            }

            // 2. должен быть заполнен адрес или задан турболендинг
            var isHrefEmpty = !href.match(/\S/);
            if (isHrefEmpty && !(turbolanding && (turbolanding.get('id') > 0))) {
                hrefWarnings.push(iget2('b-sitelinks-selector', 'required-address-or-turbo', 'Должен быть задан адрес сайта или турболендинг'));
                this._hasErrors = true;
            } else {

                if (this.bannerModel) {
                    isEqualTurbolanding = this._checkEqualToBannerTurbolanding(turbolanding, this.bannerModel);
                } else {
                    mBanner.forEach(function(model) {
                        isEqualTurbolanding ||
                            (isEqualTurbolanding = _this._checkEqualToBannerTurbolanding(turbolanding, model));

                        isEqualProtocol &&
                            (isEqualProtocol = protocol === model.get('sitelinks').get('url_protocol' + index));
                    });
                }

                // 3. должен быть одинаковый протокол в объявлениях в массовых действиях
                if (!isEqualProtocol) {
                    hrefWarnings.push(iget2('b-sitelinks-selector', 'v-vybrannyh-obyavleniyah-raznye-108', 'В выбранных объявлениях разные протоколы.'));
                }

                // 4. не должна вести на турбо-страницу
                if (u.turbo.isLinkTurbo(href)) {
                    hrefWarnings.push(iget2('b-sitelinks-selector', 'no-turbolanding-url', 'Нельзя указывать ссылку на турбо-страницу.'));
                    this._hasErrors = true;
                }

                !this._hasErrors && !(isHrefEmpty || !u.isUrl(href)) && u.urlCheck({
                    url: href,
                    callback: function(data) {
                        if (_this._hasErrors) {
                            return; // пока проверяли ссылку в форме появилась ошибка, например очистили текст
                        }
                        var modelUrl = this.model.get('href' + index),
                            isEqualDomain = true;

                        if (!data || data.url !== modelUrl) {
                            // значение, результат проверки которого мы получили, уже изменилось
                            return;
                        }

                        // 5. С сервера должен прийти положительный результат проверки
                        if (!data.code) {
                            hrefWarnings.push(data.text);
                            this._hasErrors = true;
                        } else {

                            if (this.bannerModel) {
                                isEqualDomain = this._checkDomain(index, data.domain, this.bannerModel);
                            } else if (this.hasMod('mode', 'multi')) {
                                mBanner.forEach(function(model) {
                                    isEqualDomain && (isEqualDomain = _this._checkDomain(index, data.domain, model));
                                });
                            }

                            // 6. Должен быть одинаковый домен
                            if (hasHref && !isEqualDomain) {
                                hrefWarnings.push(iget2('b-sitelinks-selector', 'ssylka-ne-vedet-na', 'Ссылка не ведет на страницу основного сайта'));
                            }
                        }

                        if (hrefWarnings.length) {
                            this._showHrefWarning(index, hrefWarnings, this._hasErrors);
                            this._hrefsValidations[index] = this._hasErrors ? 0 : 1;
                        } else {
                            this._hideHrefWarning(index);
                            this._hrefsValidations[index] = 1;
                        }

                        this._validateSitelinks();
                    },
                    ctx: this,
                    protocol: protocol
                });
            }
        }

        if (hrefWarnings.length) {
            this._showHrefWarning(index, hrefWarnings, this._hasErrors);
            this._hrefsValidations[index] = this._hasErrors ? 0 : 1;
        } else {
            this._hideHrefWarning(index);
            this._hrefsValidations[index] = 1;
        }

        this._validateSitelinks();
    },

    /**
     * Проверяет на полное совпадение сайтлинка с урлом баннера
     * @param {String} href
     * @param {MODEL} model
     * @returns {Boolean}
     * @private
     */
    _checkEqualToBannerHref: function(href, model) {
        return this._compareLinks(model.get('href_model').get('href'), href);
    },

    /**
     * Проверяет на совпадение турбо-страницы сайтлинка с турбо-страницей баннера
     * (переобпределено в модификаторе)
     * @returns {Boolean}
     * @private
     */
    _checkEqualToBannerTurbolanding: function() {
        return false;
    },

    /**
     * Проверить соответствует ли ссылка домену баннера
     * @param {Number} index
     * @param {String} hrefDomain
     * @param {MODEL} model
     * @returns {Boolean}
     * @private
     */
    _checkDomain: function(index, hrefDomain, model) {
        var bannerHrefModel = model.get('href_model'),
            domain = bannerHrefModel && bannerHrefModel.get('domain'),
            domainRedir = bannerHrefModel && bannerHrefModel.get('domain_redir'),
            domainsExclusionsRegExp = new RegExp(this.params.domainsExclusionsRegExp);

        return ('.' + domainRedir).indexOf('.' + hrefDomain) !== -1 ||
            ('.' + domain).indexOf('.' + hrefDomain) !== -1 ||
            domainsExclusionsRegExp.test(hrefDomain);
    },

    /**
     * Сравнивает два урла между собой, извлекая урлы по их индексам
     * @param {Number} index1
     * @param {Number} index2
     * @returns {Boolean}
     */
    _isLinksEqual: function(index1, index2) {
        return this._compareLinks(this.model.get('href' + index1), this.model.get('href' + index2))
    },

    /**
     * Сравнивает два урла между собой
     * @param {String} href1
     * @param {String} href2
     * @returns {Boolean}
     * @private
     */
    _compareLinks: function(href1, href2) {
        if (u.isEmpty(href1) || u.isEmpty(href2)) return false;

        href1 = u.stripWww(href1);
        href2 = u.stripWww(href2);

        return href1.replace(/\/+$/g, '') == href2.replace(/\/+$/g, '');
    },

    /**
     * Сравнивает между собой два заголовка
     * @param {Number} index1
     * @param {Number} index2
     * @returns {Boolean}
     * @private
     */
    _isTitleEqual: function(index1, index2) {
        return this.model.get('title' + index1) == this.model.get('title' + index2);
    },

    /**
     * Проверить все сайтлинки
     * @returns {BEM}
     * @private
     */
    _validateSitelinks: function() {
        var isTitleEmpty,
            isHrefEmpty,
            isDescriptionOnly,
            isEqualTitlesError = false,
            isEqualTurbolandingsError = false,
            isEmptyFieldsError = false,
            isDescriptionValid = this._isDescriptionValid(),
            isHrefValid = this._hrefsValidations.every(function(val, i) {
                return val;
            }, this);

        this._forEachSitelink(function(i) {
            isTitleEmpty = this.model.isEmpty('title' + i);
            isHrefEmpty = this.model.isEmpty('href' + i);

            isEmptyFieldsError = isEmptyFieldsError || this._isEmptyFieldsError(i);

            isDescriptionOnly = isDescriptionOnly ||
                (isTitleEmpty && isHrefEmpty && !this.model.isEmpty('description' + i) && !this._hasDesktopLanding(i));

            isEqualTitlesError = isEqualTitlesError || this._isTitleEqualToOther(i);
            isEqualTurbolandingsError = this.bannerModel && this.bannerModel.get('has_site') &&
                isEqualTurbolandingsError || this._isTurbolandingEqualToOther(i);
        });

        this.setMod(this.elem('error', 'type', 'equal-titles'), 'show', isEqualTitlesError ? 'yes' : '')
            .setMod(this.elem('error', 'type', 'equal-turbolandings'), 'show', isEqualTurbolandingsError ? 'yes' : '');

        if (this.model.isEmpty() || this.model.isValid() && isHrefValid && !isEmptyFieldsError &&
            !isEqualTurbolandingsError && !isEqualTitlesError &&
            isDescriptionValid && !isDescriptionOnly) {
            this._onValid();
        } else {
            this._onInvalid();
        }

        return this;
    },

    /**
     * Проверка на наличие десктопного турболендинга
     * @param {Number} index - индекс ссылки
     * @return {Boolean}
     * @private
     */
    _hasDesktopLanding: function(index) {
        var turbolanding = this.model.get('turbolanding' + index);
        return turbolanding && +turbolanding.get('id');
    },

    /**
     * Проверка на правильность заполнения полей
     * @param {Number} index - индекс ссылки
     * @return {Boolean}
     * @private
     */
    _isEmptyFieldsError: function(index) {
        var isTitleEmpty = this.model.isEmpty('title' + index),
            isHrefEmpty = this.model.isEmpty('href' + index),
            isDesktopLandingEmpty = !this._hasDesktopLanding(index);

        //обрабатываем ситуации заполненное название + пустой (урл|десктопный лендинг)
        //и пустое название + заполненный (урл|десктопный лендинг)
        return !isTitleEmpty && isHrefEmpty && isDesktopLandingEmpty ||
            ((!isHrefEmpty || !isDesktopLandingEmpty) && isTitleEmpty);
    },

    /**
     * Проверяет валидность описания сайтлинков
     * @returns {boolean}
     * @private
     */
    _isDescriptionValid: function() {
        var haveSameDescriptions = false,
            model = this.model,
            allDescriptions = [];

        this._forEachSitelink(function(i) {
            if (!haveSameDescriptions && !model.isEmpty('description' + i) &&
                allDescriptions.indexOf(model.get('description' + i)) > -1) {

                haveSameDescriptions = true;
            }

            allDescriptions.push(model.get('description' + i));
        });

        this.toggleMod(this.elem('error', 'type', 'equal-descriptions'), 'show', 'yes', haveSameDescriptions);

        return !haveSameDescriptions;
    },

    /**
     * Проверяет заголовок сайтлинка на совпадения с остальными
     * @param {Number} index номер сайтлинка
     * @returns {Boolean}
     * @private
     */
    _isTitleEqualToOther: function(index) {
        var isTitleEmpty = this.model.isEmpty('title' + index),
            isEqual = false,
            title = this.model.get('title' + index);

        if (isTitleEmpty) return false;

        this._forEachSitelink(function(i) {
            if (i != index) isEqual = isEqual || this._isTitleEqual(index, i);
        });

        return isEqual;
    },

    /**
     * Проверяет урл сайтлинка на совпадение с остальными
     * @param {Number} index номер сайтлинка
     * @returns {Boolean}
     * @private
     */
    _isHrefEqualToOther: function(index) {
        var isHrefEmpty = this.model.isEmpty('href' + index),
            isEqual = false;

        if (isHrefEmpty) return false;

        this._forEachSitelink(function(i) {
            if (i != index) isEqual = isEqual || this._isLinksEqual(index, i);
        });

        return isEqual;
    },

    /**
     * Проверяет турбо-страницу сайтлинка на совпадение с остальными
     * @param {Number} index номер сайтлинка
     * @returns {Boolean}
     * @private
     */
    _isTurbolandingEqualToOther: function(index) {
        var id = +this.model.get('turbolanding' + index).get('id'),
            isEqual = false;

        if (!id) return false;

        this._forEachSitelink(function(i) {
            var currentId = +this.model.get('turbolanding' + i).get('id');

            if (i !== index && id && id === currentId) {
                isEqual = true;
            }
        });

        return isEqual;
    },

    /**
     * Срабатывает при успешной проверке сайтлинков
     * @returns {BEM}
     * @private
     */
    _onValid: function() {
        this.trigger('state', { state: 'valid' });

        return this;
    },

    /**
     * Срабатывает при неуспешной проверке сайтлинков
     * @returns {BEM}
     * @private
     */
    _onInvalid: function() {
        this.trigger('state', { state: 'invalid' });

        return this;
    },

    /**
     * Показать предупреждение для некорректной ссылки
     * @param {Number} index
     * @param {Array} message
     * @param {Boolean} [isError] - является ли данное предупреждение ошибкой
     * @returns {BEM}
     * @private
     */
    _showHrefWarning: function(index, message, isError) {
        var elem = this.elem('error', 'index', '' + index);

        this.findElem(elem, 'href-error').text(message.join('\n'));
        this.setMod(elem, 'show', 'yes');

        isError && (this._hrefsValidations[index] = 0);

        return this._validateSitelinks();
    },

    /**
     * Скрыть предупреждение для ссылки
     * @param {Number} index
     * @returns {BEM}
     * @private
     */
    _hideHrefWarning: function(index) {
        var elem = this.elem('error', 'index', '' + index);

        this.findElem(elem, 'href-error').text('&nbsp;');
        this.delMod(elem, 'show');

        this._hrefsValidations[index] = 1;

        return this;
    },

    /**
     * Скопировать данные предыдущего баннера
     * @returns {BEM}
     * @private
     */
    _copy: function() {
        if (!this._prevBid) return this;

        var prevModel = BEM.MODEL.getOne({ name: 'm-banner-sitelinks', id: this._prevBid });

        if (!prevModel) return this;

        if (prevModel.isEmpty() && !confirm(iget2('b-sitelinks-selector', 'budut-skopirovany-pustye-dannye', 'Будут скопированы пустые данные'))) return this;

        this.model.update(prevModel.toJSON());
        this._checkAllHrefs();

        return this;
    },

    /**
     * Очистить сайтлинки
     * @returns {BEM}
     * @private
     */
    _clear: function() {
        this.model.clear();

        return this;
    },

    /**
     * Показывает описание к сайтлинкам если они есть
     * @private
     */
    _checkDescriptionVisibility: function() {
        var sitelinksWithDescription = false;

        this._forEachSitelink(function(i) {
            sitelinksWithDescription = sitelinksWithDescription || !!this.model.get('description' + i);
        });

        this._toggleDescriptions(sitelinksWithDescription);
    },

    /**
     * Скрывает/показывает описания к сайтлинкам
     * @param {Boolean} condition
     * @private
     */
    _toggleDescriptions: function(condition) {
        this.toggleMod('description', 'visible', condition);
    },

    /**
     * Вызывается в b-outboard-control перед показом
     * @param {Object} params
     */
    prepareToShow: function(params) {
        this._checkDescriptionVisibility();

        return this;
    },

    /**
     * Вызывается в b-outboard-control после нажатия accept-button
     */
    provideData: function() {
        //implemented in subclasses
    },

    /**
     * Вызывается в b-outboard-control после нажатия decline-button
     */
    declineChange: function() {
        //implemented in subclasses
    }

}, {
    live: true
});
