BEM.DOM.decl('b-stat-phrases-manager', {

    onSetMod: {

        js: function() {
            this.setMod('progress', 'yes');

            this.model = BEM.MODEL.create(this.params.modelName);

            this._initEvents();

            this.getNormalPhrases(this.params.phrases)
                .then(function(phrases) {
                    this.fillModel(phrases)
                        ._cacheSourcePhrases()
                        ._checkRemoveActions()
                        ._updateHeader()
                        ._setFocusOnFirstField();

                    this.delMod('progress');
                }, function(event, eventName) {
                    // Данная секция выполняется и при abort, по этому проверяем
                    if (eventName !== 'abort') {
                        this.trigger('prepare-stat-phrases-error');
                    }
                });
        },

        progress: {

            yes: function() {
                this.trigger('progress-start');

                if (this._getCommonPrice(true)) {
                    this._getCommonPrice().setMod('disabled', 'yes');
                }
            },

            '': function() {
                this.trigger('progress-end');

                if (this._getCommonPrice(true)) {
                    this._getCommonPrice().delMod('disabled');
                }
            }

        }

    },

    /**
     * Подготавливает фразы из статистики для их дальнейшего добавления в плюс-фразы групп
     * @returns {$.Deferred}
     */
    getNormalPhrases: function(phrases) {
        var deferred = $.Deferred();

        this._request().get(
            {
                cmd: 'ajaxPrepareStatPhrases',
                ulogin: u.consts('ulogin'),
                json_stat_phrases: JSON.stringify(phrases)
            },
            function(result) {
                result.error ?
                    deferred.rejectWith(this, [result.error]) :
                    deferred.resolveWith(this, arguments);
            },
            function(error) {
                deferred.rejectWith(this, arguments)
            });

        return deferred.promise();
    },

    /**
     * Заполняет данными модель(b-stat-phrases-manager)
     * @param {Array} phrases
     * @param {String} phrases.pid - id группы
     * @param {String|Number} phrases.price - цена на поиске
     * @param {String|Number} phrases.price_context - цена в сетях
     * @param {String} phrases.src_phrase - текст фразы-источника (ПЗ/ДРФ)
     * @param {String} phrases.norm_phrase - нормализированная форма фразы (ПЗ/ДРФ)
     * @param {String} phrases.norm_phrase_unquoted - нормализированная форма фразы где игнорируются кавычки (ПЗ/ДРФ) https://st.yandex-team.ru/DIRECT-63542#1488450622000
     * @param {'search_query'|'ext_phrase'} phrases.src_type - тип фразы-источника (ПЗ/ДРФ)
     * @returns {BEM}
     */
    fillModel: function(phrases) {
        var campaigns = this._groupByCampaigns(phrases);

        campaigns.forEach(function(campaign) {
            var campModel = this._getCampModel(campaign);

            campaign.phrases.forEach(function(phrase) {
                var phraseModel = campModel.get('list').add({
                    modelId: u._.uniqueId('phrase-bidable-'),
                    cid: campaign.cid,
                    price: phrase.price,
                    has_price: campaign.hasPrice,
                    has_price_context: campaign.hasPriceContext,
                    phrase: phrase.phrase,
                    price_context: phrase.price_context,
                    src_phrase: phrase.src_phrase,
                    adgroup_id: phrase.pid,
                    src_type: phrase.src_type,
                    bidForSearchLink: phrase.bid,
                    norm_phrase: phrase.norm_phrase,
                    norm_phrase_unquoted: phrase.norm_phrase_unquoted,
                    currency: this.params.currency,
                    report_row_hash: phrase.report_row_hash
                });

                this._initEventsPhrase(phraseModel);
            }, this);

            this.setMod(this._findFirstPhrase(campaign.cid), 'first', 'yes');
        }, this);

        return this;
    },

    /**
     * Отправляет запрос на добавление фраз
     * @param {Object} [requestParams]
     * @returns {$.Deferred}
     * @private
     */
    updateShowConditions: function(requestParams) {
        var deferred = $.Deferred();

        this._request().get(
            u._.extend({
                ulogin: u.consts('ulogin'),
                source: 'stat', // DIRECT-60768: Отдельный источник ставок в логах
                cmd: 'ajaxUpdateShowConditions',
                json_adgroup_retargetings: '{}',
                separate_group_oversized_error: 1,
                dont_need_qs: 1
            }, requestParams || {}),
            function(result) {
                result.error ?
                    deferred.rejectWith(this, [result.error]) :
                    deferred.resolveWith(this, arguments);
            },
            function(error) {
                deferred.rejectWith(this, arguments)
            });

        return deferred.promise();
    },

    /**
     * Сохраняет фразы и ставки, обрабатывает ответ
     * @param {Object} [requestParams] - дополнительные параметры к запросу
     * @returns {$.Deferred}
     */
    _save: function(requestParams) {
        var deferred = $.Deferred();

        this.setMod('progress', 'yes');

        requestParams = u._.extend({
            json_phrases: JSON.stringify(this._getDataForUpdateShowConditions())
        }, requestParams || {});

        this.updateShowConditions(requestParams)
            .then(function(data) {
                var sortGroups = this._sortGroups(data),
                    hasBad = !u._.isEmpty(sortGroups.bad),
                    hasOversize = !u._.isEmpty(sortGroups.oversize),
                    hasGroupsLimit = !u._.isEmpty(sortGroups.groupsLimit);

                // запоминаем добавленные фразы
                this._addedGroups = u._.extend(this._addedGroups || {}, sortGroups.good);

                if (hasBad || hasOversize || hasGroupsLimit) {
                    this._destructEmptyPhrases()
                        ._destructAddedGroups(sortGroups.good);

                    if (hasBad) {
                        this._showServerErrors(sortGroups.bad);
                    }
                }

                this.delMod('progress');

                deferred.resolve(sortGroups);
            }, function(event, eventName) {
                if (eventName !== 'abort') {
                    this.delMod('progress');
                }

                deferred.reject(eventName);
            });

        return deferred.promise();
    },

    /**
     * Возвращает первую ошибку, которая попадает в видимую область или false
     * @param {jQuery} errors
     * @param {jQuery} viewPort
     * @returns {jQuery|boolean}
     * @private
     */
    _getFirstErrorInViewPort: function(errors, viewPort) {
        var errorsLength = errors.length,
            i;

        for (i = 0; i < errorsLength; i++) {
            if (u.inviewport(errors[i], viewPort)) {
                return errors[i];
            }
        }

        return false;
    },

    /**
     * Отображает ошибки, при необходимости подкручивает полосу прокрутки к нужному месту
     * @private
     */
    _showErrors: function() {
        var viewPort = this.elem('campaigns-list'),
            errorNodes = this._getAllPhrasesModels()
                .filter(function(model) {
                    return !model.isValid();
                })
                .map(function(model) {
                    return this.findElem('phrase-item', 'model-id', model.get('modelId'));
                }, this),
            firstError = this._getFirstErrorInViewPort(errorNodes, viewPort),
            isPhraseError = false,
            needScroll = false,
            phraseNode;

        if (!firstError) {
            firstError = errorNodes[0];
            needScroll = true;
        }

        phraseNode = this.findElem(firstError, 'phrase');
        isPhraseError = !!this.findBlockInside(
            phraseNode,
            { block: 'input', modName: 'highlight-border', modVal: 'red' }
        );

        if (needScroll) {
            u.scrollNodeTo(
                firstError,
                viewPort,
                { over: 14 },
                function() {
                    isPhraseError && this._showHint(phraseNode);
                }.bind(this)
            );
        } else {
            isPhraseError && this._showHint(phraseNode);
        }
    },

    /**
     * Сохраняет фразы и ставки
     * @param {Object} [requestParams] - дополнительные параметры к запросу
     * @returns {$.Deferred}
     */
    save: function(requestParams) {
        var _this = this,
            deferred = $.Deferred(),
            confirm = BEM.blocks['b-confirm'];

        if (!this.isValid()) {
            this._showErrors();

            deferred.reject('validateErrors');

            return deferred.promise();
        }

        this.setMod('progress', 'yes');

        this.validateKeyWords()
            .then(function(result) {
                var errors = u._.compact(
                        result.map(function(item) {
                            return item.problem && item.problem.split('\n').join('<br/>');
                        })
                    ).join('<br/>'),
                    canForceSave;

                if (errors.length) {
                    canForceSave = result.every(function(item) {
                        return item.ok || !!+item.can_force_save;
                    });

                    if (canForceSave) {
                        confirm.open({
                            message: BEMHTML.apply({
                                block: 'b-stat-phrases-manager',
                                elem: 'confirm-text',
                                content: errors
                            }),
                            onYes: function() {
                                _this
                                    ._save(requestParams)
                                    .then(function(sortGroups) {
                                        deferred.resolve(sortGroups);
                                    }, function() {
                                        deferred.reject(arguments);
                                    });
                            },
                            onNo: function() {
                                _this.delMod('progress');

                                deferred.reject('validateErrors');
                            },
                            textYes: iget2('b-stat-phrases-manager', 'ok', 'ОК'),
                            textNo: iget2('b-stat-phrases-manager', 'otmena', 'Отмена')
                        });
                    } else {
                        _this.delMod('progress');

                        confirm.alert(errors);

                        deferred.reject('validateErrors');
                    }
                } else {
                    _this
                        ._save(requestParams)
                        .then(function(sortGroups) {
                            deferred.resolve(sortGroups);
                        }, function() {
                            deferred.reject(arguments);
                        });
                }
            }, function(eventName) {
                if (eventName !== 'abort') {
                    _this.delMod('progress');
                }

                deferred.reject(eventName);
            });

        return deferred.promise();
    },

    /**
     * Возвращает список фраз, добавленных в группы без изменений
     * @returns {Array}
     * @private
     */
    findSourcePhrases: function() {
        var list = this._sourcePhrases,
            addedGroups = this._addedGroups;

        this.model.fix();

        return u._.keys(addedGroups).reduce(function(sourcePhrases, adgroupId) {
            var group = addedGroups[adgroupId];

            u._.keys(group.phrases).forEach(function(phraseId) {
                var addedPhrase = group.phrases[phraseId],
                    sourcePhrase,
                    index;

                for (index = 0; index < list.length; index++) {
                    sourcePhrase = list[index];

                    if (sourcePhrase.adgroup_id === adgroupId &&
                        sourcePhrase.norm_phrase_unquoted === addedPhrase.norm_phrase_unquoted) {

                        sourcePhrases.push({
                            srcType: sourcePhrase.src_type,
                            srcPhrase: sourcePhrase.src_phrase,
                            normPhraseUnquoted: sourcePhrase.norm_phrase_unquoted,
                            adgroupId: sourcePhrase.adgroup_id
                        });

                        break;
                    }
                }
            });

            return sourcePhrases;
        }, []);
    },

    /**
     * Возвращает сгруппированные по компаниям номера групп
     * @param {Object} modifiedGroups
     * @returns {Object}
     * @private
     */
    sortGroupsByCampaigns: function(modifiedGroups) {
        var groups = this.params.groups,
            campaigns = {};

        u._.keys(modifiedGroups).forEach(function(groupId) {
            var modifiedGroup = modifiedGroups[groupId],
                copiedFrom = modifiedGroup.copied_from_pid,
                cid;

            if (copiedFrom) {
                cid = u._.get(groups[copiedFrom], 'cid');
            } else {
                cid = u._.get(groups[groupId], 'cid');
            }

            campaigns[cid] || (campaigns[cid] = { groups: [] });
            campaigns[cid].copiedFrom || (campaigns[cid].copiedFrom = copiedFrom);
            campaigns[cid].groups.push(groupId);
        });

        return campaigns;
    },

    /**
     * Проверяем ключевые фразы на пересечение с минус словами на объявление
     * @returns {this}
     */
    validateKeyWords: function() {
        var _this = this,
            validateDeferred = $.Deferred(),
            phrasesModels = this._getAllPhrasesModels(),
            phrasesByCamps,
            deferreds = [];

        phrasesByCamps = phrasesModels.reduce(function(result, model) {
            var cid = model.get('cid'),
                adgroupId = model.get('adgroup_id');

            result[cid] || (result[cid] = {});
            result[cid][adgroupId] || (result[cid][adgroupId] = []);

            result[cid][adgroupId].push(model.get('phrase'));

            return result;
        }, {});

        u._.forEach(phrasesByCamps, function(camp, cid) {
            var deferred = $.Deferred(),
                params = {
                    cid: cid,
                    cmd: 'ajaxCheckBannersMinusWords',
                    timeout: 110000,
                    ulogin: u.consts('ulogin'),
                    dont_cut_problems: '1'
                },
                adgroupIds = [],
                request = BEM.create('i-request_type_ajax', {
                    url: '/registered/main.pl',
                    cache: false,
                    dataType: 'json',
                    type: 'POST',
                    callbackCtx: this
                });

            u._.forEach(camp, function(phrases, adgroupId) {
                adgroupIds.push(adgroupId);
                params['json_key_words-' + adgroupId] = JSON.stringify(phrases);
            });

            params.adgroup_ids = adgroupIds.join(',');

            request.get(params, deferred.resolve, deferred.reject);
            deferreds.push(deferred.promise());
        });

        $.when.apply($, deferreds)
            .done(function() {
                if (_this.domElem) {
                    var data = arguments,
                        result = [];

                    if (deferreds.length === 1) {
                        result = [arguments[0]];
                    } else {
                        deferreds.forEach(function(item, index) {
                            result[index] = data[index][0];
                        });
                    }

                    validateDeferred.resolve(result);
                } else {
                    validateDeferred.reject('abort');
                }

            })
            .fail(function() {
                validateDeferred.reject();
            });

        return validateDeferred.promise();
    },

    /**
     * Возвращает количество добавляемых фраз
     * @returns {Number}
     */
    getCurrentPhrasesCount: function() {
        return this._getAllPhrasesModels().length;
    },

    isChanged: function() {
        return this.model.isChanged();
    },

    /**
     * Инстанс запроса
     * @returns {BEM}
     * @private
     */
    _request: function() {
        return this._requestInstance || (this._requestInstance = BEM.create('i-request_type_ajax', {
            url: '/registered/main.pl',
            type: 'POST',
            cache: false,
            dataType: 'json',
            timeout: 1200000,
            callbackCtx: this
        }));
    },

    /**
     * Кэширует исходные фразы
     * @returns {BEM}
     * @private
     */
    _cacheSourcePhrases: function() {
        this._sourcePhrases = this._getAllPhrasesModels().map(function(model) {
            return model.toJSON();
        });

        this.model.fix();

        return this;
    },

    /**
     * Возвращает блок composite отвечающий за список кампаний
     * @returns {BEM}
     * @private
     */
    _getCampaignsList: function() {
        return this._campaignList || (this._campaignList = this.findBlockInside('campaigns-list', 'composite'));
    },

    /**
     * Возвращает блок единой цены
     * @returns {BEM}
     * @private
     */
    _getCommonPrice: function(clearCache) {
        return !clearCache && this._commonPrice ||
            (this._commonPrice = this.findBlockInside('common-price', 'b-edit-phrase-price-stat-popup'));
    },

    /**
     * Строит шапку для списка фраз
     * @returns {BEM}
     * @private
     */
    _updateHeader: function() {
        var hasCommonPrice = false,         // На поиске
            hasCommonPriceContext = false,  // В сетях
            headerElem;

        this.model
            .get('campaigns')
            .forEach(function(campaign) {
                hasCommonPrice || (hasCommonPrice = campaign.get('hasPrice'));
                hasCommonPriceContext || (hasCommonPriceContext = campaign.get('hasPriceContext'));
            });

        headerElem = BEMHTML.apply({
            block: 'b-stat-phrases-manager',
            elem: 'header',
            statType: this.getMod('stat-type'),
            currency: this.params.currency,
            hasCommonPrice: hasCommonPrice,
            hasCommonPriceContext: hasCommonPriceContext
        });

        this.findElem('header').length ?
            BEM.DOM.replace(this.findElem('header'), headerElem) :
            BEM.DOM.prepend(this.domElem, headerElem);

        if (hasCommonPriceContext || hasCommonPrice) {
            // передаем модели фраз которые необходимо обновлять
            this._getCommonPrice(true)
                .initModels(this._getAllPhrasesModels());

            this.setMod('has-common-price', 'yes');
        } else {
            this.delMod('has-common-price');
        }

        return this;
    },

    /**
     * Ищет первый инпут с фразой и выставляет элементу фокус
     * @returns {BEM}
     * @private
     */
    _setFocusOnFirstField: function() {
        var firstPhraseInput = this.findBlockOn(this.findElem('phrase').eq(0), 'input');

        firstPhraseInput && firstPhraseInput.setMod('focused', 'yes');

        return this;
    },

    /**
     * Возвращает все модели фраз (m-stat-phrase-bidable)
     * @returns {Array}
     * @private
     */
    _getAllPhrasesModels: function() {
        return this.model.get('campaigns').reduce(function(result, campaign) {
            return result.concat(campaign.get('list').map(function(phrase) {
                return phrase;
            }));
        }, []);
    },

    /**
     * Подписка на события
     * @returns {BEM}
     * @private
     */
    _initEvents: function() {
        this._subscriptionManager = BEM.create('i-subscription-manager');

        this._subscriptionManager.wrap(this.findBlockOn('i-controls-overseer'))
            .on('remove', function(e, params) {
                this._removePhrase(params.data);
            }, this);

        this._subscriptionManager.wrap(this.model)
            .on('change', function() {
                this.trigger('change');
            }, this)
            .on('campaigns', 'remove', function(e, data) {
                this._removeCampaignItem(data.model);
                this._updateHeader();
            }, this)
            .on('campaigns', 'add', function(e, data) {
                this._addCampaignItem(data.model);

                this._subscriptionManager.wrap(data.model)
                    .on('list', 'add', function(e, data) { this._addPhraseItem(data.model); }, this)
                    .on('list', 'remove', function(e, data) { this._removePhraseItem(data.model); }, this);
            }, this);

        return this;
    },

    /**
     * Удаляет строку с фразой из DOM-дерева
     * @param {BEM.MODEL} model - модель фразы m-stat-phrase-bidable
     * @returns {BEM}
     * @private
     */
    _removePhraseItem: function(model) {
        var cid = model.get('cid'),
            phrasesList = this._getPhrasesList(cid),
            campModel = this._getCampModel(cid);

        phrasesList.remove(model.get('modelId'));

        // если в кампании больше нет фраз — удалям
        if (campModel.get('list').length() <= 0) {
            campModel.destruct();
        } else {
            this.setMod(this._findFirstPhrase(cid), 'first', 'yes');
        }

        this._checkRemoveActions();

        return this;
    },

    _findFirstPhrase: function(cid) {
        var listElem = this._getPhrasesList(cid).domElem;

        return this.findElem(listElem, 'phrase-item').eq(0);
    },

    _checkRemoveActions: function() {
        if (this._getAllPhrasesModels().length > 1) {
            this.setMod(this.elem('campaigns-list'), 'show-remove-button', 'yes');
        } else {
            this.delMod(this.elem('campaigns-list'), 'show-remove-button');
        }

        return this;
    },

    /**
     * Удаляет модель фразы
     * @param {Object} data
     * @param {String} data.cid - id кампании
     * @param {String} data.phraseId - id модели фразы
     * @returns {BEM}
     * @private
     */
    _removePhrase: function(data) {
        this._getCampModel(data.cid)
            .get('list')
            .remove(data.phraseId);

        if (this._getCommonPrice()) {
            this._getCommonPrice()
                .initModels(this._getAllPhrasesModels());
        }

        return this;
    },

    /**
     * Удаляет строку с кампанией из DOM-дерева
     * @param {BEM.MODEL} model - модель кампании (m-stat-phrases-campaign)
     * @returns {BEM}
     * @private
     */
    _removeCampaignItem: function(model) {
        this._getCampaignsList()
            .remove(model.get('modelId'));

        return this;
    },

    /**
     *
     * @typedef {Object} group
     * @property {String} groupId - id группы
     * @property {Array} phrases - массив фраз
     */

    /**
     *
     * @typedef {Object} campaign
     * @property {String} cid - id кампании
     * @property {String} campName - название кампании
     * @property {Boolean} hasPrice - флаг, ручное управление ставками на поиске
     * @property {Boolean} hasPriceContext - флаг, ручное управление ставками в сети
     * @property {Object.<String, group>}} groups - группы
     */

    /**
     * Раскладывает фразы пришедшие от сервера по кампаниям и группам
     * @param {Array} phrases
     * @returns {Object[]}
     * @private
     */
    _groupByCampaigns: function(phrases) {
        var groupsInfo = this.params.groups,
            campsInfo = this.params.campaigns,
            uniqCidsQueue = [],
            phrasesGroupedByCid = u._.groupBy(phrases, function(phrase) {
                var cid = groupsInfo[phrase.pid].cid;

                if (uniqCidsQueue.indexOf(cid) === -1) {
                    uniqCidsQueue.push(cid);
                }

                return cid;
            });

        return uniqCidsQueue.reduce(function(result, cid) {
            var phrasesSameCid = u._.sortBy(phrasesGroupedByCid[cid], 'pid'),
                // для определения возможности выставления цены на поиске/сетях достаточно посмотреть на любую фразу
                // для всех остальных фраз в рамках одной кампании будет аналогично
                firstPhrase = phrasesSameCid[0];

            return result.concat({
                cid: cid,
                phrases: phrasesSameCid,
                campName: campsInfo[cid].name,  // В группах может отсутствовать имя кампании DIRECT-62692
                // цена поиске
                hasPrice: !!firstPhrase.price || firstPhrase.price === 0,
                // цена в сетях
                hasPriceContext: !!firstPhrase.price_context || firstPhrase.price_context === 0
            });
        }, []);
    },

    /**
     * Подписка на события модели фразы
     * @param {BEM.MODEL} phraseModel
     * @private
     */
    _initEventsPhrase: function(phraseModel) {
        this._subscriptionManager
            .on(phraseModel, 'phrase', 'change', function() {
                this._validatePhrase(phraseModel);
            }, this);

        this._validatePhrase(phraseModel);
    },

    /**
     * Возвращает существующую или создает новую модель кампании
     * @param {String|Object} campaignParams - параметры модели
     * @param {String} campaignParams.cid
     * @param {String} campaignParams.campName
     * @param {Boolean} campaignParams.hasPrice
     * @param {Boolean} campaignParams.hasPriceContext
     * @returns {BEM.MODEL}
     * @private
     */
    _getCampModel: function(campaignParams) {
        if (typeof campaignParams === 'string') {
            campaignParams = { cid: campaignParams };
        }

        var campModel = this.model.get('campaigns').getById(campaignParams.cid);

        if (!campModel) {
            campModel = this.model.get('campaigns').add({
                cid: campaignParams.cid,
                modelId: campaignParams.cid,
                hasPrice: campaignParams.hasPrice,
                hasPriceContext: campaignParams.hasPriceContext,
                campName: campaignParams.campName
            });
        }

        return campModel;
    },

    /**
     * Возвращает блок composite отвечающий за список фраз в кампании
     * @param {String} id
     * @returns {BEM}
     * @private
     */
    _getPhrasesList: function(id) {
        this._phrasesLists || (this._phrasesLists = {});

        if (this._phrasesLists[id]) {
            return this._phrasesLists[id];
        } else {
            return this._phrasesLists[id] =
                this.findBlockInside(this.findElem('phrases-list', 'campaign-model-id', id), 'composite');
        }
    },

    /**
     * Добавляет строку с фразой в DOM-дерево
     * @param {BEM.MODEL} model - модель фразы (m-stat-phrase-bidable)
     * @returns {BEM}
     * @private
     */
    _addPhraseItem: function(model) {
        var phrasesList = this._getPhrasesList(model.get('cid')),
            groupInfo = this.params.groups[model.get('adgroup_id')] || {},
            campaign = this.params.campaigns[model.get('cid')],
            campModel = this._getCampModel(model.get('cid')),
            itemElem;

        phrasesList.add({
            id: model.get('modelId'),
            modelParams: {
                name: 'm-stat-phrase-bidable',
                id: model.get('modelId'),
                path: model.path()
            },
            cid: model.get('cid'),
            price: model.get('price'),
            price_context: model.get('price_context'),
            name: model.get('modelId'),
            phrase: model.get('phrase'),
            groupId: model.get('adgroup_id'),
            currency: this.params.currency,
            campaign: {
                strategy: campaign.strategy,
                type: campaign.type,
                campName: campModel.get('campName'),
                cid: campModel.get('cid')
            },
            statType: this.getMod('stat-type'),
            bidForSearchLink: model.get('bidForSearchLink'),
            groupInfo: groupInfo
        }, {
            itemOptions: {
                elemMods: { 'model-id': model.get('modelId') }
            }
        });

        if (model.get('has_price') || model.get('has_price_context')) {
            itemElem = this.elem('phrase-item', 'model-id', model.get('modelId'));

            if (model.get('has_price') && model.get('has_price_context')) {
                this.findBlockInside(itemElem, 'b-edit-phrase-price-stat-popup')
                    .initModels(model)
                    .validate();
            } else {
                this.findBlockInside(itemElem, 'b-edit-phrase-price')
                    .initModels([model])
                    .validate();
            }
        }

        return this;
    },

    /**
     * Добавляет строку кампании в DOM-дерево
     * @param {BEM.MODEL} model - модель кампании (m-stat-phrases-campaign)
     * @returns {BEM}
     * @private
     */
    _addCampaignItem: function(model) {
        var campaign = this.params.campaigns[model.get('cid')];

        this._getCampaignsList()
            .add({
                id: model.get('modelId'),
                strategy: campaign.strategy
            });

        return this;
    },

    hasUnfinishedActions: function() {
        return this._hasUnfinishedActions;
    },

    isValid: function() {
        return this.model.isValid();
    },

    /**
     * Проверяет наличие ошибок во фразе, подсвечивает фразу если они есть
     * @param {BEM.MODEL} model - модель фразы (m-stat-phrase-bidable)
     * @returns {BEM}
     * @private
     */
    _validatePhrase: function(model) {
        var itemElem = this.findElem('phrase-item', 'model-id', model.get('modelId')),
            phraseElem = this.findElem(itemElem, 'phrase'),
            phraseInput = this.findBlockOn(phraseElem, 'input'),
            validate = model.validate('phrase'),
            pricePopup;

        if (validate.errors) {
            phraseInput.setMod('highlight-border', 'red');

            // сбрасываем ставку изначально некорректного ПЗ/УП
            model.get('has_price') && model.set('price', 0);
            model.get('has_price_context') && model.set('price_context', 0);

            pricePopup = this.findBlockInside(itemElem, 'b-edit-phrase-price-stat-popup');

            if (pricePopup) {
                pricePopup.updateButtonText(model.get('price'), model.get('price_context'));
            }
        } else {
            phraseInput.delMod('highlight-border');
        }

        return this;
    },

    /**
     * Показывает попап с ошибками фразы
     * @param {jQuery} phraseElem
     * @private
     */
    _showHint: function(phraseElem) {
        if (!this._tipman) {
            this._tipman = BEM.create('tipman', {
                tipMods: { theme: 'normal' },
                popupDirections: ['bottom', 'top'],
                delay: 50
            });
        }

        this._tipman.show({
            owner: phraseElem,
            content: function() {
                var elemParams = this.elemParams(phraseElem),
                    model = this._getCampModel(elemParams.cid).get('list').getById(elemParams.modelId),
                    validate = model.validate('phrase');

                return BEMHTML.apply({
                    block: 'b-stat-phrases-manager',
                    elem: 'tooltip-errors',
                    content: validate.errors.map(function(error) {
                        return error.text;
                    }).join('\n')
                });
            }.bind(this)
        });
    },

    /**
     * Скрывает попап с ошибками фразы
     * @private
     */
    _hideHint: function() {
        this._tipman && this._tipman.hide();
    },

    /**
     * Возвращает данные для запроса добавления фраз
     * @returns {Object}
     * @private
     */
    _getDataForUpdateShowConditions: function() {
        return this._getAllPhrasesModels().reduce(function(result, model) {
            var data = model.toJSON(),
                phraseData = {};

            if (!data.phrase) {
                return result;
            }

            if (!result[data.adgroup_id]) {
                result[data.adgroup_id] = { added: [] };
            }

            phraseData.phrase = data.phrase;
            data.has_price && (phraseData.price = data.price);
            data.has_price_context && (phraseData.price_context = data.price_context);

            phraseData.autobudgetPriority = 3;
            phraseData.report_row_hash = data.report_row_hash;
            result[data.adgroup_id].added.push(phraseData);

            return result;
        }, {});
    },

    /**
     * Разбивает группы
     * @param {Object} groups
     * @returns {Object}
     * @private
     */
    _sortGroups: function(groups) {
        return u._.keys(groups).reduce(function(result, adgroupId) {
            var group = groups[adgroupId];

            if (group.groups_limit_exceeded) {
                result.groupsLimit[adgroupId] = group;

            } else if (!u._.isEmpty(group.errors_by_phrases) || !u._.isEmpty(group.errors)) {
                result.bad[adgroupId] = group;

            } else if (group.is_group_oversized) {
                result.oversize[adgroupId] = group;

            // phrases_not_added_camp_duplicates - Фразы, которые не были добавлены по причине того что они уже есть в кампании
            } else if (!u._.isEmpty(group.phrases) || !u._.isEmpty(group.phrases_not_added_camp_duplicates)) {
                result.good[adgroupId] = group;
            }

            return result;
        }, {
            bad: {},
            good: {},
            oversize: {},
            groupsLimit: {}
        });
    },

    /**
     * Записывает ошибки пришедшие с сервера в модель фразы
     * @param {Object} groups
     * @returns {BEM}
     * @private
     */
    _showServerErrors: function(groups) {
        var phrasesModels = this._getAllPhrasesModels(),
            phrasesCount = phrasesModels.length,
            index;

        u._.keys(groups).forEach(function(groupId) {
            var group = groups[groupId];

            group.errors_by_phrases.forEach(function(error) {
                var model;

                for (index = 0; index < phrasesCount; index++) {
                    model = phrasesModels[index];

                    if (model.get('phrase') === error.phrase && model.get('adgroup_id') === groupId) {
                        model.set('server_errors', error.errors.join(''));

                        this._validatePhrase(model);

                        // в errors_by_phrases могут приходить невалидные фразы не участвующие в добавлении
                        // все фразы которым не проставится currentPagePhrase = true,
                        // буду показаны в методе _getBadPhrasesText(b-stat-table-phrases-popup.js)
                        error.currentPagePhrase = true;

                        break;
                    }
                }
            }, this);
        }, this);

        return this;
    },

    /**
     * Удаляет добавленные фразы
     * @param {Object} groups
     * @returns {BEM}
     * @private
     */
    _destructAddedGroups: function(groups) {
        u._.keys(groups).forEach(function(adgroupId) {
            this._getAllPhrasesModels().forEach(function(model) {
                model.get('adgroup_id') === adgroupId && model.destruct();
            });
        }, this);

        return this;
    },

    /**
     * Удаляет пустые фразы
     * @returns {BEM}
     * @private
     */
    _destructEmptyPhrases: function() {
        this._getAllPhrasesModels().forEach(function(model) {
            !model.get('phrase') && model.destruct();
        });

        return this;
    },

    /**
     * Уничтожает блок, модели
     * @returns {BEM.DOM}
     */
    destruct: function() {
        this._request().abort();
        this._subscriptionManager.dispose();
        this._subscriptionManager.destruct();
        this.model.destruct();

        this._tipman && this._tipman.destruct();

        return this.__base.apply(this, arguments);
    }

}, {

    live: function() {

        this
            .liveBindTo('phrase', 'pointerover', function(e, data) {
                var input = this.findBlockInside(e.data.domElem, 'input');

                if (input.hasMod('highlight-border', 'red')) {
                    this._showHint(e.data.domElem);
                }
            })
            .liveBindTo('phrase', 'pointerout', function(e, data) {
                this._hideHint();
            })
            .liveInitOnBlockInsideEvent('show', 'b-edit-phrase-price-stat-popup', function() {
                this._hasUnfinishedActions = true;

                this.setMod('price-popup-shown', 'yes')
                    .trigger('unfinished-actions', true);
            })
            .liveInitOnBlockInsideEvent('hide', 'b-edit-phrase-price-stat-popup', function() {
                this._hasUnfinishedActions = false;

                this.delMod('price-popup-shown')
                    .trigger('unfinished-actions', false);
            });

        return false;
    }

});
