BEM.DOM.decl('composite', {
    /**
     * Добавляет элемент(ы) в список
     * @param {*} data - данные для элемента(ов) списка
     * @param {Object} settings - хэш с параметрами
     * @param {Number} settings.at - индекс вставки элемента/ов
     * @param {Number} settings.itemOptions - дополнительные данные для шаблонизации
     * @fires before:add - предварительное событие перед добавлением элемента/ов
     * @fires add - события после добавления элемента/ов
     */
    add: function(data, settings) {
        this._execute({
            event: 'add',
            context: {
                data: u.composite.toArray(data),
                settings: settings
            },
            action: this._add
        });
    },

    /**
     * Удаляет элемент(ы) в списке
     * @param {Number|String|Array} idData - идентификатор(ы) строчки/строчек
     * @fires before:remove - предварительное событие перед удалением элемента/ов
     * @fires remove - события после удаления элемента/ов
     */
    remove: function(idData) {
        this._execute({
            event: 'remove',
            context: { data: u.composite.toArray(idData) },
            action: this._remove
        });
    },

    /**
     * Обновляет элемент(ы) в списке
     * @param {Number|String|Object} id - id для элемента списка,
     * @param {*} data - данные для для элемента списка
     * @param {Object} settings - хэш с параметрами
     * @param {Number} settings.itemOptions - дополнительные данные для шаблонизации
     * @fires before:update - предварительное событие перед обновлением элемента/ов
     * @fires update - события после обновления элемента/ов
     */
    update: function(id, data, settings) {
        var isMulti = u._.isObject(id);

        this._execute({
            event: 'update',
            context: {
                data: !isMulti ?
                    [{ id: id, newData: data }] :
                    u._.map(id, function(data, id) {
                        return { id: id, newData: data };
                    }),
                settings: isMulti ? data : settings
            },
            action: this._update
        });
    },

    /**
     * Заменяет все элемент(ы) в списке
     * @param {*} data - данные для строчки/строчек списка
     * @param {Object} settings - хэш с параметрами
     * @param {Number} settings.at - индекс вставки элемента/ов
     * @param {Number} settings.itemOptions - дополнительные данные для шаблонизации
     * @fires before:reset - события перед заменой элемента/ов
     * @fires reset - события после замены элемента/ов
     */
    reset: function(data, settings) {
        this._execute({
            event: 'reset',
            context: {
                data: u.composite.toArray(data),
                settings: settings
            },
            action: this._reset
        });
    },

    /**
     * Очищает список
     * @fires before:clear - события перед очисткой списка
     * @fires clear - события после очистки списка
     */
    clear: function() {
        this._execute({
            event: 'clear',
            action: this._clear
        });
    },

    /**
     * Вспомогательный метод добавления строк
     * @param {Object} context - контекст
     * @param {*} context.data - данные строки
     * @param {Number} context.settings - дополнительные данные для шаблонизации
     * @private
     */
    _add: function(context) {
        var at = u._.get(context, 'settings.at'),
            data = this._templateData(context),
            wrap = this._getItemsWrapNode();

        if (at) {
            var indexNode = this._getAllItemsNodes().eq(at),
                isIndexNode = !!indexNode.length;

            BEM.DOM[isIndexNode ? 'before' : 'append'](isIndexNode ? indexNode : wrap, data);
        } else {
            BEM.DOM.append(wrap, data);
        }
    },

    /**
     * Вспомогательный метод для удаления строк
     * @param {Object} context - контекст
     * @param {*} context.data - данные строки
     * @private
     */
    _remove: function(context) {
        context.data.forEach(function(id) {
            var node = this._getItemNodeById(id);

            node && BEM.DOM.destruct(node);
        }, this);
    },

    /**
     * Вспомогательный метод для обновления строк
     * @param {Object} context - контекст
     * @param {*} context.data - данные строки
     * @param {Number} context.settings - дополнительные данные для шаблонизации
     * @private
     */
    _update: function(context) {
        context.data.forEach(function(itemData) {
            BEM.DOM.replace(this._getItemNodeById(itemData.id), this._templateData({
                data: itemData.newData,
                settings: context.settings
            }))
        }, this);
    },

    /**
     * Вспомогательный метод для замены всех строк
     * @param {Object} context - контекст
     * @param {*} context.data - данные строки
     * @param {Number} context.settings - дополнительные данные для шаблонизации
     * @private
     */
    _reset: function(context) {
        BEM.DOM.update(this._getItemsWrapNode(), this._templateData(context));
    },

    /**
     * Вспомогательный метод для очистки списка
     * @private
     */
    _clear: function() {
        this._getItemsWrapNode().empty();
    },

    /**
     * Хелпер для обработки действия над списком
     * @param {Object} hash
     * @param {Array} hash.context
     * @param {Object} hash.context.data
     * @param {Object} [hash.context.settings]
     * @param {Object} hash.action
     * @private
     */
    _execute: function(hash) {
        var eventHash = {
            event: hash.event,
            data: u._.get(hash, 'context.data', {})
        };

        this._emitEvent(eventHash, true);

        hash.action.call(this, u._.get(hash, 'context', {}));

        this._emitEvent(eventHash);
    },

    /**
     * Шаблонизирует элемент(ы)
     * @param {Object} context - обект параметров
     * @param {Array} context.data - массив данных
     * @param {Object} [context.settings] - объект дополнительных данных
     * @returns {String}
     * @private
     */
    _templateData: function(context) {
        var settings = u._.get(context, 'settings.itemOptions', {});

        return BEMHTML.apply(u.composite.toArray(context.data).map(function(item) {
            return this._getItemBemJson(item, settings);
        }, this));
    },

    /**
     * Триггерит событие действия над списком
     * @param {Object} hash - обект параметров
     * @param {Array} hash.data - массив данных
     * @param {Object} hash.event - имя события
     * @param {Boolean} [isBefore] - является ли событие предварительным
     * @returns {BEM}
     * @private
     */
    _emitEvent: function(hash, isBefore) {
        this.trigger(
            isBefore ? ('before:' + hash.event) : hash.event,
            !u._.isEmpty(hash.data) ? { items: hash.data } : null);

        return this;
    },

    /**
     * Формирует BEMJSON для элемента списка
     * @param {Object} data - массив данных для элементов списка
     * @param {Object} settings - объект с модификаторами
     * @returns {Object}
     * @private
     */
    _getItemBemJson: function(data, settings) {
        return u.composite.getItemBemJson(
            this.params.id,
            u.composite.modifyCtx(this.params.itemView, settings),
            this.params.idAttr,
            data);
    },

    /**
     * Возвращает ноду элемента по ID без кэширования результата
     * @param {String|Number} id - id строчки
     * @returns {Jquery}
     * @private
     */
    _getItemNodeById: function(id) {
        return this.findElem('item', 'item-id', u.composite.getItemModVal(this.params.id, id));
    },

    /**
     * Возвращает ноду контейнера (items) элементов списка
     * @returns {Jquery}
     * @private
     */
    _getItemsWrapNode: function() {
        return this.elem('items', 'pid', this.params.id);
    },

    /**
     * Возвращает ноды всех элементов списка без кэширования результата
     * @returns {Jquery}
     * @private
     */
    _getAllItemsNodes: function() {
        return this.findElem('item', 'pid', this.params.id);
    }
});
