u.register({
    'crypta-summary': {

        /**
         * Собирает данные для сводки
         * @param {PredictParams} data
         * @param {Object} segmentsHash
         * @param {Object} goalsHash
         * @returns {Array}
         */
        calculateSummary: function(data, segmentsHash, goalsHash) {
            var summary = [];

            u._.get(data, 'banners', []).length &&
                summary.push({ type: 'banner', sizes: data.banners });

            u._.get(data, 'geo.text', []).length &&
                summary.push({ type: 'geo', text: data.geo.text });

            u._.get(data, 'interests', []).forEach(function(group) {
                var formattedGroup = this._formatGroup(group, segmentsHash, goalsHash);

                if (formattedGroup) {
                    summary.push(formattedGroup);
                }
            }, this);

            (u._.get(data, 'platforms', []).length || u._.get(data, 'platforms', 0)) > 0 &&
                summary.push({ type: 'platforms', count: (data.platforms.length || data.platforms) });

            u._.get(data, 'formats', '').length &&
                summary.push({ type: 'formats', text: data.formats });

            u._.get(data, 'operators', '').length &&
                summary.push({ type: 'operators', text: data.operators });

            u._.get(data, 'regions', '').length &&
                summary.push({ type: 'regions', text: data.regions });

            u._.get(data, 'categories', '').length &&
                summary.push({ type: 'categories', text: data.categories });

            u._.get(data, 'zones', '').length &&
                summary.push({ type: 'zones', text: data.zones });

            return summary;
        },

        /**
         * Формирует сводку по группе интересов крипты или ретаргетинга
         * @param {RetConditionGroup} group
         * @param {Object} segmentsHash
         * @param {Object} goalsHash
         * @returns {type, [rule], parents, items}
         * @private
         */
        _formatGroup: function(group, segmentsHash, goalsHash) {
            var goal, goalWithSegmentInfo;

            group.goals = group.goals.filter(function(goal) {
                return !!goal.id;
            });

            goal = group.goals[0];

            goalWithSegmentInfo = group.goals.find(function(goal) {
                return !!segmentsHash[goal.id];
            });

            if (!this.METRIKA_GOAL_TYPES) {
                this.METRIKA_GOAL_TYPES = u['retargeting-goal-dto'].getMetrikaServerTypes();
            }

            if (this.METRIKA_GOAL_TYPES.indexOf(goal.type) !== -1) {
                return this._formatGoalsGroup(group, goalsHash);
            } else if (goalWithSegmentInfo) {
                return this._formatSegmentGroup(group, goalWithSegmentInfo.id, segmentsHash);
            }
        },

        /**
         * Формирует объект с данными группы целей для сводки
         * @param {RetConditionGroup} group
         * @param {Object} goalsHash
         * @returns {type, [rule], parents, items}
         * @private
         */
        _formatGoalsGroup: function(group, goalsHash) {
            var parents = [u['crypta'].segmentsNamesMapping['metrika']]; // в Метрике всегда один родитель

            if (!Object.keys(goalsHash).length) {
                return {
                    type: 'error',
                    parents: [],
                    items: [
                        iget2('crypta-summary', 'goal-not-found', 'Цели не найдены: {ids}', {
                            ids: group.goals.map(function(goal) { return goal.id }).join(',')
                        })
                    ]
                };
            }

            return {
                type: 'metrika',
                rule: u['retargeting'].ruleTypeNameMapping[group.type],
                parents: parents,
                items: group.goals.map(function(goal) {
                    var goalData = goalsHash[goal.id];

                    if (!goalData || !goalData.allow_to_use) {
                        return {
                            mods: { error: 'yes' },
                            text: u['retargeting'].getLostGoalName(goal)
                        }
                    }

                    return {
                        text: u['retargeting'].getGoalName(goalData), // TODO заюзать везде
                        time: goalData.type !== 'audience' && goal.time // для сегментов аудиторий время не выводим
                    }
                }, this)
            };
        },

        /**
         * Формирует объект с данными группы сегментов для сводки
         * @param {RetConditionGroup} group
         * @param {Number} goalId
         * @param {Object} segmentsHash
         * @returns {{type, [rule], parents, items}}
         * @private
         */
        _formatSegmentGroup: function(group, goalId, segmentsHash) {
            var parents;

            if (!Object.keys(segmentsHash).length || !segmentsHash[goalId]) {
                return {
                    type: 'error',
                    parents: [],
                    items: [
                        iget2('crypta-summary', 'segment-not-found', 'Сегменты не найдены: {ids}', {
                            ids: group.goals.map(function(goal) { return goal.id }).join(',')
                        })
                    ]
                }
            }

            if (segmentsHash[goalId].type === 'interests') {
                parents = this._getInterestParents(goalId, segmentsHash).slice(0, 1); // для интересов нужен только главный сегмент
            } else {
                parents = this._getInterestParents(goalId, segmentsHash).slice(0, -1); // вырезаем последний
            }

            return {
                type: segmentsHash[goalId].type,
                period: u['crypta'].periodMapping[group.interest_type],
                parents: parents,
                items: group.goals.filter(function(goal) {
                    return segmentsHash[goal.id];
                }).map(function(goal) {
                    return segmentsHash[goal.id].name;
                }, this)
            };
        },

        /**
         * Получение названий родительских сегментов
         * @param {Number} id
         * @param {Object} segmentsHash
         * @param {Array<String>} [result]
         * @returns {Array<String>}
         * @private
         */
        _getInterestParents: function(id, segmentsHash, result) {
            var segment = segmentsHash[id];

            result || (result = []);
            result.unshift(segment.name);

            if (segment.parent_id === 0) {
                result.unshift(u['crypta'].segmentsNamesMapping[segment.type]);
                return result;
            } else {
                return this._getInterestParents(segment.parent_id, segmentsHash, result);
            }
        }
    }
});
