(function() {
    BEM.MODEL.decl({ model: 'dm-base-group-with-feeds', baseModel: 'dm-base-group' }, {
        //идентификатор фида, если фид не задан, feed_id == 0
        feed_id: {
            type: 'number',
            validation: {
                rules: {
                    required: {
                        text: iget2('dm-base-group-with-feeds', 'vyberite-fid', 'Выберите фид'),
                        validate: function(val) {
                            return !!val;
                        }
                    }
                }
            }
        },

        // динамические фильтры
        feed_filters: {
            type: 'models-list',
            modelName: 'dm-feed-filter',
            validation: {
                rules: {
                    required: {
                        text: iget2('dm-base-group-with-feeds', 'dobavte-v-gruppu-filtry', 'Добавьте в группу фильтры'),
                        validate: function() {
                            return !!this.get('feed_filters').length()
                        }
                    },

                    invalidCatrgories: {
                        text: iget2('dm-base-group-with-feeds', 'nekorrektnye-usloviya-filtracii', 'Некорректные условия фильтрации'),
                        validate: function() {
                            return this.get('feed_filters').every(function(filter) {
                                return !filter.get('isIncorrect');
                            })
                        }
                    }
                }
            }
        }
    }, {
        /**
         * Создает модель фильтра
         */
        addFeedFilter: function() {
            var newId = u._.uniqueId('new-feed-filter_');

            this.get('feed_filters').add({
                filter_id: newId,
                adgroup_type: this.get('adgroup_type'),
                adgroupModelId: this.get('modelId')
            });

            return newId;
        },

        /**
         * Сохраняет фильтр. Если передан обработчик сохранения, то до обновления модели
         * управление будет передано функции сохранения
         * @param {Object} data данные, которые нужно поместить в фильтр
         * @param {Function} [savingHandler] обработчик сохранения, возвращающий Deferred ответ
         * @param {Object} [ctx] контекст обработчика сохранения
         * @returns {$.Deferred}
         */
        saveFeedFilter: function(data, savingHandler, ctx) {
            var deferred = $.Deferred();

            if (!this._areConditionsUnique(data)) {
                return deferred.reject([new Error(u.feedFilterData.errorTexts.notUnique)]).promise();
            } else {
                var filter = this.get('feed_filters').getById(data.filter_id),
                    tmpModel = BEM.MODEL.create({ name: filter.name, parentModel: this }, u._.omit(data, 'filter_id'));

                // задаём поле типа id после создания модели
                tmpModel.set('filter_id', data.filter_id);

                savingHandler ?
                    savingHandler.call(ctx, tmpModel)
                        .then(function(data) {
                            tmpModel.update(data);

                            deferred.resolve(tmpModel.toJSON());
                        })
                        .fail(function(errors) {
                            tmpModel.destruct();

                            deferred.reject(u._.isArray(errors) ? errors : [errors]);
                        }) :
                    deferred.resolve(tmpModel.toJSON());

                // если указана функции сохранения, то модель обновляется только при её успешной отработке,
                // иначе обновляется в любом случае
                return deferred.then(function(data) {
                    filter.updateFromData(data);
                    tmpModel.destruct();

                    return data;
                }).promise();
            }
        },

        /**
         * Удаляет модель фильтра по id
         * @param {String|Number} id
         */
        destroyFeedFilter: function(id) {
            var model = this.get('feed_filters').getById(id);

            model && model.destruct();
        },

        /**
         * Удаляет все модели фильтров
         */
        destroyAllFeedFilters: function() {
            this.toJSON()['feed_filters'].forEach(function(filter) {
                this.destroyFeedFilter(filter.filter_id);
            }, this);
        },

        /**
         * Возвращает флаг есть ли фильтры в группе
         */
        isAnyFeedFilter: function() {
            return !!this.get('feed_filters').length()
        },

        /**
         * Проверяет условия на уникальность
         * @param {Object} data
         * @returns {*}
         * @private
         */
        _areConditionsUnique: function(data) {
            return this.get('feed_filters').every(function(filter) {
                var goalState = filter.getConditionsState();

                return (data.filter_id !== filter.id) && (data.conditionsState.length == goalState.length) ?
                    !data.conditionsState.every(function(draftSt) {
                        return goalState.indexOf(draftSt) >= 0;
                    }) :
                    true;
            });
        },

        /**
         * Возвращает объект с данными фида
         * @param {Array} selectedItems - список выбранных категорий
         * @returns {$.Deferred<Object>}
         */
        fetchFeedData: function(selectedItems) {
            var deferred = $.Deferred(),
                feedDM = this.getFeedDM(),
                feedsParams = {
                    selectedItems: selectedItems,
                    adgroupIds: this.get('adgroup_id')
                };

            feedDM ?
                deferred.resolve(feedDM.toJSON()) :
                BEM.MODEL
                    .create({ name: 'dm-dynamic-media-feed', id: this.get('feed_id') })
                    .fetch(feedsParams)
                    .then(function(data) { deferred.resolve(data); })
                    .fail(function() { deferred.reject(); });

            return deferred.promise();
        },

        getFeedDM: function() {
            return BEM.MODEL.getOne({ name: 'dm-dynamic-media-feed', id: this.get('feed_id') }, false);
        },

        /**
         * Обновляет настройки фильтров согласно текущему набору категорий
         * @returns {$.Deferred<Object>}
         */
        rebaseFilters: function() {
            var deferred = $.Deferred();

            this.fetchFeedData().then(function(data) {
                var catIds = data.categories.map(function(cat) {
                    return cat.category_id;
                });

                this.get('feed_filters').forEach(function(filter) {
                    filter.rebaseConditions(catIds);
                });

                deferred.resolve();
            }.bind(this));

            return deferred.promise();
        },

        /**
         * Возвращает данные по счетчикам метрики из модели кампании
         * @returns {*}
         */
        getMetrikaCounters: function() {
            return this.getCampaignModel().get('metrika_counters');
        },

        /**
         * Возвращает список id категорий для фидов в группе
         * @returns {$.Deferred<String[]>}
         */
        getActiveCategoriesIds: function() {
            var deferred = $.Deferred();

            this.fetchFeedData().then(function(data) {
                var catIds = data.categories.filter(function(cat) {
                    return !cat.is_deleted;
                }).map(function(cat) {
                    return cat.category_id;
                });

                deferred.resolve(catIds);
            }.bind(this));

            return deferred.promise();
        },

        /**
         * Устанавливает диапазон цен для вкладки "Дерево категорий" для всех фидов группы
         * @param {String[]} price
         */
        setFeedsPrice: function(price) {
            this.get('feed_filters').forEach(function(feedFilterModel) {
                feedFilterModel.setPrice(price);
            });
        },

        /**
         * Удаляет диапазон цен для вкладки "Дерево категорий" для всех фидов группы
         */
        clearFeedsPrice: function() {
            this.get('feed_filters').forEach(function(feedFilterModel) {
                feedFilterModel.clearPrice();
            });
        },

        /**
         * Устанавливает галку "только в наличии" для таба "Дерево категории" для всех фидов группы
         * @param {Boolean} available
         */
        setFeedsAvailable: function(available) {
            this.get('feed_filters').forEach(function(feedFilterModel) {
                feedFilterModel.setAvailable(available);
            });
        },

        /**
         * Устанавливает галку "выбрать все" на вкладке "Дерево категорий" для всех фидов группы
         * @param {Boolean} allCategoriesSelected
         * returns {Promise}
         */
        setFeedsAllCategoriesIds: function(allCategoriesSelected) {
            return allCategoriesSelected ?
                this.getActiveCategoriesIds().then(function(categoriesIds) {
                    this.get('feed_filters').forEach(function(feedFilterModel) {
                        feedFilterModel.setCategoriesIds(categoriesIds);
                    });
                }.bind(this)) :
                $.Deferred().resolve();
        }
    });
})();
