/**
 * @typedef {Object} DynamicMediaGroupContextData
 * @property {{id: <String>, value: <String>}[]} campaignTags теги кампании
 * @property {Number} index номер группы в кампании
 * @property {String} [cid] идентификатор кампании группы
 * @property {Boolean} isSingleGroup группа - единственная редактируемая
 * @property {Boolean} isNewGroup группа создается
 * @property {Boolean} isCopyGroup группа копируется
 * @property {HierarchicalMultiplierMeta} [multipliers_meta] данные о корректировках ставок
 */
u.register({

    'dm-dynamic-media-group': {

        /**
         * Функция, формирующая данные для баннера ДМО
         * @param {Object} options
         * @param {Object} options.banner баннер; список полей см. в pick ниже
         * @param {Object} options.group группа
         * @param {{ cid: <String>, index: <Number>}} options.contextData данные, не относящиеся к баннеру
         * @returns {Object} см. модель dm-dynamic-media-banner
         */
        transformBannerData: function(options) {
            var banner = options.banner,
                result = u._.pick(banner, [
                    'bid',
                    'creative',
                    'enable',
                    'BannerID',
                    'adgroup_id',
                    'autobudget',
                    'archive',
                    'can_delete_banner',
                    'isNewBanner',
                    'showAdminAttrs',
                    'is_bs_rarely_loaded',
                    'status',
                    'statusAutobudgetShow',
                    'statusModerate',
                    'feed_id',
                    'statusShow',
                    'statusMetricaStop'
                ]);

            // DIRECT-39276, DIRECT-49050: statusModerate - поле группы, но оно нужно для блока b-banner-status,
            // подобная логике есть в предобработке данных для баннеров/групп i-banners-group-data.bemtree.xjst
            result.groupStatusModerate = options.group.statusModerate;

            // DIRECT-61150 если с сервера не приходит протокол, пытаемся взять из href
            options.banner.url_protocol || (options.banner.url_protocol = u.getUrlProtocol(options.banner.href));

            result.newBannerIndex = options.contextData.index + 1;
            result.modelId = +banner.bid ? banner.bid : 'new' + banner.newBannerIndex;
            result.cid = options.contextData.cid || 'new';

            return result;
        },

        /**
         * Функция, формирующая данные для группы ДМО
         * @param {Object} options данные о группе и другая информация
         * @param {Object} options.group группа; список полей см. в pick ниже
         * @param {Object[]} [options.campaignTags] тэги
         * @param {Number} [options.index] порядковый номер группы
         * @param {Boolean} [options.isSingleGroup]
         * @param {Boolean} [options.isNewGroup]
         * @param {Boolean} [options.isCopyGroup]
         * @param {Boolean} [options.cid]
         * @param {Boolean} [options.multipliers_meta]
         * @returns {Object} см. модель dm-dynamic-media-group
         */
        transformData: function(options) {
            // @deprecated для обратной совместимости оставлена старая логика с 2 параметрами
            if (arguments.length > 1) {
                options = u._.extend({ group: arguments[0] }, arguments[1]);
            }

            var group = options.group,
                campaignTags = options.campaignTags,
                data = u._.pick(group, [
                    'cid',
                    'adgroup_id',
                    'group_name',
                    'adgroup_type',
                    'is_completed_group',
                    'banners_quantity',
                    'banners_arch_quantity',
                    'group_banners_types',
                    'feed_id',
                    'geo',
                    'status_moderate',
                    'status_post_moderate',
                    'minus_words',
                    'banners',
                    'section',
                    'performance_filters',
                    'is_group_copy_action',
                    'used_creative_ids',
                    'tags',
                    'href_params'
                ]);

            if (!data.group_name) {
                data.group_name = iget2('dm-dynamic-media-group', 'novaya-gruppa-bannerov', 'Новая группа баннеров');
            }

            if (!data.minus_words) {
                data.minus_words = [];
            } else if (typeof data.minus_words == 'string') {
                // когда приходит в виде строки, то приводим к массиву
                // DIRECT-55545
                data.minus_words = u.minusWords.stringToArray(data.minus_words);
            } else if ( !u._.isArray(data.minus_words) ) {
                throw new Error('data.minus_words suppose to be Array or String or Falsy data');
            }

            if (u._.isArray(group.tags)) {
                group.tags = u._.groupBy(group.tags.map(function(tag) {
                    tag.tag_name && (tag.value = tag.tag_name);

                    return tag;
                }), 'id');
            }

            //по пришедшим id тегов строим полный массив данных
            data.tags = u.groupTags.makeTagsArray(group.tags, campaignTags) || [];

            //для работы с группами в которых всегда один баннер
            data.firstBid = data.firstBid || data.bid || data.banners[0] && +data.banners[0].bid || 'new';
            //adgroup_id приходит строкой
            data.adgroup_id = +data.adgroup_id;
            data.newGroupIndex = (options.index || 0) + 1;

            //для создания новой группы или копирования/мультикопирования групп
            data.modelId = ((data.adgroup_id == 0) || options.isCopyGroup) ?
                ('new' + data.newGroupIndex) :
                data.adgroup_id;

            data.isSingleGroup = options.isSingleGroup || false;
            data.isNewGroup = options.isNewGroup || false;
            data.isCopyGroup = options.isCopyGroup || false;

            data.banners = data.banners.map(function(banner, i) {
                return this.transformBannerData({
                    banner: banner,
                    group: group,
                    contextData: {
                        cid: data.cid || options.cid,
                        index: i
                    }
                });
            }, this);

            var bannerCreativeIds = data.banners.reduce(function(res, banner) {
                banner.creative && (res[banner.creative.creative_id] = true);
                return res;
            }, {});

            data.otherCreativeIds = (group.used_creative_ids || []).filter(function(creativeId) {
                return !bannerCreativeIds[creativeId];
            });

            data.isNotEmptyGroup = data.banners_quantity > 0;

            data.isEmptyGroup = !data.banners_quantity;

            options.multipliers_meta && u['i-adjustment-rates-data'].patchMultipliers(data, {
                multipliers_meta: options.multipliers_meta,
                isWithoutDeviceRestrictions: u.consts('isCpcDeviceModifiersEnabled')
            });

            data.cid = data.cid || options.cid;

            data.feed_filters = u['dm-feed-filter'].transformData({
                feedFilters: group.performance_filters || [],
                adgroupType: data.adgroup_type,
                adgroupId: data.modelId
            });

            data.feedFieldForName = {
                isChecked: Boolean(group.field_to_use_as_name),
                value: group.field_to_use_as_name || ''
            };

            data.feedFieldForBody = {
                isChecked: Boolean(group.field_to_use_as_body),
                value: group.field_to_use_as_body || ''
            };

            return data;
        }

    }

});
