/**
 * @event dropdown-chooser#change Изменение выбора
 * @type {Object} data
 * @param {string[]} data.value Итоговое значение
 * @param {string} data.itemId Идентификатор измененного элемента
 * @param {boolean} data.isSelected Значение измененного элемента
 *
 * @fires dropdown-chooser#change При изменении выбора
 */
BEM.DOM.decl('dropdown-chooser', {

    onSetMod: {

        js: function() {
            this.setValue(this.params.value);

            this.bindToDoc('keydown', this._onKeyDown);
        },

        ready: function(modName, modVal) {
            this.blockInside('control', 'button2').setMod('disabled', modVal ? '' : 'yes');
        }

    },

    /**
     * Устанавливает функцию по которой будет происходить получение данных
     * @param {function<Promise<{id:string, addition:[string], name:string, isDisabled:[boolean]}[]>>} provideData
     * @return {BEM.DOM}
     */
    setProvider: function(provideData) {
        this._provideData = provideData;

        return this.setMod('ready', provideData ? 'yes' : '');
    },

    /**
     * Устанавливает функциию получения контента для заглушки
     * @param {function<bemjson>} bypassFunc
     * @return {BEM.DOM}
     */
    setBypass: function(bypassFunc) {
        this._bypassFunc = bypassFunc;

        return this;
    },

    /**
     * Устанавливает значение
     * @param {string[]} value
     * @return {BEM.DOM}
     */
    setValue: function(value) {
        this._value = value;

        this._displayValue(value);

        return this;
    },

    /**
     * Отображает значение в кнопке
     * @param {string[]} value
     * @private
     */
    _displayValue: function(value) {
        this.blockInside('control', 'button2').setText(this._getDisplayedValue(value));
    },

    /**
     * Возвращает значение для отображения в кнопке
     * @param {string[]} value
     * @return {string}
     * @private
     */
    _getDisplayedValue: function(value) {
        return value.toString();
    },

    /**
     * Обработчик клика по кнопке
     * @param {jQuery.Event} e
     * @private
     */
    _onButtonClick: function(e) {
        this.open(e.block.domElem);
    },

    /**
     * Открывает попап со списком доступных пунктов
     * @param {jQuery} buttonDomElem Элемент на который показывает хвостик попапа
     */
    open: function(buttonDomElem) {
        buttonDomElem || (buttonDomElem = this.blockInside('control', 'button2').domElem);

        if (this._chooserPopup) {
            this._chooserPopup.close();
        } else {
            this._dropSubMan();

            Promise.all([
                this._createInitedDropdown()
                    .then(this._openDropdown.bind(this, buttonDomElem)),
                this._provideData()
            ])
                .then(function(value) {
                    var chooserPopup = value[0],
                        data = value[1],
                        items = data.items;

                    if (data.success) {
                        if (data.items.length) {
                            chooserPopup.drawItems(items, this._value);
                            chooserPopup.focusSearch();
                        } else {
                            chooserPopup.drawBypass(this._bypassFunc());
                        }
                    } else {
                        chooserPopup.drawBypass(this._bypassFunc(true));
                    }
                }.bind(this))
        }
    },

    /**
     * Показывает попап и подписывается на его события
     * @param {jQuery} owner
     * @param {BEM.DOM} chooserPopup
     * @return {BEM.DOM}
     * @private
     */
    _openDropdown: function(owner, chooserPopup) {
        this._subMan = BEM.create('i-subscription-manager');

        this._subMan.wrap(chooserPopup)
            .on('choose', this._onChooserPopupChoose, this)
            .on('open', this._onChooserPopupOpen, this)
            .on('close', this._onChooserPopupClose, this);

        return chooserPopup
            .open(owner)
            .onFirst('afterClose', this._onChooserPopupAfterClose, this);
    },

    /**
     * Генерирует и инициализируется `dropdown-chooser-popup`
     * @return {Promise<BEM.DOM>}
     * @private
     */
    _createInitedDropdown: function() {
        var chooserPopupMods = u._.pick(this.getMods(), ['multi']),
            notFoundText = this._getEmptySearchResult();

        return new Promise(function(resolve) {
            BEM.DOM.init(
                BEMHTML.apply({
                    block: 'dropdown-chooser-popup',
                    mods: chooserPopupMods,
                    notFoundText: notFoundText
                }),
                function(ctx) { resolve(ctx.bem('dropdown-chooser-popup')) }
            );
        });
    },

    /**
     * Возвращает текст неудачного поиска в попапе
     * @return {string}
     * @private
     */
    _getEmptySearchResult: function() {
        return ''
    },

    /**
     * Обработчик выбора в попапе
     * @param {jQuery.Event} e
     * @param {object} data
     * @param {string[]} data.value Итоговое значение
     * @param {string} data.itemId Идентификатор измененного элемента
     * @param {boolean} data.isSelected Значение измененного элемента
     * @private
     */
    _onChooserPopupChoose: function(e, data) {
        var chooserPopup = e.block,
            value = data.value,
            isMulti = this.hasMod('multi', 'yes');

        if (data.isSelected || isMulti) {
            this.setValue(value);

            this.trigger(
                'change',
                u._.pick(data, ['itemId', 'isSelected'])
            );

            if (!isMulti && value.length) {
                chooserPopup.close();
            }
        }
    },

    /**
     * Обработчик открытия попапа
     * @param {jQuery.Event} e
     * @private
     */
    _onChooserPopupOpen: function(e) {
        this._chooserPopup = e.block;
        this._isChooserPopupOpen = true;

        this._rotateArrow();
    },

    /**
     * Обоработчик закрытия попапа
     * @param {jQuery.Event} e
     * @private
     */
    _onChooserPopupClose: function(e) {
        this._dropSubMan();
        this._chooserPopup = null;
        this._isChooserPopupOpen = false;
        this._rotateArrow();
    },

    /**
     * Крутит стрелку вокгруг своей оси
     * @private
     */
    _rotateArrow: function() {
        this.blockInside('control', 'button2').blockInside('icon').toggleMod('direction', 'top', 'bottom');
    },

    /**
     * Обработчик события после полного закрытия попапа
     * @param {jQuery.Event} e
     * @private
     */
    _onChooserPopupAfterClose: function(e) {
        BEM.DOM.destruct(e.block.domElem);
    },

    /**
     * Сбрасывает все подписки
     * @private
     */
    _dropSubMan: function() {
        if (this._subMan) {
            this._subMan.dispose();
            this._subMan.destruct();
            this._subMan = null;
        }
    },

    destruct: function() {
        this._dropSubMan();

        this.__base.apply(this, arguments);
    },

    /**
     * Флаг о наличии реакции на ESC
     * @returns {boolean}
     */
    haveReactionToEsc: function() {
        return this._isChooserPopupOpen;
    },

    /**
     * Обработчик событие `keydown`
     * @param {jQuery.Event} e
     * @private
     */
    _onKeyDown: function(e) {
        if (BEM.blocks.keycodes.is(e.keyCode, 'ESC') && this.haveReactionToEsc()) {
            this._chooserPopup.close();
        }
    }

}, {

    live: function() {
        this
            .liveInitOnBlockInsideEvent('click', 'button2', function(e) {
                this._onButtonClick(e);
            });

        return false;
    }

});
