(function() {
    var MAX_CONDITIONS_LENGTH = 10;

    BEM.DOM.decl({ block: 'filter-conditions-list', baseBlock: 'i-filter-conditions-group' }, {
        onSetMod: {
            js: function() {
                this._availableStatesHash = BEM.create('i-smart-filters-manager', {
                    statesMap: u['i-filter-edit'].getStatesMap(),
                    filterName: 'field',
                    subFilterName: 'relation'
                });

                u.graspSelf.call(this, {
                    _addConditionBtn: 'b-control-add-button on condition-control',
                    _list: 'composite on conditions'
                });

                this.__base.apply(this, arguments);

                //ошибки полученные на валидации
                this._errors = {};

                //рисуем условия пришедшие с сервера
                this._filterModel.get('condition').length() &&
                    this._filterModel.get('condition').forEach(function(model) {
                        this._addCondition(this._buildHTMLParams(model.toJSON()));
                    }, this);
            }
        },

        /**
         * Возвращает список условий фильтра в виде массива
         * @returns {Array}
         * @private
         */
        _getConditionsList: function() {
            return u._.map(this._getConditionsHash(), function(cond) { return cond; })
        },

        /**
         * Возвращает данные для условия фильтра в том виде, который будет отправлен в bemhtml
         * @param {Object} values
         * @param {String} values.id
         * @param {Array} values.value
         * @param {String} values.field
         * @param {String} values.relation
         * @returns {{values: *, relationsOptionsList: (*|Object), conditionsOptionsList: (*|Object)}}
         * @private
         */
        _buildHTMLParams: function(values) {
            var modelParams = this.params.modelParams;

            return {
                //id необходимо для правильной отрисовки composite
                //чтобы id в composite совпадало с id в модели
                id: values.id,
                modelParams: {
                    parentName: modelParams.name,
                    parentId: modelParams.id,
                    id: values.id,
                    name: 'filter-condition-edit'
                },
                errors: this._errors[values.id],
                values: values,
                relationsOptionsList: this.getRelationsOptions(values),
                conditionsOptionsList: this.getFieldsOptions(values)
            }
        },

        /**
         * Возвращает опции для селекта выбора типа фильтра при условии текущего значения фильтра currentValues и состояния
         * доступных фильтров availableFiltersHash
         * @param {Object} currentValues текущее значения фильтра
         * @param {String} currentValues.field выбранный тип фильтра
         * @param {String} currentValues.relation выбранное отношение для типа фильтра
         * @returns {Object}
         */
        getFieldsOptions: function(currentValues) {
            var currentField = currentValues.field,
                orderedFieldsList = this._availableStatesHash.getOrderedFiltersList(),
                seperatorNum = u['i-filter-edit'].getFieldsListSeparator();

            return orderedFieldsList.map(function(field, i) {
                var selected = field == currentField,
                    isEnabled = this._availableStatesHash.isFilterEnabled(field);

                return {
                    value: field,
                    hasSeparator: seperatorNum === i + 1,
                    selected: selected,
                    disabled: !selected && !isEnabled
                };
            }, this);
        },

        /**
         * Возвращает опции для селекта выбора отношения при условии текущего значения фильтра currentValues и
         * состояния всех фильтров currentState
         * @param {Object} currentValues текущее значения фильтра
         * @param {String} currentValues.field выбранный тип фильтра
         * @param {String} currentValues.relation выбранное отношение для типа фильтра
         * @returns {Object}
         */
        getRelationsOptions: function(currentValues) {
            var currentField = currentValues.field,
                currentRelation = currentValues.relation,
                relationsList = this._availableStatesHash.getOrderedSubFiltersList(currentField);

            return relationsList.filter(function(relation) {
                //временное решение по DIRECT-54644 - пока существуют в схеме параллельно like и ilike
                //те значения, для которых нет translation мы не показываем
                return u['i-filter-edit'].getRelationTitle(relation) !== null;
            }).map(function(relation, i) {
                var selected = relation == currentRelation,
                    isEnabled = this._availableStatesHash.isSubFilterEnabled(currentField, relation);

                return {
                    value: relation,
                    selected: selected,
                    disabled: !selected && !isEnabled,
                    text: u['i-filter-edit'].getRelationTitle(relation)
                };
            }, this);
        },

        /**
         * Возвращает хэш с данными по условиям из списка
         * @returns { { id: 'String', field: 'String', relation: 'String', value: * } }
         * @private
         */
        _getConditionsHash: function() {
            var hash = {};

            this._filterModel.get('condition').forEach(function(model) {
                hash[model.get('id')] = model.toJSON();
            });

            return hash;
        },

        /**
        * Валидирует список условий фильтра
        * @returns {Object}
        */
        validate: function() {
            var validateResult = { isValid: true },
                currentConditionValidateResult,
                totalValidateResult;

            //валидируем весь список условий
            totalValidateResult = u['i-filter-edit'].validateConditionsList(this._getConditionsList());

            validateResult.isValid = totalValidateResult.isValid;

            if (totalValidateResult.errors && totalValidateResult.errors.length) {
                validateResult.errors = validateResult.errors || {};
                validateResult.errors.total = totalValidateResult.errors
            }

            //валидируем каждое условие по отдельности
            u._.forEach(this._getConditionsHash(), function(condition, condId) {
                currentConditionValidateResult = u['i-filter-edit'].validateCondition(condition);
                validateResult.isValid = validateResult.isValid && currentConditionValidateResult.isValid;

                if (!currentConditionValidateResult.isValid) {
                    validateResult.errors = validateResult.errors || {};

                    validateResult.errors[condId] = currentConditionValidateResult.errors
                }
            }, this);

            this._errors = validateResult.errors || {};

            return validateResult;
        },

        /**
         * Добавляет строку в список условий
         * @param {Array} data
         * @private
         */
        _addCondition: function(data) {
            this._list.add(data, {
                itemOptions: {}
            });
        },

        _getNewFilterData: function() {
            var data = this._availableStatesHash.getNewFilterData();

            data.id = u._.uniqueId('new_condition_id_');

            return data;
        },

        /**
         * Выполняет подписку на события через i-subscription-manager
         * @private
         */
        _initCommonEvents: function() {

            this._subscriptionManager.wrap(this.findBlockOn('i-controls-overseer'))
                .on('add', function() {
                    this._addCondition([this._buildHTMLParams(this._getNewFilterData())]);
                }, this)
                .on('remove', function(e, params) {
                    this._list.remove(params.data.itemId);
                }, this);

            this._subscriptionManager.wrap(BEM.blocks['filter-condition-edit'])
                .on('change', this._onConditionChange, this);

            this._subscriptionManager.wrap(this._list)
                .on('add', this._onNewConditionAdded, this)
                .on('remove', this._onConditionRemoved, this)
        },

        /**
         * Обновляет состояние кнопки "добавить новое условие" в зависимости от того, есть ли у нас свободные пары
         * тип условия/отношение для данного типа
         * @returns {BEM}
         * @private
         */
        _updateAddBtnState: function() {
            var currentState = this._getConditionsHash(),
                canAdd = (u._.size(currentState) < MAX_CONDITIONS_LENGTH) &&
                    !u._.isEmpty(this._availableStatesHash.getNewFilterData());

            this._addConditionBtn.setMod('state', canAdd ? '' : 'disabled');

            return this;
        },

        /**
         * Обработчик события "условие в списке изменилось"
         * @param {Object} e
         * @param {Object} data
         * @param {String} data.field поле, change на котором повлек событие
         * @param {Object} data.value текущее значение изменившегося filter-condition-edit
         * @param {String} data.value.id
         * @returns {BEM}
         * @private
         */
        _onConditionChange: function(e, data) {
            var controlValue = data.value,
                itemId = controlValue.id,
                prevControlValue = data.prevValue,
                needToUpdate = true;

            //удаляем ошибку - она стала неактуальна
            delete this._errors[itemId];

            switch (data.field) {
                case 'field':
                    //если сменился field то relation может не остаться тот же только если он доступен
                    //иначе он сменится на первый достпный
                    controlValue = this._availableStatesHash.getAdjustedNewFilterData({
                        filter: controlValue.field,
                        subFilter: controlValue.relation
                    });
                    controlValue.value = u['i-filter-edit'].getDefaultValue(controlValue);

                    this._availableStatesHash.updateAvailableHash(
                        'change-filter',
                        {
                            filter: controlValue.field,
                            subFilter: controlValue.relation,
                            prevFilter: prevControlValue.field,
                            prevSubFilter: prevControlValue.relation
                        });

                    break;
                case 'relation':
                    controlValue.value = u['i-filter-edit'].isValueNotNeededToChange(controlValue, prevControlValue) ?
                        controlValue.value :
                        u['i-filter-edit'].getDefaultValue(controlValue);

                    this._availableStatesHash.updateAvailableHash(
                        'change-sub-filter',
                        {
                            filter: controlValue.field,
                            subFilter: controlValue.relation,
                            prevFilter: prevControlValue.field,
                            prevSubFilter: prevControlValue.relation
                        });
                    break;
                case 'value':
                    needToUpdate = false;
                    break;
            }

            this._filterModel.get('condition').getById(itemId).update(controlValue);
            //без afterCurrentEvent блок перерисуется и change уже не будет считаться событием на вложенном блоке =>
            //попап закроется
            needToUpdate && this.afterCurrentEvent(function() {
                this._updateConditionsList();
            }, this);

            return this;
        },

        /**
         * Обработчик события "новое условие добавлено в список"
         * @param {Object} e
         * @param {Object} addedItemsData
         * @param {Array<Object>} addedItemsData.items
         * @returns {BEM}
         * @private
         */
        _onNewConditionAdded: function(e, addedItemsData) {
            var excludedIds = [];

            addedItemsData.items.forEach(function(itemData) {
                if (!this._filterModel.get('condition').getById(itemData.values.id)) {
                    this._filterModel.get('condition').add(itemData.values)
                }

                this._availableStatesHash.updateAvailableHash(
                    'add',
                    {
                        filter: itemData.values.field,
                        subFilter: itemData.values.relation
                    });

                excludedIds.push(itemData.values.id);
            }, this);

            this
                ._updateConditionsList(excludedIds)
                ._updateAddBtnState();

            u.scrollNodeTo(this.findElem('composite-item', 'id', addedItemsData.items[0].id), this._list.elem('items'));

            return this;
        },

        /**
         * Обработчик события "условие удалено из списка"
         * @param {Object} e
         * @param {Object} removedItemsData
         * @param {Array<String>} removedItemsData.items
         * @returns {BEM}
         * @private
         */
        _onConditionRemoved: function(e, removedItemsData) {
            removedItemsData.items.forEach(function(itemId) {
                var data = this._getConditionData(itemId);

                this._availableStatesHash.updateAvailableHash(
                    'remove',
                    {
                        filter: data.field,
                        subFilter: data.relation
                    });
                this._filterModel.get('condition').remove(itemId);
                delete this._errors[itemId];
            }, this);

            this
                ._updateConditionsList()
                ._updateAddBtnState();

            return this;
        },

        _getConditionData: function(id) {
            return this._filterModel.get('condition').getById(id).toJSON();
        },

        /**
         * Перерисовывает дочерние блоки
         * @param {Array} [excludeItemIds] id элемента, который не надо обновлять
         * @returns {BEM}
         * @private
         */
        _updateConditionsList: function(excludeItemIds) {
            var toUpdate = {},
                currentState = this._getConditionsHash();

            this.setMod('empty', u._.isEmpty(currentState) ? 'yes' : '');

            excludeItemIds = excludeItemIds || [];

            Object.keys(currentState).forEach(function(itemId) {
                if ((excludeItemIds.indexOf(itemId) == -1)) {
                    toUpdate[itemId] = this._buildHTMLParams(this._getConditionData(itemId));
                }
            }, this);

            this._list.update(toUpdate);

            return this;
        }
    });

})();
