describe('filter-conditions-tree', function() {
    var model,
        parentModel,
        block,
        sandbox;

    var testData = [
        {
            categoryId: [{ field: 'categoryId', relation: '==', value: [], id: 'condition_id_4' }],
            price:[{ field: 'price', relation: '<->', value: [{ min: '', max: '' }], id: 'condition_id_5' }],
            vendor:[{ field: 'vendor', relation: 'ilike', value: [], id: 'condition_id_6' }]
        },
        {
            categoryId: [{ field: 'categoryId', relation: '==', value: [2047], id: 'condition_id_4' }],
            price:[{ field: 'price', relation: '<->', value: [{ min: '50.00', max: '70.00' }], id: 'condition_id_5' }],
            vendor:[{ field: 'vendor', relation: 'ilike', value: ['blabla'], id: 'condition_id_6' }]
        }
    ];

    function createBlock(grouped) {
        var modelParams = { name: 'b-feed-filter-edit', id: 1 };

        parentModel = BEM.MODEL.create({ name: 'dm-dynamic-media-group', id: 1 }, {
            adgroup_type: 'performance',
            cid: 123
        });

        model = BEM.MODEL.create(modelParams, {
            adgroupModelId: 1,
            adgroup_type: 'performance',
            conditionTree: [
                grouped.categoryId[0],
                grouped.price[0],
                grouped.vendor[0]
            ]
        });

        block = u.createBlock({
            block: 'filter-conditions-tree',
            modelParams: modelParams,
            categories: [
                { category_id: '2047', parent_category_id: 0, is_deleted: false, name: 'Ремонт и новоселье', path: [] },
                { category_id: '2048', parent_category_id: 0, is_deleted: false, name: 'Ремонт и новоселье1', path: [] },
                { category_id: '2049', parent_category_id: 0, is_deleted: false, name: 'Ремонт и новоселье2', path: [] }
            ],
            grouped: grouped
        }, { inject: true });
    }


    beforeEach(function() {
        sandbox = sinon.sandbox.create({
            useFakeTimers: true
        });

        sandbox.stub(u['i-filter-edit'], 'getConditionMaxCount').callsFake(function() {
            return 10;
        });

        sandbox.stub(u['i-filter-edit'], 'isRangeValue').callsFake(function(cond) {
            switch (cond.field) {
                case 'price':
                    return true;
                default:
                    return false;
            }
        });

        sandbox.stub(u['i-filter-edit'], 'getValuePreproccessRules').callsFake(function(cond) {
            if (cond.field === 'price') {
                return {
                    type: 'range',
                    precision: 2
                }
            } else {
                return {
                    type: 'other'
                }
            }
        });

        sandbox.stub(u['i-filter-edit'], 'getDefaultValue').callsFake(function(cond) {
            switch (cond.field) {
                case 'category_id':
                    return [];
                case 'vendor':
                    return [];
                case 'price':
                    return [{ min: '', max: '' }];
            }
        })
    });

    afterEach(function() {
        sandbox.restore();
        model.destruct();
        parentModel.destruct();
        block.destruct();
    });

    it('В блок для ввода производителей должен быть передан параметр skipAdaptTextareaHeight = true', function() {
        createBlock(testData[0]);

        expect(block.findBlockOn('vendor-input', 'filter-condition-edit-control').params.skipAdaptTextareaHeight).to.be.true;
    });

    it('В блок для ввода производителей должен быть передан параметр skipCounter = true', function() {
        createBlock(testData[0]);

        expect(block.findBlockOn('vendor-input', 'filter-condition-edit-control').params.skipCounter).to.be.true;
    });

    it('При создании блок должен рисоваться с пустыми контроллами если переданы пустые данные', function() {
        createBlock(testData[0]);
        expect(block.findBlockInside('b-feed-category-chooser').getValue()).to.be.eql(testData[0].categoryId[0].value);
        expect(block.findBlockOn('price-range', 'filter-condition-edit-control').getData()).to.be.eql(testData[0].price[0].value);
        expect(block.findBlockOn('vendor-input', 'filter-condition-edit-control').getData()).to.be.eql(testData[0].vendor[0].value);
    });

    it('При создании блок должен рисоваться с заполненными контроллами если переданы не пустые данные', function() {
        createBlock(testData[1]);

        expect(block.findBlockInside('b-feed-category-chooser').getValue()).to.be.eql(testData[1].categoryId[0].value);
        expect(block.findBlockOn('price-range', 'filter-condition-edit-control').getData()).to.be.eql(testData[1].price[0].value);
        expect(block.findBlockOn('vendor-input', 'filter-condition-edit-control').getData()).to.be.eql(testData[1].vendor[0].value);
    });

    it('При изменении контролла категорий должна измениться модель', function() {
        createBlock(testData[0]);

        block.findBlockInside('b-chooser').check('2049');
        sandbox.clock.tick(500);

        expect(model.get('conditionTree').getById('condition_id_4').get('value').slice()).to.be.eql([2049]);
    });

    it('При изменении цены должна измениться модель', function() {
        createBlock(testData[0]);

        block.findBlockInside('range-input').val({ min: '50.90', max: '62.43' });
        sandbox.clock.tick(5000);
        expect(model.get('conditionTree').getById('condition_id_5').get('value').slice()).to.be.eql([{ min: '50.90', max: '62.43' }]);
    });

    it('При изменении производителей должна измениться модель', function() {
        createBlock(testData[0]);

        block.findBlockInside('vendor-input', 'input').val('bla-bla');
        sandbox.clock.tick(5000);
        expect(model.get('conditionTree').getById('condition_id_6').get('value').slice()).to.be.eql(['bla-bla']);
    });
});
