describe('i-adapter-time-targeting', function() {
    var utils = u['i-adapter-time-targeting'],
        timeZones = [
            {
                "timezone": "Europe/Kaliningrad",
                "name": "Калининград (MSK -01:00)",
                "id": "131",
                "msk_offset": "-01:00",
                "gmt_offset": "+02:00",
                "group_nick": "russia",
                "offset": 7200,
                "country_id": "225",
                "timezone_id": "131",
                "offset_str": "(MSK -01:00)"
            }, {
                "gmt_offset": "+03:00",
                "group_nick": "russia",
                "msk_offset": "+00:00",
                "name": "Москва",
                "timezone": "Europe/Moscow",
                "id": "130",
                "country_id": "225",
                "offset_str": "",
                "timezone_id": "130",
                "offset": 10800
            }
        ],
        DEFAULT_TIMEZONE_ID = 130,
        DEFAULT_TIMEZONE_TEXT = iget2('i-adapter-time-targeting', 'moskva', 'Москва'),

        DEFAULT_HOLIDAY_SHOWING_FROM = 8,
        DEFAULT_HOLIDAY_SHOWING_TO = 20,
        DEFAULT_HOLIDAY_SHOWING_COEFFICIENT = 100;

    describe('Дефолтные значения', function() {
        [
            {
                preset: 'worktime',
                name: 'По рабочим дням'
            },
            {
                preset: 'other',
                name: 'Другое'
            },
            {
                preset: 'all',
                name: 'Круглосуточно'
            }
        ].forEach(function(preset) {
            describe(' для пресета "' + preset.name + '"', function() {
                var defaults = utils.getDefault({
                    "timezone_groups": timeZones
                }, preset.preset);

                it('Расширенный режим запрещен', function() {
                    expect(defaults.isExtendModeAvailable).to.be.false;
                });

                it('Дефолтный часовой пояс - ' + DEFAULT_TIMEZONE_TEXT, function() {
                    expect(defaults.value.timeZone.id).to.equal(DEFAULT_TIMEZONE_ID);
                });

                it('Дефолтное время - ' + preset.name, function() {
                    expect(defaults.value.timeTargetCode).to.equal(u['b-time-targeting'][preset.preset === 'all' ? 'CODE_ALL_TIME' : 'CODE_WORK_TIME']);
                });

                it('Настройки праздников выключены', function() {
                    expect(defaults.value.isHolidaySettingsEnabled).to.be.false;
                });

                it('Настройки рабочих дней ' + preset.preset === 'all' ? 'выключены' : 'включены', function() {
                    expect(defaults.value.isWorkingWeekendEnabled).to.equal(preset.preset !== 'all')
                });

                it('Пресет - "' + preset.name + '"', function() {
                    expect(defaults.value.preset).to.equal(preset.preset)
                })
            });
        });
    });

    describe('Обработка данных с сервера', function() {
        var defaultValues = {
            timetargeting: {
                time_target_preset: 'worktime',
                timeTarget: '1ABCDEFGHIJKLMNOPQRSTUVWX2ABCDEFGHIJKLMNOPQRSTUVWX3ABCDEFGHIJKLMNOPQRSTUVWX4ABCDEFGHIJKLMNOPQRSTUVWX5ABCDEFGHIJKLMNOPQRSTUVWX6ABCDEFGHIJKLMNOPQRSTUVWX7ABCDEFGHIJKLMNOPQRSTUVWX',
                time_target_working_holiday: 1,
                time_target_holiday: 0,
                timezone_id: '130',
                time_target_holiday_dont_show: 0,
                time_target_holiday_from: undefined,
                time_target_holiday_to: undefined,
                time_target_holiday_coef: undefined,
                timeTargetMode: 'simple',
                timezone_text: 'Москва'
            },
            strategy: {
                is_autobudget: '',
                search: { name: "default" },
                is_search_stop: '',
                net: { name: "stop" },
                is_net_stop: "1",
                name: ""
            }
        };

        function runCheck(data, toCheck, checkValue) {
            var serverData,
                cache = JSON.stringify(data);

            beforeEach(function() {
                serverData = utils.fromServer({
                    timezone_groups: timeZones,
                    campaign: JSON.parse(cache)
                });
            });

            switch(toCheck) {
                case 'strategy':
                    it('Стратегия: ' + checkValue.name +'. Расширенный режим ' + (checkValue.value ? 'запрещен' : 'разрешен'), function() {
                        expect(serverData.isExtendModeAvailable).to.equal(checkValue.result);
                    });
                    break;
                case 'coefficient':
                    it('Коэффициент показа в праздники ' + checkValue.name, function() {
                        expect(serverData.value.holidayShowSettings.coefficient).to.equal(checkValue.result);
                    });
                    break;
                case 'from':
                    it('Начало времени показа в праздники ' + checkValue.name, function() {
                        expect(serverData.value.holidayShowSettings.showingFrom).to.equal(checkValue.result);
                    });
                    break;
                case 'to':
                    it('Окончание времени показа в праздники ' + checkValue.name, function() {
                        expect(serverData.value.holidayShowSettings.showingTo).to.equal(checkValue.result);
                    });
                    break;
                case 'doShow':
                    it('Показы в праздники ' + checkValue.name, function() {
                        expect(serverData.value.holidayShowSettings.isShowing).to.equal(checkValue.result === 1);
                    });
                    break;
                case 'mode':
                    it('Режим показа: ' + checkValue.name, function() {
                        expect(serverData.value.isExtendModeOn).to.equal(checkValue.result);
                    });
                    break;
                case 'timezone':
                    it('Часовой пояс: ' + checkValue.name, function() {
                        expect(serverData.value.timeZone.id).to.equal(checkValue.result);
                    });
                    break;
                case 'holidays':
                    it('Показы в праздники ' + checkValue.name, function() {
                        expect(serverData.value.isHolidaySettingsEnabled).to.equal(checkValue.result === 1);
                    });
                    break;
                case 'workingHolidays':
                    it('Показы в рабочие выходные ' + checkValue.name, function() {
                        expect(serverData.value.isWorkingWeekendEnabled).to.equal(checkValue.result === 1);
                    });
                    break;
                case 'timeTargets':
                    it('Режим показа ' + checkValue.name, function() {
                        expect(serverData.value.timeTargetCode).to.equal(checkValue.result);
                    });
                    break;
                case 'presets':
                    it('Пресет ' + checkValue.name, function() {
                        expect(serverData.value.preset).to.equal(checkValue.result);
                    });
                    break;
            }
        }

        describe('Настройка: стратегия.', function() {
            [
                {
                    name: 'автобюджет',
                    value: 1,
                    result: false
                },
                {
                    name: 'не автобюджет',
                value: 0,
                    result: true
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: true
                },
                {
                    name: '- не приходит с сервера',
                    value: undefined,
                    result: true
                }
            ].forEach(function(strategy) {
                var data = defaultValues;
                data.strategy.is_autobudget = strategy.value;
                runCheck(data, 'strategy', strategy);
            });
        });

        describe('Настройка: коэффициент показа.', function() {
            [
                {
                    name: 'не задан',
                    value: undefined,
                    result: DEFAULT_HOLIDAY_SHOWING_COEFFICIENT
                },
                {
                    name: '90%',
                    value: 90,
                    result: 90
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: DEFAULT_HOLIDAY_SHOWING_COEFFICIENT
                }
            ].forEach(function(coefficient) {
                var data = defaultValues;
                data.timetargeting.time_target_holiday_coef = coefficient.value;
                runCheck(data, 'coefficient', coefficient);
            });
        });

        describe('Настройка: начало времени показа в праздники.', function() {
            [
                {
                    name: 'не задан',
                    value: undefined,
                    result: DEFAULT_HOLIDAY_SHOWING_FROM
                },
                {
                    name: '0',
                    value: 0,
                    result: 0
                },
                {
                    name: '10.00',
                    value: 10,
                    result: 10
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: 0
                }
            ].forEach(function(showValue) {
                var data = defaultValues;
                data.timetargeting.time_target_holiday_from = showValue.value;
                runCheck(data, 'from', showValue);
            });
        });

        describe('Настройка: окончание времени показа в праздники.', function() {
            [
                {
                    name: 'не задан',
                    value: undefined,
                    result: DEFAULT_HOLIDAY_SHOWING_TO
                },
                {
                    name: '0',
                    value: 0,
                    result: 0
                },
                {
                    name: '19.00',
                    value: 19,
                    result: 19
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: 0
                }
            ].forEach(function(showValue) {
                var data = defaultValues;
                data.timetargeting.time_target_holiday_to = showValue.value;
                runCheck(data, 'to', showValue);
            });
        });

        describe('Настройка: показы в праздники.', function() {
            [
                {
                    name: 'включены',
                    value: 1
                },
                {
                    name: 'выключены',
                    value: 0
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: 0
                },
                {
                    name: '- не приходит с сервера',
                    value: undefined,
                    result: 0
                }
            ].forEach(function(showValue) {
                var data = defaultValues;
                data.timetargeting.time_target_holiday_dont_show = !showValue.value;
                showValue.result = showValue.result || showValue.value;
                runCheck(data, 'doShow', showValue);
            });
        });

        describe('Настройка: режимы показа.', function() {
            [
                {
                    name: 'простой',
                    value: 'simple',
                    result: false
                },
                {
                    name: 'расширенный',
                    value: 'extend',
                    result: true
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: false
                },
                {
                    name: '- не приходит с сервера',
                    value: undefined,
                    result: false
                }
            ].forEach(function(mode) {
                var data = defaultValues;
                data.strategy.is_autobudget = 0;
                data.timetargeting.timeTargetMode = mode.value;
                runCheck(data, 'mode', mode);
            });
        });

        describe('Настройка: часовые пояса.', function() {
            [
                {
                    name: 'Москва',
                    value: '130'
                },
                {
                    name: 'Калининград',
                    value: '131'
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: DEFAULT_TIMEZONE_ID
                },
                {
                    name: '- не приходит с сервера',
                    value: undefined,
                    result: DEFAULT_TIMEZONE_ID
                }
            ].forEach(function(timezone) {
                var data = defaultValues;
                data.timetargeting.timezone_id = timezone.value;
                timezone.result = timezone.result || timezone.value;
                runCheck(data, 'timezone', timezone);
            });
        });

        describe('Настройка: учитывать показы в праздники.', function() {
            [
                {
                    name: 'учитываются',
                    value: 1
                },
                {
                    name: 'не учитываются',
                    value: 0
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: 0
                },
                {
                    name: '- не приходит с сервера',
                    value: undefined,
                    result: 0
                }
            ].forEach(function(holiday) {
                var data = defaultValues;
                data.timetargeting.time_target_holiday = holiday.value;
                holiday.result = holiday.result || holiday.value;
                runCheck(data, 'holidays', holiday);
            });
        });

        describe('Настройка: учитывать показы в рабочие выходные.', function() {
            [
                {
                    name: 'учитываются',
                    value: 1
                },
                {
                    name: 'не учитываются',
                    value: 0
                },
                {
                    name: '- с сервера приходит пустая строка',
                    value: '',
                    result: 0
                },
                {
                    name: '- не приходит с сервера',
                    value: undefined,
                    result: 0
                }
            ].forEach(function(workingHoliday) {
                var data = defaultValues;
                data.timetargeting.time_target_working_holiday = workingHoliday.value;
                workingHoliday.result = workingHoliday.result || workingHoliday.value;
                runCheck(data, 'workingHolidays', workingHoliday);
            });
        });

        describe('Настройка: график показа.', function() {
            [
                {
                    name: 'приходит с сервера',
                    value: '1ABCDEFGHIJKLMNOPQRSTUVWX2ABCDEFGHIJKLMNOPQRSTUVWX3ABCDEFGHIJKLMNOPQRSTUVWX4ABCDEFGHIJKLMNOPQRSTUVWX5ABCDEFGHIJKLMNOPQRSTUVWX6ABCDEFGHIJKLMNOPQRSTUVWX7ABCDEFGHIJKLMNOPQRSTUVWX',
                    result: '1ABCDEFGHIJKLMNOPQRSTUVWX2ABCDEFGHIJKLMNOPQRSTUVWX3ABCDEFGHIJKLMNOPQRSTUVWX4ABCDEFGHIJKLMNOPQRSTUVWX5ABCDEFGHIJKLMNOPQRSTUVWX6ABCDEFGHIJKLMNOPQRSTUVWX7ABCDEFGHIJKLMNOPQRSTUVWX'
                },
                {
                    name: 'не приходит с сервера',
                    value: undefined,
                    result: u['b-time-targeting'].CODE_ALL_TIME
                },
                {
                    name: '- приходит пустая строка',
                    value: '',
                    result: u['b-time-targeting'].CODE_ALL_TIME
                }
            ].forEach(function(timeTarget) {
                var data = defaultValues;
                data.timetargeting.timeTarget = timeTarget.value;
                runCheck(data, 'timeTargets', timeTarget);
            });
        });

        describe('Настройка: пресеты.', function() {
            [
                {
                    name: 'По рабочим дням',
                    value: 'worktime'
                },
                {
                    name: 'Круглосуточно',
                    value: 'all'
                },
                {
                    name: 'Другое',
                    value: 'other'
                },
                {
                    name: 'Не приходит с сервера',
                    value: undefined,
                    result: 'other'
                },
                {
                    name: 'пустая строка',
                    value: '',
                    result: 'other'
                }
            ].forEach(function(preset) {
                var data = defaultValues;
                data.timetargeting.time_target_preset = preset.value;
                preset.result = preset.result || preset.value;
                runCheck(data, 'presets', preset);
            });
        });
    });

    describe('Обработка данных для передачи на сервер', function() {
        var defaultData = {
            "timeZone": {
                "id": "130",
                "text": "Москва"
            },
            "timeTargetCode": "1ABCDEFGHIJKLMNOPQRSTUVWX2ABCDEFGHIJKLMNOPQRSTUVWX3ABCEFGHIKLMNOPQRSTUVWX4ABCDEFGHIJKLMNOPQRSTUVWX5ABCDEFGHIJKLMNOPQRSTUVWX6ABCDEFGHIJKLMNOPQRSTUVWX7ABCDEFGHIJKLMNOPQRSTUVWX",
            "isHolidaySettingsEnabled": false,
            "isWorkingWeekendEnabled": true,
            "holidayShowSettings": {
                "isShowing": true,
                "showingFrom": 8,
                "showingTo": 20,
                "coefficient": 100
            },
            "isExtendModeOn": false,
            "preset": "worktime"
        },
        timezones = [
            {
                value: {
                    "id": "130",
                    "text": "Москва"
                },
                name: 'Москва',
                result: '130'
            },
            {
                value: {
                    "id": "131",
                    "text": "Калининград"
                },
                name: 'Калининград',
                result: '131'
            }
        ],
        holidaySettingsEnabled = [
            {
                value: true,
                name: 'включен',
                result: '1'
            },
            {
                value: false,
                name: 'выключен',
                result: '0'
            }
        ],
        isWorkingWeekendEnabled = [
            {
                value: true,
                name: 'учитывается',
                result: '1'
            },
            {
                value: false,
                name: 'не учитывается',
                result: '0'
            }
        ],
        isShowingOnHolidays = [
            {
                value: true,
                name: 'включен',
                result: '0'
            },
            {
                value: false,
                name: 'выключен',
                result: '1'
            }
        ],
        isExtendModeOn = [
            {
                value: true,
                name: 'включен',
                result: 'extend'
            },
            {
                value: false,
                name: 'выключен',
                result: 'simple'
            }
        ],
        preset = [
            {
                value: 'all',
                result: 'all',
                name: 'круглосуточно'
            },
            {
                value: 'worktime',
                result: 'worktime',
                name: 'по рабочим дням'
            },
            {
                value: 'other',
                result: 'other',
                name: 'другое'
            }
        ];

        [
            { name: 'Пресет', params: preset, field: 'preset', resultField: 'time_target_preset'},
            { name: 'Расширенный режим', params: isExtendModeOn, field: 'isExtendModeOn', resultField: 'timeTargetMode'},
            { name: 'Показы по праздникам', params: isShowingOnHolidays, field: 'isShowing', resultField: 'time_target_holiday_dont_show', isHolidays: true},
            { name: 'Показы по рабочим выходным', params: isWorkingWeekendEnabled, field: 'isWorkingWeekendEnabled', resultField: 'time_target_working_holiday'},
            { name: 'Учитывать показы по праздникам', params: holidaySettingsEnabled, field: 'isHolidaySettingsEnabled', resultField: 'time_target_holiday'},
            { name: 'Часовые пояса', params: timezones, field: 'timeZone', resultField: 'timezone_id'},
            { name: 'Код таргетинга', field: 'timeTargetCode', resultField: 'timeTarget'},
            { name: 'Начало показов по праздникам', field: 'showingFrom', resultField: 'time_target_holiday_from', isHolidays: true},
            { name: 'Окончание показов по праздникам', field: 'showingTo', resultField: 'time_target_holiday_to', isHolidays: true},
            { name: 'Коэффициент показов по праздникам', field: 'coefficient', resultField: 'time_target_holiday_coef', isHolidays: true}
        ].forEach(function(paramGroup) {
            describe('Настройка: ' + paramGroup.name + ', ', function() {
                var settings = defaultData,
                    defaultValue = paramGroup.isHolidays ? settings.holidayShowSettings[paramGroup.field].toString() : settings[paramGroup.field].toString(),
                    serverData;

                !paramGroup.params && (paramGroup.params = [ { name: 'по умолчанию', value: defaultValue, result: defaultValue }]);

                paramGroup.params.forEach(function(param) {
                    it('Значение: ' + param.name, function() {
                        paramGroup.isHolidays ? settings.holidayShowSettings[paramGroup.field] = param.value : settings[paramGroup.field] = u._.cloneDeep(param.value);
                        serverData = utils.toServer(settings);

                        expect(serverData[paramGroup.resultField]).to.equal(param.result);
                    })
                })
            })
        })
    })
});
