/**
 * @typedef {Object} HierarchicalMultipliers
 * @property {{is_enabled: <Boolean>}} retargeting_multiplier
 * @property {{is_enabled: <Boolean>}} demography_multiplier
 * @property {{is_enabled: <Boolean>}} mobile_multiplier
 * @property {{is_enabled: <Boolean>}} video_multiplier
 */

/**
 * @typedef {Object} HierarchicalMultiplierMeta
 * @property {Number} pct_max
 * @property {Number} pct_min
 * @property {Number} default_multiplier
 * @property {Number} max_conditions
 */

/**
 * @typedef {Object} HierarchicalMultipliersMeta
 * @property {HierarchicalMultiplierMeta} retargeting_multiplier
 * @property {HierarchicalMultiplierMeta} demography_multiplier
 * @property {HierarchicalMultiplierMeta} mobile_multiplier
 * @property {HierarchicalMultiplierMeta} video_multiplier
 * @property {HierarchicalMultiplierMeta} performance_tgo_multiplier
 */

/**
 * @typedef {Object} HierarchicalMultipliersPct
 * @property {Number} cpm_banner
 * @property {Number} inpage
 * @property {Number} instream_web
 * @property {Number} interstitial
 */

(function() {
    /**
     * Возвращает значение корректировки ставки, которое будет отображаться в интерфейсе
     * @param {Number} multiplier значение корректировки
     * @param {Number} defaultMultiplier корректировка по умолчанию
     * @returns {Number}
     */
    function getInputValue(multiplier, defaultMultiplier) {
        if (multiplier > defaultMultiplier) {
            return multiplier - defaultMultiplier;
        }

        return defaultMultiplier - multiplier;
    }

    /**
     * Формирует данные базовой модели корректировки ставки
     * @param {String} modelId
     * @param {Object} data
     * @param {Number} data.multiplier_pct значение корректировки
     * @param {Boolean} data.is_enabled флаг "включенности" корректировки
     * @param {HierarchicalMultipliersMeta} metaData дополнительные данные
     */
    function getBaseMultiplierModelData(modelId, data, metaData, entity) {
        return u._.extend(
            {
                modelId: modelId,
                // !!RATES_NAME.hasOwnProperty('is_enabled')  || !RATES_NAME.multiplier_pct:
                // делаем is_enabled по умолчанию true
                isEnabled: data.hasOwnProperty('is_enabled') || !data.multiplier_pct,
                input: getInputValue(data.multiplier_pct, metaData.default_multiplier),
                sign: !data.multiplier_pct || data.multiplier_pct >= metaData.default_multiplier ?
                    'increment' :
                    'decrement',
                coefficient: data.multiplier_pct,
                mediaType: entity.adgroup_type
            },
            metaData
        );
    }

    u.register({

        'i-adjustment-rates-data': {
            /**
             * Расширяет группу данными по корректировкам ставок
             * @param {{ modelId: <String>, hierarchical_multipliers: HierarchicalMultipliers }} entity расширяемая группа
             * @param {Object} contextData дополнительные данные
             * @param {HierarchicalMultipliersMeta} contextData.multipliers_meta дополнительные данные
             * @param {String} [contextData.modelId]
             * @param {Boolean} [contextData.isDevicesEnabled] Разрешены ли корректировки для устройств
             * @param {Boolean} [contextData.isWithoutDeviceRestrictions] Разрешено ли отключать все мобильные устройства
             * и устанавливать десктопы на максимум
             * @param {Boolean} [contextData.isCpmGroupTypeEnabled] Разрешены ли корректировки по типу группы
             */
            patchMultipliers: function(entity, contextData) {
                if (!entity.hierarchicalMultipliers) {
                    var multipliers = entity.hierarchical_multipliers || {},
                        meta = contextData.multipliers_meta,

                        retargeting = multipliers.retargeting_multiplier || {},
                        demography = multipliers.demography_multiplier || {},
                        mobile = multipliers.mobile_multiplier || {},
                        desktop = multipliers.desktop_multiplier || {},
                        video = multipliers.video_multiplier || {},
                        performanceTGO = multipliers.performance_tgo_multiplier || {},
                        bannerType = multipliers.banner_type_multiplier || {},
                        inventory = multipliers.inventory_multiplier || {},
                        weather = multipliers.weather_multiplier || {},
                        trafficJam = u._.find(
                            multipliers.expression_multipliers,
                            { type: 'express_traffic_multiplier' }
                        ) || {},

                        retargetingConditions = retargeting.conditions || {},
                        demographyConditions = demography.conditions || [],

                        defaultMultiplier = 100,

                        retargetingMeta = {
                            pct_max: u._.get(meta, 'retargeting_multiplier.pct_max'),
                            pct_min: u._.get(meta, 'retargeting_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier
                        },
                        demographyMeta = {
                            pct_max: u._.get(meta, 'demography_multiplier.pct_max'),
                            pct_min: u._.get(meta, 'demography_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier
                        },
                        mobileMeta = {
                            pct_max: u._.get(meta, 'mobile_multiplier.pct_max'),
                            pct_min: u._.get(meta, 'mobile_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier
                        },
                        videoMeta = {
                            pct_max: u._.get(meta, 'video_multiplier.pct_max'),
                            pct_min: u._.get(meta, 'video_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier
                        },
                        performanceTGOMeta = {
                            pct_max: u._.get(meta, 'performance_tgo_multiplier.pct_max'),
                            pct_min: u._.get(meta, 'performance_tgo_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier
                        },
                        devicesMeta = {
                            mobile_pct_max: u._.get(meta, 'mobile_multiplier_with_feature.pct_max'),
                            mobile_pct_min: u._.get(meta, 'mobile_multiplier_with_feature.pct_min'),
                            desktop_pct_max: !contextData.isWithoutDeviceRestrictions ?
                                u._.get(meta, 'desktop_multiplier_with_mobile_os_feature.pct_max') :
                                u._.get(meta, 'desktop_multiplier.pct_max'),
                            desktop_pct_min: !contextData.isWithoutDeviceRestrictions ?
                                u._.get(meta, 'desktop_multiplier_with_mobile_os_feature.pct_min') :
                                u._.get(meta, 'desktop_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier,
                            isMobileSwitchOffEnabled: contextData.isWithoutDeviceRestrictions
                        },
                        cpmGroupTypeMeta = {
                            banners_pct_max: u._.get(meta, 'banner_type_multiplier.pct_max'),
                            banners_pct_min: u._.get(meta, 'banner_type_multiplier.pct_min'),
                            inventory_pct_max: u._.get(meta, 'inventory_multiplier.pct_max'),
                            inventory_pct_min: u._.get(meta, 'inventory_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier
                        },
                        weatherMeta = {
                            pct_max: u._.get(meta, 'weather_multiplier.pct_max'),
                            pct_min: u._.get(meta, 'weather_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier,
                        },
                        trafficJamMeta = {
                            pct_max: u._.get(meta, 'express_traffic_multiplier.pct_max'),
                            pct_min: u._.get(meta, 'express_traffic_multiplier.pct_min'),
                            default_multiplier: defaultMultiplier,
                        },
                        modelId = entity.modelId || contextData.modelId,
                        modelData = {},
                        devicesRates,
                        trafficJamRates,
                        isDevicesEnabled = contextData.isDevicesEnabled;

                    // видео
                    modelData.video = getBaseMultiplierModelData(modelId, video, videoMeta, entity);

                    // Смарт-объявления
                    modelData['performance-tgo'] = getBaseMultiplierModelData(modelId, performanceTGO, performanceTGOMeta, entity);

                    // ретаргетинг
                    modelData.retargeting = {
                        modelId: modelId,
                        isEnabled: !retargeting.hasOwnProperty('is_enabled') || !!(+retargeting.is_enabled),
                        max_conditions: u._.get(meta, 'retargeting_multiplier.max_conditions'),

                        common_control: u._.extend(
                            {
                                modelId: 'common-retargeting' + modelId,
                                isCommon: true
                            },
                            retargetingMeta),
                        rates: u._.keys(retargetingConditions).map(function(retargetingId) {
                            return u._.extend(
                                {
                                    sign: retargetingConditions[retargetingId].multiplier_pct >= defaultMultiplier ?
                                        'increment' :
                                        'decrement',
                                    modelId: retargetingId,
                                    retargetingId: retargetingId,
                                    input: getInputValue(
                                        retargetingConditions[retargetingId].multiplier_pct,
                                        retargetingMeta.default_multiplier
                                    )
                                },
                                retargetingMeta
                            );
                        }),
                        mediaType: entity.adgroup_type
                    };

                    // демография
                    modelData.demography = {
                        modelId: modelId,
                        isEnabled: !demography.hasOwnProperty('is_enabled') || !!(+demography.is_enabled),
                        max_conditions: u._.get(meta, 'demography_multiplier.max_conditions'),
                        canChangeSign: true,
                        common_control: u._.extend(
                            { modelId: 'common-demography' + modelId, isCommon: true },
                            demographyMeta),
                        rates: demographyConditions.reduce(function(res, condition) {
                            var rateData = u._.extend(
                                {
                                    sign: !condition.multiplier_pct && condition.multiplier_pct !== 0 ||
                                        condition.multiplier_pct >= defaultMultiplier ? 'increment' : 'decrement',
                                    age: condition.age === null ? 'all' : condition.age,
                                    gender: condition.gender === null ? 'all' : condition.gender,
                                    input: getInputValue(condition.multiplier_pct, demographyMeta.default_multiplier)
                                },
                                demographyMeta);

                            // раньше было значение 45+,
                            // в задаче https://st.yandex-team.ru/DIRECT-66902 вместо 45+ появились 2 новых значения: 45-54 и 55+
                            // данное условие обеспечивает обратную совместимость
                            if (condition.age === '45-') {
                                res.push(u._.extend({}, rateData, { age: '45-54' }));
                                res.push(u._.extend({}, rateData, { age: '55-' }));
                            } else {
                                res.push(rateData);
                            }

                            return res;
                        }, []),
                        mediaType: entity.adgroup_type
                    };

                    modelData.mobile = getBaseMultiplierModelData(
                        modelId,
                        // Если доступны корректировки на устройство, либо выбран конкрентный тип OS для мобильного
                        isDevicesEnabled || mobile.os_type ?
                            u._.omit(mobile, ['multiplier_pct']) :
                            mobile,
                        mobileMeta,
                        entity
                    );

                    devicesRates = isDevicesEnabled ?
                        u._.compact([
                            u._.isUndefined(mobile.multiplier_pct) ?
                                false :
                                u._.extend(
                                    {
                                        input: getInputValue(mobile.multiplier_pct, defaultMultiplier),
                                        sign: !mobile.multiplier_pct && mobile.multiplier_pct !== 0 ||
                                            mobile.multiplier_pct >= defaultMultiplier ? 'increment' : 'decrement',
                                        device: u._.get(mobile, 'os_type') || 'all_mobile'
                                    },
                                    devicesMeta
                                ),
                            u._.isUndefined(desktop.multiplier_pct) ?
                                false :
                                u._.extend(
                                    {
                                        input: getInputValue(desktop.multiplier_pct, defaultMultiplier),
                                        sign: !desktop.multiplier_pct && desktop.multiplier_pct !== 0 ||
                                            desktop.multiplier_pct >= defaultMultiplier ? 'increment' : 'decrement',
                                        device: 'desktop'
                                    },
                                    devicesMeta
                                )
                        ]) :
                        [];

                    modelData.devices = {
                        modelId: modelId,
                        isEnabled: isDevicesEnabled,
                        max_conditions: 2,
                        common_control: u._.extend(
                            { modelId: 'common-devices' + modelId, isCommon: true },
                            devicesMeta
                        ),
                        rates: devicesRates,
                        choosedDevices: u._.map(devicesRates, 'device'),
                        isMobileSwitchOffEnabled: contextData.isWithoutDeviceRestrictions
                    };

                    modelData['cpm-group-type'] = {
                        modelId: modelId,
                        isEnabled: contextData.isCpmGroupTypeEnabled,
                        bannersMaxConditions: u.consts('BANNER_TYPE_MULTIPLIER_TYPES').length,
                        inventoryMaxConditions: u.consts('INVENTORY_MULTIPLIER_TYPES').length,

                        common_control: u._.extend(
                            { modelId: 'common-cpm-group-type' + modelId, isCommon: true },
                            cpmGroupTypeMeta
                        ),
                        rates: []
                            .concat(bannerType.conditions || [])
                            .concat(inventory.conditions || [])
                            .map(function(condition) {
                                var defMultiplier = cpmGroupTypeMeta.default_multiplier;

                                return u._.extend(
                                    {
                                        type: condition.multiplier_type,
                                        sign: condition.multiplier_pct >= defMultiplier ? 'increment' : 'decrement',
                                        input: getInputValue(condition.multiplier_pct, defMultiplier)
                                    },
                                    cpmGroupTypeMeta
                                );
                            }),
                        mediaType: entity.adgroup_type
                    };

                    modelData.weather = u['i-adjustment-rates-data'].CNFtoWeather(weather, weatherMeta, modelId);

                    trafficJamRates = (trafficJam.adjustments || []).map(function(item) {
                        var multiplierPct = Number(item.multiplier_pct);

                        return u._.extend(
                            {
                                value: u._.get(item, 'condition[0][0].value'),
                                sign: multiplierPct >= defaultMultiplier ? 'increment' : 'decrement',
                                input: getInputValue(multiplierPct, defaultMultiplier)
                            },
                            trafficJamMeta
                        );
                    });

                    modelData['traffic-jam'] = {
                        modelId: modelId,
                        isEnabled: trafficJam.is_enabled === '1' || !multipliers.expression_multipliers,
                        max_conditions: u['b-rates-chooser'].trafficJam.getValues().length,
                        common_control: u._.extend(
                            { modelId: 'common-traffic-jam' + modelId, isCommon: true },
                            trafficJamMeta
                        ),
                        rates: trafficJamRates,
                        choosedValues: u._.map(trafficJamRates, 'value')
                    };

                    entity.hierarchicalMultipliers = modelData;
                }
            },

            weatherToCNF: function(data) {
                if (data.setType === 'and') {
                    return [{
                        multiplier_pct: data.common_control.coefficient,
                        expression: data.rates.reduce(function(result, rate) {
                            var parameter = rate.parameter;

                            if (rate.parameter === 'temp') {
                                result.push(
                                    [{ parameter: parameter, operation: 'le', value: Math.max(rate.le, rate.ge) }],
                                    [{ parameter: parameter, operation: 'ge', value: Math.min(rate.le, rate.ge) }]
                                )
                            } else {
                                result.push(
                                    rate.values.map(function(value) { // массив дизъюнкций
                                        return { parameter: parameter, operation: 'eq', value: value };
                                    })
                                );
                            }

                            return result;
                        }, [])
                    }];
                }

                return data.rates.reduce(function(result, rate) {
                    var parameter = rate.parameter,
                        isTemp = rate.parameter === 'temp';

                    return result.concat(
                        (isTemp || rate.values.length) ?
                            {
                                multiplier_pct: rate.coefficient,
                                expression: isTemp ?
                                    [ // массив конъюнкций
                                         [{ parameter: parameter, operation: 'le', value: Math.max(rate.le, rate.ge) }],
                                         [{ parameter: parameter, operation: 'ge', value: Math.min(rate.le, rate.ge) }]
                                    ] :
                                    [ // массив конъюнкций
                                        rate.values.map(function(value) { // массив дизъюнкций
                                            return { parameter: parameter, operation: 'eq', value: value };
                                        })
                                    ]
                            } :
                            []
                    );
                }, []);
            },

            CNFtoWeather: function(data, meta, modelId) {
                var conditions = data.conditions || [],
                    conditionsCount = conditions.length,
                    setType = conditionsCount ? (conditionsCount > 1 ? 'or' : 'and') : 'or',
                    firstCondition = conditions[0],
                    weatherConsts = u.consts('WEATHER_MULTIPLIER_CONSTRAINTS');

                return {
                    modelId: modelId,
                    setType: setType,
                    isEnabled: true,
                    max_conditions: 20,
                    rates: conditions.reduce(function(result, condition) {
                        var grouped = u._.groupBy(u._.flatten(condition.expression), 'parameter');

                        u._.forEach(grouped, function(items, parameter) {
                            result.push(
                                u._.assign(
                                    { parameter: parameter },
                                    setType === 'and' ?
                                        { input: meta.default_multiplier } :
                                        {
                                            input: getInputValue(condition.multiplier_pct, meta.default_multiplier),
                                            sign: !condition.multiplier_pct ||
                                                condition.multiplier_pct >= meta.default_multiplier ?
                                                    'increment' :
                                                    'decrement'
                                        },
                                    parameter === 'temp' ?
                                        items.reduce(function(result, item) {
                                            return u._.set(result, item.operation, item.value);
                                        }, {}) :
                                        {
                                            values: u._.map(items, 'value'),
                                            valuesEnum: u._.get(weatherConsts, [parameter, 'values'], [])
                                        },
                                    meta
                                )
                            );
                        });

                        return result;
                    }, []),
                    common_control: u._.assign(
                        { modelId: 'common-weather' + modelId },
                        setType === 'and' && firstCondition ?
                            {
                                input: getInputValue(firstCondition.multiplier_pct, meta.default_multiplier),
                                sign: !firstCondition.multiplier_pct ||
                                    firstCondition.multiplier_pct >= meta.default_multiplier ?
                                        'increment' :
                                        'decrement'
                            } :
                            {},
                        meta
                    )
                };
            },

            /**
             * Возвращает значение корректировок ставок, которые используются в прогнозе
             * @param {HierarchicalMultipliers} hierarchicalMultipliers
             * @returns {HierarchicalMultipliersPct}
             */
            getMultipliersPct: function(hierarchicalMultipliers) {
                var multipliersForForecast = ['banner_type_multiplier', 'inventory_multiplier'],
                    multipliers = u._.pick(hierarchicalMultipliers, multipliersForForecast);

                return Object.keys(multipliers).reduce(function(res, key) {
                    multipliers[key].conditions.forEach(function(type) {
                        res[type.multiplier_type] = type.multiplier_pct;
                    });

                    return res;
                }, {});
            }
        }
    });
})();
