/**
 * @typedef {Object} Point
 * @property {Number} x - количество миллисекунд
 * @property {Number} y - значение
 */

/**
 * @typedef {Object} Row
 * @property {String} name - заголовок для легенды
 * @property {String} column - тип столбца
 * @property {String} [groupBy] - тип среза
 * @property {Array<Point>} data - список точек
 */

/**
 * @typedef {Object} Filters
 * @property {Array<String>} columns - список выбранных столбцов
 * @property {Array<String>} groupBy - список выбранных срезов
 */

BEM.decl('i-chart-stat-data', {}, {

    /**
     * Совершает запрос
     * @param {Object} params
     * @param {Object} [ctx]
     * @returns {$.Deferred}
     */
    get: function(params, ctx) {
        var deferred = $.Deferred(),
            request = BEM.create('i-request_type_ajax', {
                url: '/registered/main.pl',
                cache: false,
                dataType: 'json',
                type: 'POST',
                timeout: 180000,
                callbackCtx: this
            });

        params = u._.extend({ columns: [], groupBy: [] }, params);

        request.get(
            this._getSubmitParams(params),
            function(result) {
                if (result.error) {
                    deferred.rejectWith(ctx || this, [result.error]);
                } else {
                    deferred.resolveWith(ctx || this, [{
                        type: result.type,
                        series: this.provideData(result, {
                            columns: params.columns,
                            groupBy: params.groupBy
                        })
                    }]);
                }
            },
            function(error) {
                deferred.rejectWith(ctx || this, arguments);
            });

        // обрываем запрос
        deferred.fail(function(e, eventName) {
            eventName === 'abort' && request.abort();
        });

        return deferred;
    },

    /**
     * Возвращает параметры для запроса
     * @param {Object} params
     * @param {Object} params.date
     * @param {String} params.date.from - начало периода
     * @param {String} params.date.to - конец периода
     * @param {String} [params.cid] - id кампании
     * @param {Number} [params.multi_clients_mode] - флаг мультиклиентности, принимает значения 1 или ничего
     * @param {String} [params.ulogin] - логин клиента
     * @param {String} [params.statType] - тип станицы статистики
     * @param {Number} [params.withNds] - флаг включенной опции НДС
     * @param {String} [params.groupByDate] - периоды по которым группировать статистику
     * @param {Object} [params.filters] - фильтры (формат 'b-statistic-filters-editor')
     * @param {Array<String>} [params.columns] - список столбцов
     * @param {Array<String>} [params.groupBy] - список срезов
     * @returns {Object}
     * @private
     */
    _getSubmitParams: function(params) {
        var submitParams = {
            show_stat: 1,
            cmd: 'showStat',
            format: 'plot',
            date_from: params.date.from,
            date_to: params.date.to,
            with_nds: params.withNds || 0,
            group_by_date: params.groupByDate,
            goals: params.goals,
            json_filters: JSON.stringify(params.filters),
            json_columns: JSON.stringify(params.columns),
            json_group_by: JSON.stringify(params.groupBy),
            json_columns_positions: JSON.stringify(params.columns),
            json_group_by_positions: JSON.stringify(params.groupBy),
            attribution_model: params.attribution_model
        };

        params.multi_clients_mode && (submitParams.multi_clients_mode = params.multi_clients_mode);
        params.ulogin && (submitParams.ulogin = params.ulogin);

        params.cid && (submitParams.cid = params.cid);
        params.statType && (submitParams.stat_type = params.statType);

        return submitParams;
    },

    /**
     * Возвращает данные для графика
     * https://wiki.yandex-team.ru/Direkt/TechnicalDesign/Master-otchetov-grafiki/
     * @param {Object} data
     *  @param {Array<String>} data.slice_values - все возможные значения среза (id кампаний, баннеров, текст фраз)
     *  @param {Array<Object>} data.stat - список с данными в разрезе на каждую дату в периоде, за который строится график
     * @param {Filters} filters
     * @returns {Array<Row>|[]}
     */
    provideData: function(data, filters) {
        switch (data.type) {
            case 'columns':
                return this._provideColumns(data, filters);
            case 'slices':
                return this._provideSlices(data, filters);
            default:
                return [];
        }
    },

    /**
     * Формирует данные для графика выбранным столбцам
     * @param {Object} data
     * @param {Filters} filters
     * @returns {Array<Row>}
     * @private
     */
    _provideColumns: function(data, filters) {
        return filters.columns.map(function(columnName) {
            return {
                name: columnName,
                column: columnName,
                data: data.stat.map(function(columnData) {
                    // по хорошему должен быть slice а не slices
                    var sliceData = columnData.slices;

                    return {
                        x: u.moment(columnData.stat_date, 'YYYY-MM-DD').valueOf(),
                        y: Number(sliceData[columnName] || this.DEFAULT_VALUE)
                    };
                }, this)
            };
        }, this);
    },

    /**
     * Формирует данные для графика по выбранному срезу и столбцу
     * @param {Object} data
     *  @param {Array<String>} data.slice_values - все возможные значения среза (id кампаний, баннеров, текст фраз)
     *  @param {Array<Object>} data.stat - список с данными в разрезе на каждую дату в периоде, за который строится график
     * @param {Filters} filters
     * @returns {Array<Row>}
     * @private
     */
    _provideSlices: function(data, filters) {
        var groupBy = filters.groupBy[0],
            columnName = filters.columns[0];

        return data.slice_values.map(function(sliceName, sliceIndex) {
            var seriesName,
                seriesData = data.stat.map(function(columnData) {
                    var sliceData = columnData.slices[sliceName] || {};

                    // пытаемся получить понятное/читаемое для клиента значение среза
                    if (this._sliceNameToSeriesName[groupBy] && typeof seriesName === 'undefined') {
                        seriesName = sliceData[this._sliceNameToSeriesName[groupBy]];

                        // Некотрые названия форматируем на клинте
                        if (typeof this._formatSeriesName[groupBy] === 'function' && seriesName !== 'undefined') {
                            seriesName = this._formatSeriesName[groupBy](seriesName);
                        }
                    }

                    return {
                        x: u.moment(columnData.stat_date, 'YYYY-MM-DD').valueOf(),
                        y: Number(sliceData[columnName]) || this.DEFAULT_VALUE
                    };
                }, this);

            // если понятное/читаемое для клиента значение среза не относится к специальным,
            // то присваиваем значение из slice_values
            if (typeof seriesName === 'undefined') {
                seriesName = sliceName;
            }

            // страховка от срезов, значения которых отсутствуют
            // например если у клиента не заданы «Метки» то sliceName === ''
            if (!seriesName) {
                // notSet ипользуется в i-utils__text-store.utils.js
                seriesName = 'notSet';
            }

            return {
                name: seriesName,
                groupBy: groupBy,
                column: columnName,
                data: seriesData
            };
        }, this);
    },

    /**
     * Возвращает данные по конкретному значению среза
     * @param {Object} params
     * @param {Array} params.slices - список с данными по всем значениям среза
     * @param {String} params.groupBy - тип среза
     * @param {String} params.sliceName - значение среза
     * @param {String} params.columnName - тип столбца
     * @returns {Object}
     * @private
     */
    _getSliceData: function(params) {
        var slices = params.slices,
            sliceName = params.sliceName,
            groupBy = params.groupBy,
            data;

        if (this._specialSlices[groupBy]) {
            data = this._specialSlices[groupBy](slices, sliceName);
        } else {
            data = u._.find(slices, function(item) {
                return sliceName === item[groupBy];
            });
        }

        // приравниваем к нулю, если нет данных
        if (!data) {
            data = {};
            data[params.columnName] = this.DEFAULT_VALUE;
        }

        return data;
    },

    _formatSeriesName: {
        banner: function(bid) {
            return iget2('i-chart-stat-data', 'm-s', 'M-{foo}', {
                foo: bid
            });
        }
    },

    /**
     * Срезы значение которых необходимо брать из специальный полей
     */
    _sliceNameToSeriesName: {
        campaign: 'camp_name',
        adgroup: 'adgroup_name',
        banner: 'bid',
        region: 'region_name',
        tags: 'tag_names',
        page_group: 'page_name',
        retargeting_coef: 'coef_ret_cond_name',
        physical_region: 'physical_region_name',
        deal: 'deal_name',
        content_targeting: 'content_targeting_name'
    },

    /**
     * Содержит частные случаи получения данных для среза
     */
    _specialSlices: {
        campaign: function(slices, sliceValue) {
            return u._.find(slices, function(item) {
                return sliceValue === item.cid;
            });
        },
        adgroup: function(slices, sliceValue) {
            return u._.find(slices, function(item) {
                return sliceValue === item.adgroup_id;
            });
        },
        banner: function(slices, sliceValue) {
            return u._.find(slices, function(item) {
                return sliceValue === item.bid;
            });
        },
        retargeting_coef: function(slices, sliceValue) {
            return u._.find(slices, function(item) {
                return sliceValue === item.coef_ret_cond_id;
            });
        }
    },

    DEFAULT_VALUE: 0

});
