(function() {
    /**
     * @typedef {Object} BannerSummaryData
     * @property {String} modelId идентификатор модели объявления
     * @property {String} bid идентификатор объявления
     * @property {Boolean} isNewBanner является ли объявление новым
     * @property {Number} [newBannerIndex] индекс объявления, если оно является новым
     */
    /**
     * @typedef {Object} GroupSummaryData
     * @property {String} modelId идентификатор модели группы
     * @property {Boolean} isNewGroup является ли группа новой
     * @property {Boolean} isCopyGroup является ли группа - копией какой-то другой
     * @property {Number} [newGroupIndex] индекс группы, если она является новой
     * @property {BannerSummaryData[]} banners данные об объявлениях
     */

        /**
         * Говорит, является ли узел из дерева ошибок - описанием одной или нескольких ошибок
         * @param {*} item узел из дерева структуры ошибок
         * @returns {Boolean}
         */
    var isErrorsDescription = function(item) {
            return Array.isArray(item) && item.length > 0 || !u._.isEmpty(u._.get(item, 'object_errors'));
        },
        /**
         * Говорит, если ли в переданном словаре ошибок сущности какие-то ошибки
         * Просто наличие объекта object_errors не обязательно означает, что есть ошибки
         * По ключам могут присутствовать null вместо массивов или пустые массивы
         *
         * @param {Object} errorsHash хеш с ошибками; см. object_errors в i-utils__error
         * @returns {Boolean}
         */
        objectHasErrors = function(errorsHash) {
            return u._.values(errorsHash).some(function(value) {
                return isErrorsDescription(value) ||
                    (value && value.array_errors && value.array_errors.some(isErrorsDescription));
            });
        };

    BEM.decl('i-error-groups-summary', {}, {
        /**
         * Извлекает из данных о группе данные, необходимые для отображения в сводке ошибок о группе
         * @param {Object} group данные из модели группы
         * @returns {GroupSummaryData}
         */
        groupToSummaryData: function(group) {
            var result = u._.pick(group, ['modelId', 'isNewGroup', 'isCopyGroup', 'newGroupIndex']);

            result.banners = group.banners.map(function(banner) {
                return u._.pick(banner, ['modelId', 'bid', 'newBannerIndex', 'isNewBanner']);
            });

            return result;
        },
        /**
         * Приводит "серверный" формат ошибок в формат, котороый принимает b-error-groups-summary
         * @param {GroupSummaryData[]} groupsData данные о группах
         * @param {Object[]} errorsData данные об ошибках в "серверном" формате
         * @param {Object} externalChecker - содержит внешние функции для проверки ошибок в баннере/группе
         * @returns {GroupSummaryCtx}
         */
        summarizeErrors: function(groupsData, errorsData, externalChecker) {
            externalChecker = u._.extend({
                bannerHasErrors: function() { return false; },
                groupHasErrors: function() { return false; }
            }, externalChecker || {});

            var genericErrors = u._.get(errorsData, 'groups.generic_errors', []),
                groupErrors = u._.get(errorsData, 'groups.array_errors', []),
                title = groupErrors.reduce(function(acc, groupErrors, index) {
                    if (u._.get(groupErrors, 'object_errors.group_name', []).length > 0) {
                        acc.push(groupsData[index]);
                    }

                    return acc;
                }, []),
                groups = groupErrors.reduce(function(acc, groupErrors, index) {
                    var objectErrorsHash = u._.omit(
                            u._.get(groupErrors, 'object_errors', {}),
                            ['group_name', 'banners']
                        ),
                        genericErrorsArray = u._.get(groupErrors, 'generic_errors', {});

                    if (objectHasErrors(objectErrorsHash) ||
                        isErrorsDescription(genericErrorsArray) ||
                        externalChecker.groupHasErrors(objectErrorsHash)) {

                        acc.push(groupsData[index]);
                    }

                    return acc;
                }, []),
                banners = groupErrors.reduce(function(acc, groupErrors, groupIndex) {
                    var bannersErrors = u._.get(groupErrors, 'object_errors.banners.array_errors', []);

                    bannersErrors.forEach(function(banner, bannerIndex) {
                        var group = groupsData[groupIndex],
                            bannerErrorsHash = u._.get(banner, 'object_errors', []),
                            genericBannerErrorsArray = u._.get(banner, 'generic_errors', []);

                        if (objectHasErrors(bannerErrorsHash) ||
                            isErrorsDescription(genericBannerErrorsArray) ||
                            externalChecker.bannerHasErrors(bannerErrorsHash)) {

                            acc.push({
                                group: group,
                                banner: group.banners[bannerIndex]
                            });
                        }
                    });

                    return acc;
                }, []);

            return {
                generic: genericErrors,
                title: title,
                groups: groups,
                banners: banners
            };
        }
    });

}());
