/**
 * @typedef FilterData сериализованный фильтр, подготовленный к сохранению
 * @see dm-feed-filter
 */

/**
 * @typedef {Object} FilterValidationController контроллер реакции на валидацию фильтра
 * @property {Function} save функция, выполняющая сохранение фильтра
 * @property {ValidationControllerErrorDisplayCallback} showErrors функция отображения ошибок
 * @property {Function} onRequestError реакция на ошибку запроса валидации
 */

/**
 * Функция отображения ошибок валидации в контроллере валидации фильтра
 * @callback ValidationControllerErrorDisplayCallback
 * @param {Array<{ description: String }>} errors ошибки для отображения
 */

/**
 * @typedef {Object} FilterValidationResult результат валидации фильтра
 * @proeprty {Object} validationResult результат валидации в "универсальном" формате @see i-utils__error
 * @property {Number} filterPosition позиция валидируемого фильтра в списке фильтров
 */

BEM.decl('i-feed-filter-validate', {}, {
    /**
     * Валидирует фильтр filterData в рамках группы groupModel
     * @param {FilterData} filterData валидируемый фильтр
     * @param {BEM.MODEL<dm-dynamic-group>} groupModel модель группы, которой принаджежит филь
     * @param {{ cid: String, ulogin: String }} meta метаданные запроса
     * @returns {Deferred<FilterValidationResult>}
     */
    validate: function(filterData, groupModel, meta) {
        var dataToSend = groupModel.provideData(),
            def = $.Deferred(),
            filterPosition = groupModel.get('feed_filters').length(),
            validationRequest = BEM.create('i-request_type_ajax', {
                url: '/registered/main.pl',
                type: 'POST',
                dataType: 'json',
                cache: false
            });

        //отправляем все фильтры, потому что нужна проверка на уникальность
        groupModel.get('feed_filters').some(function(item, index) {
            if (item.get('filter_id') == filterData.filter_id) {
                filterPosition = index;
                return true;
            }
            return false;
        });

        dataToSend.dynamic_conditions[filterPosition] = filterData;

        validationRequest.get({
            cmd: 'ajaxValidateDynamicConditions',
            cid: meta.cid,
            ulogin: meta.ulogin,
            json_groups: JSON.stringify([dataToSend])
        }, function(result) {
            def.resolve({
                validationResult: result,
                filterPosition: filterPosition
            });
        }, function(error) {
            def.reject(error);
        });

        return def;
    },

    /**
     * Конвертирует данные из пригодных для сохранения в dm в данные пригодные для валидации
     * @param {FilterEditData} filterModelData
     * @param {BEM.MODEL<dm-dynamic-group>} parentGroupModel
     * @returns {FilterData}
     */
    modelDataToValidationData: function(filterModelData, parentGroupModel) {
        var model = BEM.MODEL.create({
                id: 'validation',
                name: 'dm-feed-filter',
                parentName: 'dm-dynamic-group',
                parentId: parentGroupModel.id
            }, u._.assign({}, filterModelData)),
            filterData = model.provideData();

        filterData.filter_id = filterModelData.filter_id;
        filterData.adgroup_id = filterModelData.adgroup_id;
        model.destruct();

        return filterData;
    },

    /**
     * Обрабатывает результат валидации с помощью переданного контроллера
     * @param {Deferred<FilterValidationResult>} validationPromise
     * @param {FilterValidationController} controller
     * @returns {Deffered}
     */
    processValidationResult: function(validationPromise, controller) {
        return validationPromise.then(function(result) {
            var validationResult = result.validationResult,
                filterPosition = result.filterPosition,
                errors;

            if (validationResult.errors) {
                errors = u._.get(
                    validationResult,
                    [
                        'errors', 'groups', 'array_errors', 0, 'object_errors', 'dynamic_conditions',
                        'array_errors', filterPosition
                    ]
                );

                //alkaline@todo DIRECT-59737 передавать errors "как есть" и сделать попап редактирования фильтра "настоящим" i-error-view-interface
                if (!errors) {
                    errors = u._.flatten(u._.values(u.error.flatten(validationResult)));
                }
            }

            if (errors) {
                controller.showErrors(errors);
            } else {
                controller.save();
            }
        }).fail(function() {
            controller.onRequestError();
        });
    }
});
