(function() {
    /**
     * @typedef {Object} Range
     * @property {Number} min
     * @property {Number} max

     * @typedef {Object} RangeProcessed
     * @property {String} min
     * @property {String} max
     *
     * @typedef {Object} ClientCondition
     * @property {String} [id]
     * @property {String} field
     * @property {String} relation
     * @property {String|RangeProcessed} value
     *
     * @typedef {Object} ServerCondition
     * @property {String} [id]
     * @property {String} field
     * @property {String} relation
     * @property {String|Number} value
     */

    u.register({
        'i-filter-edit__preprocess': {
            /**
             * Применяет к массиву значений положенный для их типа препроцесс
             * @param {String} field
             * @param {String} relation
             * @param {Array<String|Number|Range>} valuesArray
             * @returns {Array<String|RangeProcessed>}
             */
            preprocessValuesArray: function(field, relation, valuesArray) {
                return valuesArray.map(function(value) {
                    return this.preprocessValue(field, relation, value)
                }, this);

            },

            /**
             * Применяет к значению положенные для его типа препроцесс
             * @param {String} field
             * @param {String} relation
             * @param {String|Number|Range} value
             * @returns {String|RangeProcessed}
             */
            preprocessValue: function(field, relation, value) {
                var preprocessRules = u['i-filter-edit'].getValuePreproccessRules(field, relation);

                if (preprocessRules.type === 'range') {
                    return {
                        min: formatNumber(value.min, preprocessRules.precision),
                        max: formatNumber(value.max, preprocessRules.precision)
                    }
                } else if (preprocessRules.type === 'number') {
                    return formatNumber(value, preprocessRules.precision)
                } else {
                    return value;
                }

            },

            /**
             * Преобразует список условий из формата, который понимает сервер, в формат клиента
             * @param {Array<ServerCondition>} conditions
             * @returns {Array<ClientCondition>}
             */
            serverToClient: function(conditions) {
                return conditions.map(function(cond) {
                    cond.value = cond.value || [];

                    var preprocessRules = u['i-filter-edit'].getValuePreproccessRules(cond.field, cond.relation),
                        value = cond.value.map(function(part) {
                            if (preprocessRules.type === 'range') {
                                var values = part.split('-');

                                return this.preprocessValue(cond.field, cond.relation, {
                                    min: values[0],
                                    max: values[1]
                                });
                            }

                            return this.preprocessValue(cond.field, cond.relation, part);
                        }, this);

                    return {
                        id: cond.id || u._.uniqueId('condition_id_'),
                        field: cond.field,
                        relation: cond.relation,
                        value: value
                    };
                }, this);
            },

            /**
             * Преобразует условия из клиентского формата в серверный
             * @param {Array<ClientCondition>} conditions
             * @param {Boolean} [applyFilter] - флаг "Отфильтровывать из списка условий условия с пустым value"
             * @returns {Array<ServerCondition>}
             */
            clientToServer: function(conditions, applyFilter) {
                if (applyFilter) {
                    conditions = conditions.filter(function(cond) {
                        //оставляем только непустые значения если пришел флаг  filterUndefined
                        return cond.value && cond.value.length;
                    });
                }

                conditions = conditions.map(function(cond) {
                    return {
                        field: cond.field,
                        relation: cond.relation,
                        value: preprocessValueToServer(cond.field, cond.relation, cond.value)
                    }
                });

                return conditions
            },

            preprocessValueToServer: preprocessValueToServer
        }
    });

    /**
     * Преобразует из клиентского в серверный формат конкретное значение
     * @param {String} field
     * @param {String} relation
     * @param {Array<String|RangeProcessed>} valuesArray
     * @returns {Array<String|Number>}
     */
    function preprocessValueToServer(field, relation, valuesArray) {
        var preprocessRules = u['i-filter-edit'].getValuePreproccessRules(field, relation),
            controlType = u['i-filter-edit'].getValueControlTypeByCondition({ field: field, relation: relation });

        return controlType === 'exists' ?
            valuesArray :
            (valuesArray || []).map(function(value) {
                if (preprocessRules.type === 'range') {
                    return value.min + '-' + value.max;
                } else if (preprocessRules.type === 'number') {
                    //сервер не ест строку для числового типа. Строки нужны  только на клиенте для правильного форматирования
                    return +value;
                } else {
                    return value;
                }
            });
    }

    /**
     * Округляет число с заданной точностью и чистит его от мусорных символов
     * @param {Number} val
     * @param {Number} [precision]
     * @returns {String}
     */
    function formatNumber(val, precision) {
        if (u._.isEmpty(val + '')) return '';

        val = val + '';

        //если precision 0 то это не цена и дополнительные преобразования не нужны
        if (!precision) return val;

        var rawValue = u.numberFormatter.clear(val, {
            fail: '',
            precision: precision
        });

        return u.numberFormatter.roundFormatted(rawValue, {
            precision: precision,
            fail: null
        });
    }
})();
