(function() {

    u.register({
        'i-filter-edit': {
            //конфиг один для всех экземпляров i-filter-edit на данной странице - по этому хранится в утилитах
            //если возникнет потребность в нескольких конфигах на одной странице - _config и loadConfig надо
            //будет перенести в i-filter-edit.js
            _schema: null,

            /**
             * Устанавливает схему данных
             * @param {Object} schema
             */
            setSchema: function(schema) {
                this._schema = schema;
            },

            getStatesMap: function() {
                var statesMap = { orderedFiltersNames: this._schema.definitions.itemsOrder },
                    fieldData,
                    relationsList;

                Object.keys(this._schema.definitions.rule).forEach(function(fieldName) {
                    fieldData = this._schema.definitions.rule[fieldName];
                    relationsList = fieldData.fields.relation.type.enum;

                    statesMap[fieldName] = {
                        orderedSubFiltersNames: relationsList
                    };

                    relationsList.forEach(function(relation) {
                        statesMap[fieldName][relation] = {
                            //в будущем тут могут быть constraints - но пока просто пустой объект
                        }
                    })
                }, this);

                return statesMap;
            },

            /**
             * Возвращает название типа фильтра
             * @param {String} field тип фильтра
             * @returns {String}
             */
            getFieldTitle: function(field) {
                return field;
            },

            /**
             * Возвращает "человеческое название" отношения
             * @param {String} relation
             * @returns {String}
             */
            getRelationTitle: function(relation) {
                return this._schema.definitions.translations.rule[relation];
            },

            /**
             * Возвращает скорректированное название условия если оно есть в translations.field
             * Иначе возвращает переданное значение
             * @param {String} conditionName
             * @returns {String}
             */
            getConditionTitle: function(conditionName) {
                return u._.get(this._schema.definitions.translations.field, conditionName) || conditionName;
            },

            getBooleanValueTitle: function(value) {
                return this._schema.definitions.translations.boolean[value] || value;
            },

            _applyValidationRule: function(rules, value) {
                var errors = [];

                rules.constraints && rules.constraints.every(function(rule) {
                    var currentResult = u['i-filter-edit__validator'].validate(rule, value);

                    !currentResult && errors.push({
                        rule: rule.name,
                        text: rule.error
                    });

                    //если не прошла валидация, то продолжаем валидировать только если не стоит флаг break (остановиться если не прошла валидация)
                    if (!currentResult) {
                        return !rule.break
                    }

                    return true;
                });

                return errors;
            },

            /**
             * Валидация по конфигу
             * @param {Object} condition
             * @param {String} condition.field
             * @param {*} condition.value
             * @param {String} condition.relation
             * @returns {{isValid: boolean, errors: Array}}
             */
            validateCondition: function(condition) {
                var field = condition.field,
                    relation = condition.relation,
                    isComposite = this.isValueCompositeList(condition),
                    isValid = true,
                    rules = this._getValuesRulesForCondition(field, relation),
                    itemErrors,
                    errors = {
                        total: [],
                        items: {}
                    };

                //применяем правила валидации для всего условия
                errors.total = errors.total.concat(this._applyValidationRule(rules, condition.value));
                isValid = isValid && !errors.total.length;

                //валидируем каждое значение массива отдельно если есть rules.items - правила валидации на отдельный элемент массива
                rules.items && condition.value && u._.forEach(condition.value, function(value, index) {
                    itemErrors = this._applyValidationRule(rules.items, value);
                    //если контрол композит - то записываем ошибки для каждого item-а  в отдельный массив
                    if (isComposite) {
                        errors.items[index] = itemErrors;
                    } else {
                        errors.total = errors.total.concat(itemErrors);
                    }

                    isValid = isValid && !itemErrors.length;
                }, this);

                return {
                    isValid: isValid,
                    errors: errors
                };

            },

            /**
             *
             * @param {Array} conditionList
             * @returns {{isValid: boolean, errors: Array}}
             */
            validateConditionsList: function(conditionList) {
                var rules = this._schema.checkedStruct.constraints,
                    errors = [];

                rules.forEach(function(rule) {
                    var currentResult = u['i-filter-edit__validator'].validate(rule, conditionList);

                    !currentResult && errors.push({
                        rule: rule.name,
                        text: rule.error
                    });
                });

                return {
                    errors: errors,
                    isValid: !errors.length
                };
            },

            /**
             * Возвращает правила из схемы для работы с value для текущих field и relation
             * @param {String} field
             * @param {String} relation
             * @private
             */
            _getValuesRulesForCondition: function(field, relation) {
                var rulesForValue = this._schema.definitions.rule[field].fields.value;

                if (!rulesForValue.typeByCondition) return rulesForValue;

                //пытаемся найти ограничения на value зависящее от condition.relation
                var index = u._.findIndex(rulesForValue.typeByCondition, function(data) {
                    return data.condition.in.indexOf(relation) !== -1;
                });

                //если не находим подходящего правила в typeByCondition, то берем правило из секции default
                return index !== -1 ?
                    rulesForValue.typeByCondition[index] :
                    rulesForValue.default;
            },

            /**
             * condition.value - всегда массив элементов,
             * может показываться как поле ввода textarea (значения через запятую) или в виде отдельных строк с  (+)(-)
             *
             */
            isValueCompositeList: function(condition) {
                //по флагу isCompositeList определяется показывать поле ввода в виде textarea  или в виде отдельных строк с  (+)(-)
                return this._getValuesRulesForCondition(condition.field, condition.relation).isCompositeList;
            },

            /**
             * Возвращает тип поля ввода для value в зависимости от выбранного condition и имеющейся для него схемы
             * @param {Object} condition
             * @param {String} condition.field
             * @param {String} condition.relation
             * @param {*} condition.value
             * @returns {'exists'|'composite', 'textarea'}
             */
            getValueControlTypeByCondition: function(condition) {
                var rules = this._getValuesRulesForCondition(condition.field, condition.relation),
                    maxCount = this.getConditionMaxCount(condition);

                if (condition.relation == 'exists') {
                    return 'exists'
                } else if (rules.type === 'array' && rules.items.type.enum && maxCount == 1) {
                    return 'enum-one';
                } else if (rules.type === 'array' && rules.items.type.enum) {
                    return 'enum';
                } else if (maxCount == 1) {
                    return 'input-one';
                } else {
                    return this.isValueCompositeList(condition) ? 'composite' : 'textarea';
                }
            },

            /**
             * Возвращает дефолтное пустое значение для данного типа
             * @param {Object} condition
             * @returns {*}
             */
            getDefaultValue: function(condition) {
                var rules = this._getValuesRulesForCondition(condition.field, condition.relation),
                    maxCount = this.getConditionMaxCount(condition);

                if (rules.type === 'array') {
                    if (rules.items.type.enum) {
                        //для селекта у которого можно выбрать только одно значение
                        //по умолчанию выбираем первое
                        return maxCount === 1 ?
                            [rules.items.type.enum[0]] :
                            [];
                    } else {
                        //если значение - композит, то отрисовываем один элемент композита с пустым значением
                        if (this.isValueCompositeList(condition)) {
                            return this.isRangeValue(condition) ? [{ min: '', max: '' }] : [''];
                        } else {
                            return [];
                        }
                    }
                } else {
                    //exists
                    return ['1'];
                }
            },

            /**
             * Нужно ли изменять значение поля value после изменение relation
             * При изменении field значение не сохраняем
             * Применяем следующие правила:
             * если для текстового параметра выбрали "содержит", ввели значение, переключили на "не содержит" - значение остается
             * для числового параметра выбрали "больше", ввели значение, переключили на "меньше", "равно" - значение остается
             * для числового параметра выбрали "равно", ввели несколько значений через запятую, переключили на "меньше"/"больше" не сохранять значение
             * для селектов выбора из диапазона не сохраняем, тк там разные значения совсем
             * @param {Object} newCond
             * @param {Object} prevCond
             * @returns {boolean}
             */
            isValueNotNeededToChange: function(newCond, prevCond) {
                if (newCond.field !== prevCond.field) {
                    return false;
                }
                //При каких переходах сохраняем значение value:
                //'not ilike' -> 'ilike'
                //ilike -> 'not ilike'
                //'<' -> '>'
                //'>' -> '<'
                //'<' -> '=='
                //'>' -> '=='
                var needToSaveValue = {
                    'not ilike': ['ilike'],
                    ilike: ['not ilike'],
                    '<': ['>', '=='],
                    '>': ['<', '==']
                };

                if (needToSaveValue[prevCond.relation] &&
                    needToSaveValue[prevCond.relation].indexOf(newCond.relation) !== -1) {

                    return true;
                }

                return false;
            },

            getEnabledOptionsForEnumValue: function(condition) {
                var rules = this._getValuesRulesForCondition(condition.field, condition.relation);

                return rules.type === 'array' ? rules.items.type.enum : rules.type.enum;
            },

            getConditionMaxCount: function(condition) {
                var rules = this._getValuesRulesForCondition(condition.field, condition.relation),
                    maxCount;

                rules.constraints && rules.constraints.forEach(function(rule) {
                    if (rule.name === 'itemsCountMax') {
                        maxCount = rule.parameters;
                    }
                });

                return maxCount;
            },

            /**
             * Определяет, являются ли значение/элемент массива значения диапазоном
             * @param {Object} condition
             * @returns {boolean}
             */
            isRangeValue: function(condition) {
                var rules = this._getValuesRulesForCondition(condition.field, condition.relation);

                return rules.items.type === 'range';
            },

            /**
             * Возвращает номер элемента (начиная с нуля) после которого при отрисовке нужно вставить разделитель
             * @returns {*}
             */
            getFieldsListSeparator: function() {
                return this._schema.definitions.frontendExtendedFieldsFrom;
            },

            /**
             * По условию condition формирует массив с данными необходимыми для препроцессинга значений
             * @param {String} field
             * @param {String} relation
             * @returns {{}}
             */
            getValuePreproccessRules: function(field, relation) {
                var rules = this._getValuesRulesForCondition(field, relation),
                    type = rules.type.enum ? 'enum' : rules.items.type;

                if (type === 'range') {
                    return {
                        //форматируем как диапазон чисел с заданной точностью
                        type: 'range',
                        precision: rules.items.frontendPrecision
                    }
                } else if (type === 'number' || type === 'integer' || type === 'string_or_number') {
                    return {
                        //форматируем как число с заданной точностью
                        type: 'number',
                        precision: rules.items.frontendPrecision || 0
                    }
                } else {
                    return {
                        //форматировать не надо никак
                        type: 'other'
                    }
                }
            }
        }
    });

})();
