/**
 * Блок-обработчик для кнопки "Изменить счетчик метрики" в массовых действиях на кампании
 * Триггерит событие 'metrika-changed' для блока b-mass-actions_type_camps, который
 * уничтожает блок
 */
BEM.decl('i-mass-action-metrika-counters', {
    onSetMod: {
        js: function() {
            var params = this.params;

            this._affectedCids = params.affectedCids;
            this._unAffectedCids = params.unAffectedCids;
            this._defaultCounter = params.defaultCounter;
            this._performanceCids = params.performanceCids;
            this._performanceCount = params.performanceCids.length;
            this._readonly = params.readonly;

            this._subMan = BEM.create('i-subscription-manager');
        }
    },

    /**
     * Открывает попап со счетчиками метрики
     */
    openPopup: function() {
        this._initPopup().show();
    },

    /**
     * Деструтор
     */
    destruct: function() {
        this._subMan.dispose();
        this.__base.apply(this, arguments);
    },

    /**
     * инициализация попап
     * @returns {BEM}
     * @private
     */
    _initPopup: function() {
        var cids = this._affectedCids,
            performanceCount = this._performanceCount;

        this._popup = BEM.DOM.blocks['b-modal-popup-decorator'].create(null, { bodyScroll: false });
        this._metrikaCounters = this._popup.setPopupContent({
            block: 'b-metrika-counters-popup',
            mods: { type: 'mass-actions' },
            isPerformance: performanceCount == cids.length,
            needCheckCountersOwner: true,
            counters: this._defaultCounter,
            readonly: this._readonly
        });
        // подготовка формы списка счетчиков метрики
        this._metrikaCounters.prepareData();

        this._initEvents();

        return this._popup;
    },

    /**
     * инициализирует события на попапе
     * @private
     */
    _initEvents: function() {
        var popup = this._popup,
            metrikaCounters = this._metrikaCounters;

        this._subMan.wrap(metrikaCounters)
            .on('replace', function(event, counters) {
                this._onMetrikaCountersAction('replace', counters);
            }, this)
            .on('add', function(event, counters) {
                this._onMetrikaCountersAction('add', counters);
            }, this)
            .on('deleteAll', function() {
                this._onMetrikaCountersAction('deleteAll');
            }, this)
            .on('cancel', function() {
                popup.hide({ force: true });
            }, this)
            .on('buttons-disabled', function(e, data) {
                // прячем крестик на попапе
                popup.toggleClose(data.disabled);
            }, this);

        this._subMan.wrap(popup)
            .on('close-blocked', function() {
                BEM.blocks['b-user-dialog']
                    .confirm({
                        message: iget2('i-mass-action-metrika-counters', 'u-vas-est-nesohranennye', 'У вас есть несохраненные изменения. Продолжить?'),
                        confrimButtonText: iget2('i-mass-action-metrika-counters', 'da', 'Да'),
                        cancelButtonText: iget2('i-mass-action-metrika-counters', 'net', 'Нет'),
                        parentPopup: popup.getPopup(),
                        onConfirm: function() {
                            popup.hide({ force: true });
                        },
                        callbackCtx: this
                    });
            }, this)
            .on('close', function() {
                popup.destruct();
                this.trigger('closed');
            }, this);

    },

    /**
     * Метод, возвращающий массив счетчиков по строке
     * Java-ручки требуют строго массив, Perl-овые принимают строку
     * При этом в Perl можно отправлять строки с пробелами и конечными запятыми
     * @param {String} countersStr - строка счетчиков
     * @returns {Array}
     * @private
     */
    _getCountersFromString: function(countersStr) {
        if (!countersStr) return [];

        var countersArray = countersStr.trim() ?
            countersStr.split(/[,\s]+/g) :
            [];

        return u._.filter(countersArray, function(counter) {
            return counter != '';
        });
    },

    /**
     * Обработка кнопок на попапе (добавить-заменить-удалить)
     * @param {String} action - тип действия над попапом
     * @param {String} counters - список новых счетчиков
     * @private
     */
    _onMetrikaCountersAction: function(action, counters) {
        var countersArray = this._getCountersFromString(counters),
            countersCount = countersArray.length;

        BEM.blocks['b-metrika2'].params({
            params: {
                ShowCamps: {
                    massAction: 'editMetrikaCounters' + action
                }
            }
        });

        if (this._performanceCount > 0 && countersCount > 1 && action == 'replace') {
            this._alertPerformance(action, countersCount, function() {
                this._saveMetrikaCounters(action, countersArray);
            }.bind(this));
        } else {
            this._saveMetrikaCounters(action, countersArray);
        }
    },

    /**
     * Отправка на сервер действий над счетчиками и обработка результата
     * @param {String} action - тип действия над попапом
     * @param {Array} countersArray - список новых счетчиков
     * @returns {Promise}
     * @private
     */
    _saveMetrikaCounters: function(action, countersArray) {
        var cids = this._affectedCids,
            unAffectedCids = this._unAffectedCids,
            popup = this._popup,
            metrikaAction = BEM.blocks['i-web-api-request'].metrikaCounters[action],
            text = this._createText('success', action, this._affectedCids, countersArray.length);

        if (unAffectedCids.length > 0) {
            text += '<br>';
            text += this._createText('unAffected', action, unAffectedCids, countersArray.length);
        }

        return metrikaAction({
            ulogin: u.consts('ulogin'),
            cids: cids,
            counters: countersArray
        })
            .then(function(data) {
                if (data.success) {
                    BEM.blocks['b-user-dialog'].alert({
                        message: text,
                        onCancel: function() {
                            popup.hide({ force: true });
                        }
                    });
                } else {
                    this._alertErrors(action, data.validation_result.errors, countersArray.length);
                }
            }.bind(this))
            .catch(function() {
                throw iget2('i-mass-action-metrika-counters', 'proizoshla-neizvestnaya-oshibka', 'Произошла неизвестная ошибка');
            });
    },

    /**
     * Предупреждение для Смарт-кампаний (у них строго 1 счетчик всегда, можно только заменять)
     * @param {String} action - тип действия над счетчиками
     * @param {Number} countersCount - количество счетчиков
     * @param {Function} callback - действие при подтверждении
     * @private
     */
    _alertPerformance: function(action, countersCount, callback) {
        var popup = this._popup,
            text = this._createText('performance', action, this._performanceCids, countersCount),
            metrikaCounters = this._metrikaCounters;

        BEM.blocks['b-user-dialog']
            .confirm({
                message: text,
                confrimButtonText: iget2('i-mass-action-metrika-counters', 'da', 'Да'),
                cancelButtonText: iget2('i-mass-action-metrika-counters', 'net', 'Нет'),
                parentPopup: popup.getPopup(),
                onConfirm: callback,
                onCancel: function() {
                    metrikaCounters.onCountersSaveFinished();
                },
                callbackCtx: this
            });
    },

    /**
     * Отображение ошибок от сервера
     * @param {String} action - тип действия над счетчиками
     * @param {Array} errors - список ошибок
     * @param {Number} countersCount - количество счетчиков
     * @private
     */
    _alertErrors: function(action, errors, countersCount) {
        var unAffectedCids = this._unAffectedCids,
            errorsGroups = u._.groupBy(errors, function(error) {
                return error.code;
            }),
            errorCids = [],
            errorsTexts = u._.map(errorsGroups, function(errorGroup, errorCode) {
                var cids = errorGroup
                    .map(function(error) {
                        var num = error.path.replace(/([a-z]+)\[(\d+)\]/, '$2');

                        return this._affectedCids[num];
                    }.bind(this))
                    .filter(function(cid) {
                        return !!cid;
                    });
                errorCids = errorCids.concat(cids);

                return this._createText(errorCode, action, cids, countersCount);
            }.bind(this));

        if (unAffectedCids.length > 0) {
            errorsTexts.push(this._createText('unAffected', action, unAffectedCids, countersCount));
        }

        if (this._affectedCids.length > errorCids.length && errorCids.length > 0) {
            errorsTexts.push(this._createText('partSuccess', action,
                u._.difference(this._affectedCids, errorCids), countersCount));
        }

        BEM.blocks['b-user-dialog'].alert({
            message: errorsTexts.join('<br/>'),
            onCancel: function() {
                this._popup.hide({ force: true });
            }.bind(this)
        });
    },

    /**
     * Метод, генерирующий все тексты предупреждения
     * @param {String} code - код (про что текст)
     * @param {String} action - действие
     * @param {Array} cids - массив cid-ов
     * @param {Number} countersCount - количество счетчиков
     * @returns {String}
     * @private
     */
    _createText: function(code, action, cids, countersCount) {
        var cidsKey = cids.length > 1 ? 'cids' : 'cid',
            countersKey = (countersCount > 1 || countersCount == 0) ? 'counters' : 'counter',
            cidsTexts = cids.join(', '),

            texts = {
                success: {
                    add: {
                        cid: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'ukazannyy-schyotchik-metriki-dobavlen',
                                'Указанный счётчик Метрики добавлен в выбранную кампанию.'
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'ukazannye-schyotchiki-metriki-dobavleny',
                                'Указанные счётчики Метрики добавлены в выбранную кампанию.'
                            )
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'ukazannyy-schyotchik-metriki-dobavlen-100',
                                'Указанный счётчик Метрики добавлен в выбранные кампании.'
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'ukazannye-schyotchiki-metriki-dobavleny-101',
                                'Указанные счётчики Метрики добавлены в выбранные кампании.'
                            )
                        }
                    },
                    replace: {
                        cid: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'ukazannyy-schyotchik-metriki-ustanovlen',
                                'Указанный счётчик Метрики установлен в выбранную кампанию.'
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'ukazannye-schyotchiki-metriki-ustanovleny',
                                'Указанные счётчики Метрики установлены в выбранную кампанию.'
                            )
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'ukazannyy-schyotchik-metriki-ustanovlen-102',
                                'Указанный счётчик Метрики установлен в выбранные кампании.'
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'ukazannye-schyotchiki-metriki-ustanovleny-103',
                                'Указанные счётчики Метрики установлены в выбранные кампании.'
                            )
                        }
                    },
                    deleteAll: {
                        cid: {
                            counters: iget2('i-mass-action-metrika-counters', 'schyotchiki-metriki-udaleny-iz', 'Счётчики Метрики удалены из выбранной кампании.')
                        },
                        cids: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchiki-metriki-udaleny-iz-104',
                                'Счётчики Метрики удалены из выбранных кампаний.'
                            )
                        }
                    }
                },
                partSuccess: {
                    add: {
                        cid: {
                            counter: iget2('i-mass-action-metrika-counters', 'v-kampaniyu-s-schyotchik', 'В кампанию {foo} счётчик Метрики был добавлен.', {
                                foo: cidsTexts
                            }),
                            counters: iget2('i-mass-action-metrika-counters', 'v-kampaniyu-s-schyotchiki', 'В кампанию {foo} счётчики Метрики были добавлены.', {
                                foo: cidsTexts
                            })
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'v-ostalnye-vybrannye-kampanii',
                                'В остальные выбранные кампании счётчик Метрики был добавлен.'
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'v-ostalnye-vybrannye-kampanii-105',
                                'В остальные выбранные кампании счётчики Метрики были добавлены.'
                            )
                        }
                    },
                    replace: {
                        cid: {
                            counter: iget2('i-mass-action-metrika-counters', 'v-kampanii-s-schyotchik', 'В кампании {foo} счётчик Метрики был изменён.', {
                                foo: cidsTexts
                            }),
                            counters: iget2('i-mass-action-metrika-counters', 'v-kampanii-s-schyotchiki', 'В кампании {foo} счётчики Метрики были изменены.', {
                                foo: cidsTexts
                            })
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'v-ostalnyh-vybrannyh-kampaniyah',
                                'В остальных выбранных кампаниях счётчик Метрики был изменён.'
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'v-ostalnyh-vybrannyh-kampaniyah-106',
                                'В остальных выбранных кампаниях счётчики Метрики были изменены.'
                            )
                        }
                    },
                    deleteAll: {
                        cid: {
                            counters: iget2('i-mass-action-metrika-counters', 'iz-kampanii-s-schyotchiki', 'Из кампании {foo} счётчики Метрики были удалены.', {
                                foo: cidsTexts
                            })
                        },
                        cids: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'iz-ostalnyh-vybrannyh-kampaniy',
                                'Из остальных выбранных кампаний счётчики Метрики были удалены.'
                            )
                        }
                    }
                },
                unAffected: {
                    add: {
                        cid: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchik-ne-byl-dobavlen',
                                'Счётчик не был добавлен к кампании {foo}, так как для неё действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchiki-ne-byli-dobavleny',
                                'Счётчики не были добавлены к кампании {foo}, так как для неё действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchik-ne-byl-dobavlen-107',
                                'Счётчик не был добавлен к кампаниям {foo}, так как для них действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchiki-ne-byli-dobavleny-108',
                                'Счётчики не были добавлены к кампаниям {foo}, так как для них действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    },
                    replace: {
                        cid: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchik-ne-byl-zamenyon',
                                'Счётчик не был заменён в кампании {foo}, так как для неё действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchiki-ne-byli-zameneny',
                                'Счётчики не были заменены в кампании {foo}, так как для неё действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchik-ne-byl-zamenyon-109',
                                'Счётчик не был заменён в кампаниях {foo}, так как для них действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchiki-ne-byli-zameneny-110',
                                'Счётчики не были заменены в кампаниях {foo}, так как для них действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    },
                    deleteAll: {
                        cid: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchiki-ne-byli-udaleny',
                                'Счётчики не были удалены у кампании {foo}, так как для неё действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchiki-ne-byli-udaleny-111',
                                'Счётчики не были удалены у кампаний {foo}, так как для них действие неприменимо.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    }
                },
                performance: {
                    replace: {
                        cid: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-odnoy-kampanii-smart',
                                'Для одной кампании Смарт-баннеров можно указать только один счётчик. В кампании Смарт-баннеров {foo} счётчик Метрики будет заменён на первый из списка. Продолжить?',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-odnoy-kampanii-smart-112',
                                'Для одной кампании Смарт-баннеров можно указать только один счётчик. В кампаниях Смарт-баннеров {foo} счётчик Метрики будет заменён на первый из списка. Продолжить?',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    }
                },
                'CampaignDefectIds.Nums.MAX_METRIKA_COUNTERS_LIST_SIZE': {
                    add: {
                        cid: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'ne-udalos-dobavit-schyotchik',
                                'Не удалось добавить счётчик Метрики к кампании {foo}. Для каждой кампании возможно добавить не более 5 счётчиков.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'ne-udalos-dobavit-schyotchiki',
                                'Не удалось добавить счётчики Метрики к кампании {foo}. Для каждой кампании возможно добавить не более 5 счётчиков.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'ne-udalos-dobavit-schyotchik-113',
                                'Не удалось добавить счётчик Метрики к кампаниям {foo}. Для каждой кампании возможно добавить не более 5 счётчиков.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'ne-udalos-dobavit-schyotchiki-114',
                                'Не удалось добавить счётчики Метрики к кампаниям {foo}. Для каждой кампании возможно добавить не более 5 счётчиков.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    }
                },
                'CampaignDefectIds.Gen.CANT_ADD_OR_DELETE_METRIKA_COUNTERS_TO_PERFORMANCE_CAMPAIGN': {
                    add: {
                        cid: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'novyy-schyotchik-metriki-ne',
                                'Новый счётчик Метрики не был добавлен к кампании Смарт-баннеров {foo}. Для одной кампании Смарт-баннеров можно указать только один счётчик.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'novye-schyotchiki-metriki-ne',
                                'Новые счётчики Метрики не были добавлены к кампании Смарт-баннеров {foo}. Для одной кампании Смарт-баннеров можно указать только один счётчик.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'novyy-schyotchik-metriki-ne-115',
                                'Новый счётчик Метрики не был добавлен к кампаниям Смарт-баннеров {foo}. Для одной кампании Смарт-баннеров можно указать только один счётчик.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'novye-schyotchiki-metriki-ne-116',
                                'Новые счётчики Метрики не были добавлены к кампаниям Смарт-баннеров {foo}. Для одной кампании Смарт-баннеров можно указать только один счётчик.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    },
                    deleteAll: {
                        cid: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchik-metriki-ne-byl',
                                'Счётчик Метрики не был удалён у кампании Смарт-баннеров {foo}. Счётчик Метрики обязательное поле у кампаний Смарт-баннеров.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'schyotchik-metriki-ne-byl-117',
                                'Счётчик Метрики не был удалён у кампаний Смарт-баннеров {foo}. Счётчик Метрики обязательное поле у кампаний Смарт-баннеров.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    }
                },
                'CollectionDefectIds.CANNOT_CONTAIN_NULL': {
                    add: {
                        cid: {
                            counter: iget2('i-mass-action-metrika-counters', 'nevernye-vhodnye-dannye', 'Неверные входные данные.'),
                            counters: iget2('i-mass-action-metrika-counters', 'nevernye-vhodnye-dannye', 'Неверные входные данные.')
                        }
                    },
                    replace: {
                        cid: {
                            counter: iget2('i-mass-action-metrika-counters', 'nevernye-vhodnye-dannye', 'Неверные входные данные.'),
                            counters: iget2('i-mass-action-metrika-counters', 'nevernye-vhodnye-dannye', 'Неверные входные данные.')
                        }
                    },
                    deleteAll: {
                        cid: {
                            counter: iget2('i-mass-action-metrika-counters', 'nevernye-vhodnye-dannye', 'Неверные входные данные.'),
                            counters: iget2('i-mass-action-metrika-counters', 'nevernye-vhodnye-dannye', 'Неверные входные данные.')
                        }
                    }
                },
                'CollectionDefectIds.SIZE_CANNOT_BE_LESS_THAN_MIN': {
                    add: {
                        cid: {
                            counters: iget2('i-mass-action-metrika-counters', 'ne-ukazan-schyotchik-metriki', 'Не указан счётчик Метрики, добавление невозможно.')
                        }
                    },
                    replace: {
                        cid: {
                            counters: iget2('i-mass-action-metrika-counters', 'ne-ukazan-schyotchik-metriki-118', 'Не указан счётчик Метрики, замена невозможна.')
                        }
                    }
                },
                'CampaignDefectIds.Gen.METRIKA_COUNTER_IS_UNAVAILABLE': {
                    add: {
                        cid: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampanii-s-dobavlenie',
                                'Для кампании {foo} добавление счётчика невозможно, так как счётчик недоступен.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampanii-s-dobavlenie-119',
                                'Для кампании {foo} добавление счётчиков невозможно, так как счётчики недоступны.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampaniy-s-dobavlenie',
                                'Для кампаний {foo} добавление счётчика невозможно, так как счётчик недоступен.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampaniy-s-dobavlenie-120',
                                'Для кампаний {foo} добавление счётчиков невозможно, так как счётчики недоступны.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    },
                    replace: {
                        cid: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampanii-s-zamena',
                                'Для кампании {foo} замена счётчика невозможна, так как счётчик недоступен.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampanii-s-zamena-121',
                                'Для кампании {foo} замена счётчиков невозможна, так как счётчики недоступны.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counter: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampaniy-s-zamena',
                                'Для кампаний {foo} замена счётчика невозможна, так как счётчик недоступен.',
                                {
                                    foo: cidsTexts
                                }
                            ),
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampaniy-s-zamena-122',
                                'Для кампаний {foo} замена счётчиков невозможна, так как счётчики недоступны.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    },
                    deleteAll: {
                        cid: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampanii-s-udalenie',
                                'Для кампании {foo} удаление счётчиков невозможно, так как счётчики недоступны.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        },
                        cids: {
                            counters: iget2(
                                'i-mass-action-metrika-counters',
                                'dlya-kampaniy-s-udalenie',
                                'Для кампаний {foo} удаление счётчиков невозможно, так как счётчики недоступны.',
                                {
                                    foo: cidsTexts
                                }
                            )
                        }
                    }
                }
            };

        return (((texts[code] || {})[action] || {})[cidsKey] || {})[countersKey] || iget2('i-mass-action-metrika-counters', 'neizvestnaya-oshibka', 'Неизвестная ошибка');
    }
});
