describe('i-mass-action-metrika-counters', function() {
    var block,
        defaultCtx = {
            affectedCids: ['12345', '67890'],
            unAffectedCids: [],
            performanceCids: [],
            defaultCounter: '1234567890'
        },
        sandbox,
        spyDialog = function(type) {
            sandbox.spy(BEM.blocks['b-user-dialog'], type);
        },
        restoreDialog = function(type) {
            BEM.blocks['b-user-dialog'][type].restore();
        },
        stubRequest = function(action, response) {
            sandbox.stub(BEM.blocks['i-web-api-request'].metrikaCounters, action).callsFake(function() {
                return {
                    then: function(resolver) {
                        resolver(response);

                        return {
                            catch: function() {}
                        }
                    }
                }
            });
        };

    beforeEach(function() {
        sandbox = sinon.sandbox.create({ useFakeTimers: true });
    });

    afterEach(function() {
        sandbox.restore();
    });

    describe('Обработка успешных ответов от сервера', function() {
        describe('в 1+ кампании', function() {
            beforeEach(function() {
                spyDialog('alert');
                block = BEM.create('i-mass-action-metrika-counters', defaultCtx);
                block.openPopup();

                stubRequest('add', { success: true });
                stubRequest('replace', { success: true });
                stubRequest('deleteAll', { success: true });
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            it('При успешном добавлении 1 счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('add', '1234567890');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Указанный счётчик Метрики добавлен в выбранные кампании.');
            });

            it('При успешном добавлении 2 счетчиков показывается сообщение', function() {
                block._metrikaCounters.trigger('add', '1234567890, 24343434');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Указанные счётчики Метрики добавлены в выбранные кампании.');
            });

            it('При успешном удаленни показывается сообщение', function() {
                block._metrikaCounters.trigger('deleteAll');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчики Метрики удалены из выбранных кампаний.');
            });

            it('При успешной замене на 1 счетчик показывается сообщение', function() {
                block._metrikaCounters.trigger('replace', '1234567');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Указанный счётчик Метрики установлен в выбранные кампании.');
            });

            it('При успешной замене на 2 счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('replace', '1234567, 2112424');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Указанные счётчики Метрики установлены в выбранные кампании.');
            });
        });

        describe('в 1 кампанию', function() {
            beforeEach(function() {
                spyDialog('alert');
                block = BEM.create('i-mass-action-metrika-counters',
                    u._.extend({}, defaultCtx, { affectedCids: ['12345'] }));
                block.openPopup();

                stubRequest('add', { success: true });
                stubRequest('replace', { success: true });
                stubRequest('deleteAll', { success: true });
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            it('При успешном добавлении 1 счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('add', '1234567890');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Указанный счётчик Метрики добавлен в выбранную кампанию.');
            });

            it('При успешном добавлении 2 счетчиков показывается сообщение', function() {
                block._metrikaCounters.trigger('add', '1234567890, 24343434');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Указанные счётчики Метрики добавлены в выбранную кампанию.');
            });

            it('При успешном удаленни показывается сообщение', function() {
                block._metrikaCounters.trigger('deleteAll');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчики Метрики удалены из выбранной кампании.');
            });

            it('При успешной замене на 1 счетчик показывается сообщение', function() {
                block._metrikaCounters.trigger('replace', '1234567');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Указанный счётчик Метрики установлен в выбранную кампанию.');
            });

            it('При успешной замене на 2 счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('replace', '1234567, 2112424');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Указанные счётчики Метрики установлены в выбранную кампанию.');
            });
        });
    });

    describe('Обработка ошибок от сервера', function() {
        describe('При добавлении "лишнего" (5+) счетчика показывается сообщение об ошибке', function() {
            beforeEach(function() {
                spyDialog('alert');
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            describe('Для 1 кампании', function() {
                beforeEach(function() {
                    var errorMsg = {
                        "validation_result": {
                            "errors": [
                                {
                                    "path": "cids[0]",
                                    "code": "CampaignDefectIds.Nums.MAX_METRIKA_COUNTERS_LIST_SIZE",
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                }
                            ]
                        },
                        "success": false
                    };

                    stubRequest('add', errorMsg);
                    block = BEM.create('i-mass-action-metrika-counters',
                        u._.extend({}, defaultCtx, { affectedCids: ['12345'] }));
                    block.openPopup();
                });

                it('1 счетчик', function() {
                    block._metrikaCounters.trigger('add', '1234567890');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Не удалось добавить счётчик Метрики к кампании 12345. Для каждой кампании возможно добавить не более 5 счётчиков.');
                });

                it('2 счетчика', function() {
                    block._metrikaCounters.trigger('add', '1234567890, 333445566');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Не удалось добавить счётчики Метрики к кампании 12345. Для каждой кампании возможно добавить не более 5 счётчиков.');
                });
            });

            describe('Для 1+ кампании', function() {
                beforeEach(function() {
                    var errorMsg = {
                        "validation_result": {
                            "errors": [
                                {
                                    "path": "cids[0]",
                                    "code": "CampaignDefectIds.Nums.MAX_METRIKA_COUNTERS_LIST_SIZE",
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                },
                                {
                                    "path": "cids[1]",
                                    "code": "CampaignDefectIds.Nums.MAX_METRIKA_COUNTERS_LIST_SIZE",
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                }
                            ]
                        },
                        "success": false
                    };

                    stubRequest('add', errorMsg);
                    block = BEM.create('i-mass-action-metrika-counters', defaultCtx);
                    block.openPopup();
                });

                it('1 счетчик', function() {
                    block._metrikaCounters.trigger('add', '1234567890');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Не удалось добавить счётчик Метрики к кампаниям 12345, 67890. Для каждой кампании возможно добавить не более 5 счётчиков.');
                });

                it('2 счетчика', function() {
                    block._metrikaCounters.trigger('add', '1234567890, 3344566');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Не удалось добавить счётчики Метрики к кампаниям 12345, 67890. Для каждой кампании возможно добавить не более 5 счётчиков.');
                });
            });
        });

        describe('При невалидных данных', function() {
            beforeEach(function() {
                spyDialog('alert');
                block = BEM.create('i-mass-action-metrika-counters', defaultCtx);
                block.openPopup();

                var errorMsg = {
                    "validation_result": {
                        "errors": [
                            {
                                "path": "metrika_counters",
                                "code": "CollectionDefectIds.CANNOT_CONTAIN_NULL",
                                "text": "Неверное значение",
                                "params": {
                                    "max": 5,
                                    "min": null
                                },
                                "description": null
                            }
                        ]
                    },
                    "success": false
                };

                stubRequest('add', errorMsg);
                stubRequest('replace', errorMsg);
                stubRequest('deleteAll', errorMsg);
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            it('Обрабатывается ошибка CANNOT_CONTAIN_NULL при добавлении', function() {
                block._metrikaCounters.trigger('add', '1234567890');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Неверные входные данные.');
            });

            it('Обрабатывается ошибка CANNOT_CONTAIN_NULL при замене', function() {
                block._metrikaCounters.trigger('replace', '1234567890');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Неверные входные данные.');
            });

            it('Обрабатывается ошибка CANNOT_CONTAIN_NULL при удалении', function() {
                block._metrikaCounters.trigger('deleteAll', '1234567890');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Неверные входные данные.');
            });
        });

        describe('Отсутствие счетчика', function() {
            beforeEach(function() {
                spyDialog('alert');
                block = BEM.create('i-mass-action-metrika-counters', defaultCtx);
                block.openPopup();

                var errorMsg = {
                    "validation_result": {
                        "errors": [
                            {
                                "path": "metrika_counters",
                                "code": "CollectionDefectIds.SIZE_CANNOT_BE_LESS_THAN_MIN",
                                "text": "Неверное значение",
                                "params": {
                                    "max": 5,
                                    "min": null
                                },
                                "description": null
                            }
                        ]
                    },
                    "success": false
                };

                stubRequest('add', errorMsg);
                stubRequest('replace', errorMsg);
                stubRequest('deleteAll', errorMsg);
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            it('Обрабатывается ошибка при добавлении', function() {
                block._metrikaCounters.trigger('add', '');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Не указан счётчик Метрики, добавление невозможно.');
            });

            it('Обрабатывается ошибка при замене', function() {
                block._metrikaCounters.trigger('replace', '');

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Не указан счётчик Метрики, замена невозможна.');
            });
        });

        describe('Неизвестная ошибка', function() {
            beforeEach(function() {
                spyDialog('alert');

                var errorMsg = {
                    "validation_result": {
                        "errors": [
                            {
                                "path": "",
                                "code": "JUST_SOME_ERROR",
                                "text": "Неверное значение",
                                "params": {
                                    "max": 5,
                                    "min": null
                                },
                                "description": null
                            }
                        ]
                    },
                    "success": false
                };

                stubRequest('add', errorMsg);
                stubRequest('replace', errorMsg);
                stubRequest('deleteAll', errorMsg);
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            ['add', 'replace', 'deleteAll'].forEach(function(action) {
                [['12345'], ['12345', '67890']].forEach(function(cids) {
                    ['111', '111, 222'].forEach(function(counters, index) {
                        it('Для действия ' + action + ', ' +
                            cids.length +' кампании/кампаний, ' +
                            index + ' счетчика/счетчиков ' +
                            'обрабатывается неизвестная ошибка', function() {

                            block = BEM.create('i-mass-action-metrika-counters', u._.extend({}, defaultCtx, {
                                affectedCids: cids
                            }));
                            block.openPopup();

                            block._metrikaCounters.trigger(action, counters);

                            expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                                .to.include('Неизвестная ошибка');
                        });
                    });
                });
            });
        });

        describe('Смарт-баннеры', function() {
            beforeEach(function() {
                spyDialog('alert');
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            describe('При добавлении счетчика показывается сообщение об ошибке', function() {
                describe('Для 1 кампании', function() {
                    beforeEach(function() {
                        var errorMsg = {
                            "validation_result": {
                                "errors": [
                                    {
                                        "path": "cids[0]",
                                        // eslint-disable-next-line max-len
                                        "code": "CampaignDefectIds.Gen.CANT_ADD_OR_DELETE_METRIKA_COUNTERS_TO_PERFORMANCE_CAMPAIGN",
                                        "text": "Неверное значение",
                                        "params": {
                                            "max": 5,
                                            "min": null
                                        },
                                        "description": null
                                    }
                                ]
                            },
                            "success": false
                        };

                        stubRequest('add', errorMsg);
                        sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                            options.onConfirm.apply(options.callbackCtx);
                        });
                        block = BEM.create('i-mass-action-metrika-counters', u._.extend({}, defaultCtx, {
                            performanceCids: ['12345'],
                            affectedCids: ['12345']
                        }));
                        block.openPopup();
                    });

                    it('для 1 счетчика', function() {
                        block._metrikaCounters.trigger('add', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Новый счётчик Метрики не был добавлен к кампании Смарт-баннеров 12345. Для одной кампании Смарт-баннеров можно указать только один счётчик.');
                    });

                    it('для 2 счетчиков', function() {
                        block._metrikaCounters.trigger('add', '1234567890, 554556');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Новые счётчики Метрики не были добавлены к кампании Смарт-баннеров 12345. Для одной кампании Смарт-баннеров можно указать только один счётчик.');
                    });
                });

                describe('Для 1+ кампании', function() {
                    beforeEach(function() {
                        var errorMsg = {
                            "validation_result": {
                                "errors": [
                                    {
                                        "path": "cids[0]",
                                        // eslint-disable-next-line max-len
                                        "code": "CampaignDefectIds.Gen.CANT_ADD_OR_DELETE_METRIKA_COUNTERS_TO_PERFORMANCE_CAMPAIGN",
                                        "text": "Неверное значение",
                                        "params": {
                                            "max": 5,
                                            "min": null
                                        },
                                        "description": null
                                    },
                                    {
                                        "path": "cids[1]",
                                        // eslint-disable-next-line max-len
                                        "code": "CampaignDefectIds.Gen.CANT_ADD_OR_DELETE_METRIKA_COUNTERS_TO_PERFORMANCE_CAMPAIGN",
                                        "text": "Неверное значение",
                                        "params": {
                                            "max": 5,
                                            "min": null
                                        },
                                        "description": null
                                    }
                                ]
                            },
                            "success": false
                        };

                        stubRequest('add', errorMsg);
                        sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                            options.onConfirm.apply(options.callbackCtx);
                        });

                        block = BEM.create('i-mass-action-metrika-counters', u._.extend({}, defaultCtx, {
                            performanceCids: ['12345', '67890'],
                            affectedCids: ['12345', '67890']
                        }));
                        block.openPopup();
                    });

                    it('для 1 счетчика', function() {
                        block._metrikaCounters.trigger('add', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Новый счётчик Метрики не был добавлен к кампаниям Смарт-баннеров 12345, 67890. Для одной кампании Смарт-баннеров можно указать только один счётчик.');
                    });

                    it('для 2 счетчиков', function() {
                        block._metrikaCounters.trigger('add', '1234567890, 4443434');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Новые счётчики Метрики не были добавлены к кампаниям Смарт-баннеров 12345, 67890. Для одной кампании Смарт-баннеров можно указать только один счётчик.');
                    });
                });
            });

            describe('При удалении счетчика показывается сообщение об ошибке', function() {
                it('Для 1 кампании', function() {
                    var errorMsg = {
                        "validation_result": {
                            "errors": [
                                {
                                    "path": "cids[0]",
                                    // eslint-disable-next-line max-len
                                    "code": "CampaignDefectIds.Gen.CANT_ADD_OR_DELETE_METRIKA_COUNTERS_TO_PERFORMANCE_CAMPAIGN",
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                }
                            ]
                        },
                        "success": false
                    };

                    stubRequest('deleteAll', errorMsg);
                    sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                        options.onConfirm.apply(options.callbackCtx);
                    });
                    block = BEM.create('i-mass-action-metrika-counters', u._.extend({}, defaultCtx, {
                        performanceCids: ['12345'],
                        affectedCids: ['12345']
                    }));
                    block.openPopup();

                    block._metrikaCounters.trigger('deleteAll');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Счётчик Метрики не был удалён у кампании Смарт-баннеров 12345. Счётчик Метрики обязательное поле у кампаний Смарт-баннеров.');
                });

                it('Для 1+ кампании', function() {
                    var errorMsg = {
                        "validation_result": {
                            "errors": [
                                {
                                    "path": "cids[0]",
                                    // eslint-disable-next-line max-len
                                    "code": "CampaignDefectIds.Gen.CANT_ADD_OR_DELETE_METRIKA_COUNTERS_TO_PERFORMANCE_CAMPAIGN",
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                },
                                {
                                    "path": "cids[1]",
                                    // eslint-disable-next-line max-len
                                    "code": "CampaignDefectIds.Gen.CANT_ADD_OR_DELETE_METRIKA_COUNTERS_TO_PERFORMANCE_CAMPAIGN",
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                }
                            ]
                        },
                        "success": false
                    };

                    stubRequest('deleteAll', errorMsg);
                    sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                        options.onConfirm.apply(options.callbackCtx);
                    });

                    block = BEM.create('i-mass-action-metrika-counters', u._.extend({}, defaultCtx, {
                        performanceCids: ['12345', '67890'],
                        affectedCids: ['12345', '67890']
                    }));
                    block.openPopup();

                    block._metrikaCounters.trigger('deleteAll');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Счётчик Метрики не был удалён у кампаний Смарт-баннеров 12345, 67890. Счётчик Метрики обязательное поле у кампаний Смарт-баннеров.');
                });
            });
        });

        describe('Частичный успех', function() {
            beforeEach(function() {
                var errorMsg = {
                    "validation_result": {
                        "errors": [
                            {
                                "path": "cids[0]",
                                // eslint-disable-next-line max-len
                                "code": "CampaignDefectIds.Gen.CANT_ADD_OR_DELETE_METRIKA_COUNTERS_TO_PERFORMANCE_CAMPAIGN",
                                "text": "Неверное значение",
                                "params": {
                                    "max": 5,
                                    "min": null
                                },
                                "description": null
                            }
                        ]
                    },
                    "success": false
                };

                stubRequest('add', errorMsg);
                stubRequest('replace', errorMsg);
                stubRequest('deleteAll', errorMsg);
                sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                    options.onConfirm.apply(options.callbackCtx);
                });
            });

            describe('1 успешная кампания', function() {
                beforeEach(function() {
                    block = BEM.create('i-mass-action-metrika-counters', u._.extend({}, defaultCtx, {
                        performanceCids: ['12345'],
                        affectedCids: ['12345', '67890']
                    }));
                    block.openPopup();
                    spyDialog('alert');
                });

                afterEach(function() {
                    block._popup.hide({ force: true });
                    block.destruct();
                    restoreDialog('alert');
                });

                it('При удалении показывается сообщение об успехе', function() {
                    block._metrikaCounters.trigger('deleteAll');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Из кампании 67890 счётчики Метрики были удалены.');
                });

                describe('1 счетчик', function() {
                    it('При добавлении показывается сообщение об успехе', function() {
                        block._metrikaCounters.trigger('add', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('В кампанию 67890 счётчик Метрики был добавлен.');
                    });

                    it('При замене показывается сообщение об успехе', function() {
                        block._metrikaCounters.trigger('replace', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('В кампании 67890 счётчик Метрики был изменён.');
                    });
                });

                describe('1+ счетчик', function() {
                    it('При добавлении показывается сообщение об успехе', function() {
                        block._metrikaCounters.trigger('add', '1234567890, 445');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('В кампанию 67890 счётчики Метрики были добавлены.');
                    });

                    it('При замене показывается сообщение об успехе', function() {
                        block._metrikaCounters.trigger('replace', '1234567890, 5545');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('В кампании 67890 счётчики Метрики были изменены.');
                    });
                });
            });

            describe('1+ успешная кампания', function() {
                beforeEach(function() {
                    block = BEM.create('i-mass-action-metrika-counters', u._.extend({}, defaultCtx, {
                        performanceCids: ['12345'],
                        affectedCids: ['12345', '67890', '88900']
                    }));
                    block.openPopup();
                    spyDialog('alert');
                });

                afterEach(function() {
                    block._popup.hide({ force: true });
                    block.destruct();
                    restoreDialog('alert');
                });

                it('При удалении показывается сообщение об успехе', function() {
                    block._metrikaCounters.trigger('deleteAll');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Из остальных выбранных кампаний счётчики Метрики были удалены.');
                });

                describe('1 счетчик', function() {
                    it('При добавлении показывается сообщение об успехе', function() {
                        block._metrikaCounters.trigger('add', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('В остальные выбранные кампании счётчик Метрики был добавлен.');
                    });

                    it('При замене показывается сообщение об успехе', function() {
                        block._metrikaCounters.trigger('replace', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('В остальных выбранных кампаниях счётчик Метрики был изменён.');
                    });
                });

                describe('1+ счетчик', function() {
                    it('При добавлении показывается сообщение об успехе', function() {
                        block._metrikaCounters.trigger('add', '1234567890, 445');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('В остальные выбранные кампании счётчики Метрики были добавлены.');
                    });

                    it('При замене показывается сообщение об успехе', function() {
                        block._metrikaCounters.trigger('replace', '1234567890, 5545');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('В остальных выбранных кампаниях счётчики Метрики были изменены');
                    });
                });
            });
        });

        describe('Недоступный счетчик', function() {
            beforeEach(function() {
                spyDialog('alert');
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            describe('Для 1 кампании', function() {
                beforeEach(function() {
                    var errorMsg = {
                        "validation_result": {
                            "errors": [
                                {
                                    "path": "cids[0]",
                                    "code": 'CampaignDefectIds.Gen.METRIKA_COUNTER_IS_UNAVAILABLE',
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                }
                            ]
                        },
                        "success": false
                    };

                    stubRequest('add', errorMsg);
                    stubRequest('replace', errorMsg);
                    stubRequest('deleteAll', errorMsg);

                    block = BEM.create('i-mass-action-metrika-counters',
                        u._.extend({}, defaultCtx, { affectedCids: ['12345'] }));
                    block.openPopup();
                });

                it('Удаление счетчиков', function() {
                    block._metrikaCounters.trigger('deleteAll');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Для кампании 12345 удаление счётчиков невозможно, так как счётчики недоступны.');
                });

                describe('1 счетчик', function() {
                    it('Добавление счетчиков', function() {
                        block._metrikaCounters.trigger('add', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Для кампании 12345 добавление счётчика невозможно, так как счётчик недоступен.');
                    });

                    it('Замена счетчиков', function() {
                        block._metrikaCounters.trigger('replace', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Для кампании 12345 замена счётчика невозможна, так как счётчик недоступен.');
                    });
                });

                describe('1+ счетчик', function() {
                    it('Добавление счетчиков', function() {
                        block._metrikaCounters.trigger('add', '1234567890, 7777');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Для кампании 12345 добавление счётчиков невозможно, так как счётчики недоступны.');
                    });

                    it('Замена счетчиков', function() {
                        block._metrikaCounters.trigger('replace', '1234567890, 7777');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Для кампании 12345 замена счётчиков невозможна, так как счётчики недоступны.');
                    });
                });
            });

            describe('Для 1+ кампании', function() {
                beforeEach(function() {
                    var errorMsg = {
                        "validation_result": {
                            "errors": [
                                {
                                    "path": "cids[0]",
                                    "code": 'CampaignDefectIds.Gen.METRIKA_COUNTER_IS_UNAVAILABLE',
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                },
                                {
                                    "path": "cids[1]",
                                    "code": 'CampaignDefectIds.Gen.METRIKA_COUNTER_IS_UNAVAILABLE',
                                    "text": "Неверное значение",
                                    "params": {
                                        "max": 5,
                                        "min": null
                                    },
                                    "description": null
                                }
                            ]
                        },
                        "success": false
                    };

                    stubRequest('add', errorMsg);
                    stubRequest('replace', errorMsg);
                    stubRequest('deleteAll', errorMsg);

                    block = BEM.create('i-mass-action-metrika-counters',
                        u._.extend({}, defaultCtx, { affectedCids: ['12345', '67890'] }));
                    block.openPopup();
                });

                it('Удаление счетчиков', function() {
                    block._metrikaCounters.trigger('deleteAll');

                    expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                        .to.include('Для кампаний 12345, 67890 удаление счётчиков невозможно, так как счётчики недоступны.');
                });

                describe('1 счетчик', function() {
                    it('Добавление счетчиков', function() {
                        block._metrikaCounters.trigger('add', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Для кампаний 12345, 67890 добавление счётчика невозможно, так как счётчик недоступен.');
                    });

                    it('Замена счетчиков', function() {
                        block._metrikaCounters.trigger('replace', '1234567890');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Для кампаний 12345, 67890 замена счётчика невозможна, так как счётчик недоступен.');
                    });
                });

                describe('1+ счетчик', function() {
                    it('Добавление счетчиков', function() {
                        block._metrikaCounters.trigger('add', '1234567890, 7777');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Для кампаний 12345, 67890 добавление счётчиков невозможно, так как счётчики недоступны.');
                    });

                    it('Замена счетчиков', function() {
                        block._metrikaCounters.trigger('replace', '1234567890, 7777');

                        expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                            .to.include('Для кампаний 12345, 67890 замена счётчиков невозможна, так как счётчики недоступны.');
                    });
                })
            })
        })
    });

    describe('Обработка событий попапа', function() {
        beforeEach(function() {
            block = BEM.create('i-mass-action-metrika-counters', defaultCtx);
            block.openPopup();

            sandbox.spy(BEM.blocks['i-web-api-request'].metrikaCounters, 'add');
            sandbox.spy(BEM.blocks['i-web-api-request'].metrikaCounters, 'replace');
            sandbox.spy(BEM.blocks['i-web-api-request'].metrikaCounters, 'deleteAll');
        });

        afterEach(function() {
            block._popup.hide({ force: true });
            block.destruct();
        });

        it('При нажатии на Добавить дергается ручка Добавления', function() {
            block._metrikaCounters.trigger('add', '1234567890');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.add.called).to.be.true;
        });

        it('При нажатии на Заменить дергается ручка Замены', function() {
            block._metrikaCounters.trigger('replace', '1234567890');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.replace.called).to.be.true;
        });

        it('При нажатии на Удалить дергается ручка Удаления', function() {
            block._metrikaCounters.trigger('deleteAll');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.deleteAll.called).to.be.true;
        });

        it('При отмене попап прячется', function() {
            block._metrikaCounters.trigger('cancel');

            expect(block._popup.isShown()).to.be.false;
        });

        it('При закрытии попапа попап уничтожается', function() {
            sandbox.spy(block._popup, 'destruct');
            block._metrikaCounters.trigger('cancel');

            expect(block._popup.destruct.called).to.be.true;
        });

        it('При закрытии попапа вызывается событие closed', function() {
            sandbox.spy(block, 'trigger');
            block._metrikaCounters.trigger('cancel');

            expect(block.trigger.calledWith('closed')).to.be.true;
        });

        it('По событию buttons-disabled вызывается метод отключения крестика у попапа', function() {
            sandbox.spy(block._popup, 'toggleClose');
            block._metrikaCounters.trigger('buttons-disabled', { disabled: true });

            expect(block._popup.toggleClose.called).to.be.true;
        });

        it('При закрытии без force:true при наличии изменений вызывается диалог подтверждения', function() {
            sandbox.stub(block._metrikaCounters, 'isChanged').callsFake(function() {
                var deferred = $.Deferred();

                deferred.resolve(true);

                return deferred;
            });
            spyDialog('confirm');

            block._popup.hide();
            expect(BEM.blocks['b-user-dialog'].confirm.called).to.be.true;
            restoreDialog('confirm');
        });

        it('При закрытии без force:true без изменений не вызывается диалог подтверждения', function() {
            sandbox.stub(block._metrikaCounters, 'isChanged').callsFake(function() {
                var deferred = $.Deferred();

                deferred.resolve(false);

                return deferred;
            });
            spyDialog('confirm');

            block._popup.hide();
            expect(BEM.blocks['b-user-dialog'].confirm.called).to.be.false;
            restoreDialog('confirm');
        });
    });

    describe('Предупреждения для Смарт-кампаний', function() {
        describe('Для 1 кампании', function() {
            beforeEach(function() {
                block = BEM.create('i-mass-action-metrika-counters', u._.extend(
                    {}, defaultCtx, { performanceCids: ['12345'] }));
                block.openPopup();
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();

            });

            describe('Показ предупреждения', function() {
                beforeEach(function() {
                    spyDialog('confirm');
                });

                afterEach(function() {
                    restoreDialog('confirm');
                });

                it('При добавлении 1 счетчика НЕ показывается предупреждение', function() {
                    block._metrikaCounters.trigger('add', '1234567890');
                    expect(BEM.blocks['b-user-dialog'].confirm.called).to.be.false;
                });

                it('При замене на более чем 1 счетчик показывается предупреждение', function() {
                    block._metrikaCounters.trigger('replace', '1234567890, 123567');
                    expect(BEM.blocks['b-user-dialog'].confirm.args[0][0].message)
                        .to.include('Для одной кампании Смарт-баннеров можно указать только один счётчик. В кампании Смарт-баннеров 12345 счётчик Метрики будет заменён на первый из списка. Продолжить?');
                });

                it('При замене на 1 счетчик не показывается предупреждение', function() {
                    block._metrikaCounters.trigger('replace', '1234567890');
                    expect(BEM.blocks['b-user-dialog'].confirm.called).to.be.false;
                });

                it('При удалении НЕ показывается предупреждение', function() {
                    block._metrikaCounters.trigger('deleteAll');
                    expect(BEM.blocks['b-user-dialog'].confirm.called).to.be.false;
                });
            });

            it('При подтверждении на предупреждении дергается ручка', function() {
                sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                    options.onConfirm.apply(options.callbackCtx);
                });
                sandbox.spy(BEM.blocks['i-web-api-request'].metrikaCounters, 'replace');

                block._metrikaCounters.trigger('replace', '1234567890, 123567');
                expect(BEM.blocks['i-web-api-request'].metrikaCounters.replace.called).to.be.true;
            });

            it('При отмене на предупреждении дергается метод окончания обработки', function() {
                sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                    options.onCancel.apply(options.callbackCtx);
                });
                sandbox.spy(block._metrikaCounters, 'onCountersSaveFinished');

                block._metrikaCounters.trigger('replace', '1234567890, 123567');
                expect(block._metrikaCounters.onCountersSaveFinished.called).to.be.true;
            });
        });

        describe('Для 1+ кампании', function() {
            beforeEach(function() {
                block = BEM.create('i-mass-action-metrika-counters', u._.extend(
                    {}, defaultCtx, { performanceCids: ['12345', '67890'] }));
                block.openPopup();
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
            });

            describe('Показ предупреждения', function() {
                beforeEach(function() {
                    spyDialog('confirm');
                });

                afterEach(function() {
                    restoreDialog('confirm');
                });

                it('При добавлении 1 счетчика НЕ показывается предупреждение', function() {
                    block._metrikaCounters.trigger('add', '1234567890');
                    expect(BEM.blocks['b-user-dialog'].confirm.called).to.be.false;
                });

                it('При замене на более чем 1 счетчик показывается предупреждение', function() {
                    block._metrikaCounters.trigger('replace', '1234567890, 123567');
                    expect(BEM.blocks['b-user-dialog'].confirm.args[0][0].message)
                        .to.include('Для одной кампании Смарт-баннеров можно указать только один счётчик. В кампаниях Смарт-баннеров 12345, 67890 счётчик Метрики будет заменён на первый из списка. Продолжить?');
                });

                it('При замене на 1 счетчик НЕ показывается предупреждение', function() {
                    block._metrikaCounters.trigger('replace', '1234567890');
                    expect(BEM.blocks['b-user-dialog'].confirm.called).to.be.false;
                });

                it('При удалении НЕ показывается предупреждение', function() {
                    block._metrikaCounters.trigger('deleteAll');
                    expect(BEM.blocks['b-user-dialog'].confirm.called).to.be.false;
                });
            });

            it('При подтверждении на предупреждении дергается ручка', function() {
                sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                    options.onConfirm.apply(options.callbackCtx);
                });
                sandbox.spy(BEM.blocks['i-web-api-request'].metrikaCounters, 'replace');

                block._metrikaCounters.trigger('replace', '1234567890, 123567');
                expect(BEM.blocks['i-web-api-request'].metrikaCounters.replace.called).to.be.true;
            });

            it('При отмене на предупреждении дергается метод окончания обработки', function() {
                sandbox.stub(BEM.blocks['b-user-dialog'], 'confirm').callsFake(function(options) {
                    options.onCancel.apply(options.callbackCtx);
                });
                sandbox.spy(block._metrikaCounters, 'onCountersSaveFinished');

                block._metrikaCounters.trigger('replace', '1234567890, 123567');
                expect(block._metrikaCounters.onCountersSaveFinished.called).to.be.true;
            })
        });
    });

    describe('Сообщения о неприменимости действия', function() {
        describe('Для 1 кампании', function() {
            beforeEach(function() {
                spyDialog('alert');
                block = BEM.create('i-mass-action-metrika-counters',
                    u._.extend({}, defaultCtx, { unAffectedCids: ['54321'] })
                );
                block.openPopup();

                stubRequest('add', { success: true });
                stubRequest('replace', { success: true });
                stubRequest('deleteAll', { success: true });
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            it('При добавлении 1 счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('add', '1234567890');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчик не был добавлен к кампании 54321, так как для неё действие неприменимо.');
            });

            it('При добавлении 1+ счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('add', '1234567890, 445545');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчики не были добавлены к кампании 54321, так как для неё действие неприменимо.');
            });

            it('При замене 1 счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('replace', '1234567890');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчик не был заменён в кампании 54321, так как для неё действие неприменимо.');
            });

            it('При замене 1+ счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('replace', '1234567890, 766767');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчики не были заменены в кампании 54321, так как для неё действие неприменимо.');
            });

            it('При удалении счетчиков показывается сообщение', function() {
                block._metrikaCounters.trigger('deleteAll');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчики не были удалены у кампании 54321, так как для неё действие неприменимо.');
            });
        });

        describe('Для 1+ кампании', function() {
            beforeEach(function() {
                spyDialog('alert');
                block = BEM.create('i-mass-action-metrika-counters',
                    u._.extend({}, defaultCtx, { unAffectedCids: ['54321', '9876'] })
                );
                block.openPopup();

                stubRequest('add', { success: true });
                stubRequest('replace', { success: true });
                stubRequest('deleteAll', { success: true });
            });

            afterEach(function() {
                block._popup.hide({ force: true });
                block.destruct();
                restoreDialog('alert');
            });

            it('При добавлении 1 счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('add', '1234567890');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчик не был добавлен к кампаниям 54321, 9876, так как для них действие неприменимо.');
            });

            it('При добавлении 1+ счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('add', '1234567890, 445545');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчики не были добавлены к кампаниям 54321, 9876, так как для них действие неприменимо.');
            });

            it('При замене 1 счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('replace', '1234567890');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчик не был заменён в кампаниях 54321, 9876, так как для них действие неприменимо.');
            });

            it('При замене 1+ счетчика показывается сообщение', function() {
                block._metrikaCounters.trigger('replace', '1234567890, 766767');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчики не были заменены в кампаниях 54321, 9876, так как для них действие неприменимо.');
            });

            it('При удалении счетчиков показывается сообщение', function() {
                block._metrikaCounters.trigger('deleteAll');

                sandbox.clock.tick(1000);

                expect(BEM.blocks['b-user-dialog'].alert.args[0][0].message)
                    .to.include('Счётчики не были удалены у кампаний 54321, 9876, так как для них действие неприменимо.');
            });
        });
    });

    describe('Предобработка списка счетчиков', function() {
        beforeEach(function() {
            block = BEM.create('i-mass-action-metrika-counters', defaultCtx);
            block.openPopup();

            sandbox.spy(BEM.blocks['i-web-api-request'].metrikaCounters, 'add');
            sandbox.spy(BEM.blocks['i-web-api-request'].metrikaCounters, 'replace');
            sandbox.spy(BEM.blocks['i-web-api-request'].metrikaCounters, 'deleteAll');
        });

        afterEach(function() {
            block._popup.hide({ force: true });
            block.destruct();
        });

        it('1 счетчик обрабатывается корректно', function() {
            block._metrikaCounters.trigger('add', '12345');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.add.args[0][0].counters).to.deep.equal(['12345']);
        });

        it('2 счетчика через запятую обрабатываются корректно', function() {
            block._metrikaCounters.trigger('add', '12345, 6789');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.add.args[0][0].counters).to.deep.equal(['12345', '6789']);
        });

        it('2 счетчика через пробел обрабатываются корректно', function() {
            block._metrikaCounters.trigger('add', '12345 6789');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.add.args[0][0].counters).to.deep.equal(['12345', '6789']);
        });

        it('3 счетчика через пробел и запятую обрабатывается корректно', function() {
            block._metrikaCounters.trigger('add', '12345 6789, 7777');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.add.args[0][0].counters)
                .to.deep.equal(['12345', '6789', '7777']);
        });

        it('3 счетчика через запятую и пробел обрабатывается корректно', function() {
            block._metrikaCounters.trigger('add', '12345, 6789 7777');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.add.args[0][0].counters)
                .to.deep.equal(['12345', '6789', '7777']);
        });

        it('2 счетчика с запятой в конце обрабатывается корректно', function() {
            block._metrikaCounters.trigger('add', '12345, 6789,');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.add.args[0][0].counters).to.deep.equal(['12345', '6789']);
        });

        it('2 счетчика с пробелом и запятой в конце обрабатывается корректно', function() {
            block._metrikaCounters.trigger('add', '12345 6789,');

            expect(BEM.blocks['i-web-api-request'].metrikaCounters.add.args[0][0].counters).to.deep.equal(['12345', '6789']);
        });
    })
});
